function [swimmer, NNSw1, NNSw2, NNSw3] = SwimmersTorusRichardson( ...
    nSwimmers, ...
    swimmerModel, ...
    swimmerSize, ...
    swimmerDisc, ...
    discrType, ...
    x0, ...
    bodyFrame, ...
    procType, ...
    blockSize, ...
    omega)
%SWIMMERTORUSRICHARDSON SETS UP SWIMMER STRUCTS FOR MULTI-TORUS with
%Richardson extrapolation
%
% Inputs:
%	NSWIMMER		- Number of swimmers
%	SWIMMERMODEL 	- Function handle for generating swimmer discretisation
%                       and velocity at time t
%	SWIMMERSIZE 	- Size parameters for swimmer [a, R, lambda, L]
%	SWIMMERDISC 	- Number of discretisation points for the swimmer 
%					   [nForce, nQuad]
%   X0              - Initial location in space 
%   BODYFRAME       - Body frame = [Direction, normal, binormal]
%   OMEGA           - Rotational velocity of helix
%
%
% Code for the manuscript:
%   "The Art of Coarse Stokes: Richardson extrapolation improves the accuracy and efficiency of the method of regularized stokeslets" 
% D.J. Smith & M.T. Gallagher
%

% Author: M.T. Gallagher
% Email: m dot t dot gallagher at bham dot ac dot uk
% www.gitlab.com/meuriggallagher
% Copyright: M.T. Gallagher 2020

% Initialise swimmers
swimmer = cell(nSwimmers, 1);
NN1 = cell(nSwimmers, 1);
NN2 = cell(nSwimmers, 1);
NN3 = cell(nSwimmers, 1);

% Set up swimmer struct
for iSw = 1 : nSwimmers
    % Generate helix
    [x, X1, NN1{iSw}] = GenerateTorus(swimmerSize{iSw}, ...
        swimmerDisc{iSw}(1:2), x0{iSw}, discrType{iSw}, procType, blockSize);
    [~, X2, NN2{iSw}] = GenerateTorus(swimmerSize{iSw}, ...
        swimmerDisc{iSw}([1,3]), x0{iSw}, discrType{iSw}, procType, blockSize);
    [~, X3, NN3{iSw}] = GenerateTorus(swimmerSize{iSw}, ...
        swimmerDisc{iSw}([1,4]), x0{iSw}, discrType{iSw}, procType, blockSize);
    
	swimmer{iSw} = struct( ...
		'fn', 	 	swimmerModel, ...
        'x0',       x0{iSw}, ...
        'b10',      bodyFrame{iSw}(:,1), ...
        'b20',      bodyFrame{iSw}(:,2), ...
        'surfaceArea',     4 * pi^2 * swimmerSize{iSw}(1) * swimmerSize{iSw}(2), ...
		'model', 	struct( ...
            'R',        swimmerSize{iSw}(1), ...
            'r',        swimmerSize{iSw}(2), ...        
            'x',        x, ...
            'X1',       X1, ...
            'X2',       X2, ...
            'X3',       X3, ...
            'omega',    omega));
end

% Merge Swimmer NN matrices
NNSw1 = NN1{1};
NNSw2 = NN2{1};
NNSw3 = NN3{1};
for iSw = 2 : nSwimmers
    NNSw1 = MergeNNMatrices(NNSw1, NN1{iSw});
    NNSw2 = MergeNNMatrices(NNSw2, NN2{iSw});
    NNSw3 = MergeNNMatrices(NNSw3, NN3{iSw});
end

end
