function [swimmer, NNSw] = SwimmersTorus( ...
    nSwimmers, ...
    swimmerModel, ...
    swimmerSize, ...
    swimmerDisc, ...
    discrType, ...
    x0, ...
    bodyFrame, ...
    procType, ...
    blockSize, ...
    omega)
%SWIMMERTORUS SETS UP SWIMMER STRUCTS FOR MULTI-TORUS
%
% Inputs:
%	NSWIMMER		- Number of swimmers
%	SWIMMERMODEL 	- Function handle for generating swimmer discretisation
%                       and velocity at time t
%	SWIMMERSIZE 	- Size parameters for swimmer [a, R, lambda, L]
%	SWIMMERDISC 	- Number of discretisation points for the swimmer 
%					   [nForce, nQuad]
%   X0              - Initial location in space 
%   BODYFRAME       - Body frame = [Direction, normal, binormal]
%   OMEGA           - Rotational velocity of helix
%
%
% Code for the manuscript:
%   "The Art of Coarse Stokes: Richardson extrapolation improves the accuracy and efficiency of the method of regularized stokeslets" 
% D.J. Smith & M.T. Gallagher
%

% Author: M.T. Gallagher
% Email: m dot t dot gallagher at bham dot ac dot uk
% www.gitlab.com/meuriggallagher
% Copyright: M.T. Gallagher 2020

% Initialise swimmers
swimmer = cell(nSwimmers, 1);
NN = cell(nSwimmers, 1);

% Set up swimmer struct
for iSw = 1 : nSwimmers
    % Generate helix
    [x, X, NN{iSw}] = GenerateTorus(swimmerSize{iSw}, ...
        swimmerDisc{iSw}, x0{iSw}, discrType{iSw}, procType, blockSize);
    
	swimmer{iSw} = struct( ...
		'fn', 	 	swimmerModel, ...
        'x0',       x0{iSw}, ...
        'b10',      bodyFrame{iSw}(:,1), ...
        'b20',      bodyFrame{iSw}(:,2), ...
		'model', 	struct( ...
            'R',        swimmerSize{iSw}(1), ...
            'r',        swimmerSize{iSw}(2), ...        
            'x',        x, ...
            'X',        X, ...
            'omega',    omega));
end

% Merge Swimmer NN matrices
NNSw = NN{1};
for iSw = 2 : nSwimmers
    NNSw = MergeNNMatrices(NNSw, NN{iSw});
end

end
