%SWIMMERCHLMAY.M SETS UP SWIMMER STRUCTS FOR MULTI-CELEGANS 
%
% Inputs:
%	NCELEGANS 		- Number of swimmers
%	CELEGANSMODEL 	- Function handle for generating C.Elegans discretisaion
%	BEATMODEL 	- Function handle for C.Elegans flagellar beat
%	CELEGANSSIZE 	- Size parameters for C.Elegans head in multiples
%					of the length [R0,H0,T0], where H0 and T0 are the 
%					lengths of the head and tail sections respectively
%	CELEGANSDISC 	- Number of discretisation points for the C.Elegans 
%					[fR,fTh,qR,qTh]
%	NBEATS 		- Number of beats to set up C.Elegans
%   X0          - Initial location in space 
%   BODYFRAME   - Body frame = [Direction, normal, binormal];
%	ARGS 		- Struct for changing C.Elegans beat parameters
%					ARGS{ISW}.PHASE, ARGS{ISW}.K sets the phase and wavenumber
%
function swimmer = SwimmersCElegans(ncElegans,cElegansModel, ...
	cElegansSize,cElegansDisc,x0,bodyFrame,epsilon)

% Initialise swimmers
swimmer = cell(ncElegans,1);

% Set up swimmer struct
for iSw = 1 : ncElegans
	swimmer{iSw} = struct( ...
		'fn', 	 	cElegansModel, ...
        'x0',       x0{iSw}, ...
        'b10',      bodyFrame{iSw}(:,1), ...
        'b20',      bodyFrame{iSw}(:,2), ...
		'model', 	struct( ...
			'F', 				[], ...
			'tangentAngleFn',	[], ...
			'fR', 				cElegansDisc(1), ...
			'fTh', 				cElegansDisc(2), ...
			'qR',				cElegansDisc(3), ...
			'qTh',				cElegansDisc(4), ...
			'L0',				1, ...
			'R0',				cElegansSize(1), ...
			'H0',				cElegansSize(2), ...
			'T0',				cElegansSize(3), ...
            'eps',              epsilon));

	% Waveform
	c = 5.3; d = 3.1; freq = 1;
	swimmer{iSw}.model.tangentAngleFn = @(s,t) (pi*(d*s - c) ...
    	.* sin(pi*(s - 2*t/freq)) + d*cos(pi*(s - 2*t/freq))) / pi^2 ...
    	- (pi*(d*0 - c) * sin(pi*(0 - 2*t/freq)) ...
    	+ d*cos(pi*(0 - 2*t/freq))) / pi^2;

	% Swimmer position and velocities
	swimmer{iSw}.model.F = ConstructCElegansInterpolant( ...
    	swimmer{iSw}.model.tangentAngleFn);
end

end
