#
# Copyright (C) 2013-2018 The ESPResSo project
#
# This file is part of ESPResSo.
#
# ESPResSo is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# ESPResSo is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
include "myconfig.pxi"

cimport numpy as np
import numpy as np
from . cimport utils
from espressomd.utils cimport *
from . cimport particle_data
from .interactions import BondedInteraction
from .interactions import BondedInteractions
from .interactions cimport bonded_ia_params
from copy import copy
from globals cimport max_seen_particle, time_step, n_part, n_rigidbonds, max_seen_particle_type, swimming_particles_exist, FIELD_SWIMMING_PARTICLES_EXIST, mpi_bcast_parameter
import collections
import functools
import types
from espressomd.utils import nesting_level, array_locked, is_valid_type
from espressomd.utils cimport make_array_locked
from .grid cimport box_geo, folded_position, unfolded_position

PARTICLE_EXT_FORCE = 1


def _COORD_FIXED(coord):
    return 2L << coord

COORDS_FIX_MASK = _COORD_FIXED(0) | _COORD_FIXED(1) | _COORD_FIXED(2)
COORDS_ALL_FIXED = _COORD_FIXED(0) & _COORD_FIXED(1) & _COORD_FIXED(2)
PARTICLE_EXT_TORQUE = 16
ROT_X = 2
ROT_Y = 4
ROT_Z = 8

# List of particle attributes for pickle and the like
# Autogenerated from the class. Everything which is of the same
# type as ParticleHandle.pos (getter_wrapper)
particle_attributes = []
for d in dir(ParticleHandle):
    if type(getattr(ParticleHandle, d)) == type(ParticleHandle.pos):
        if not d in ["pos_folded"]:
            particle_attributes.append(d)

cdef class ParticleHandle:
    def __cinit__(self, int _id):
        self._id = _id

    cdef int update_particle_data(self) except -1:
        self.particle_data = get_particle_data_ptr(get_particle_data(self._id))

    def __str__(self):
        res = collections.OrderedDict()
        # Id and pos first, then the rest
        res["id"] = self.id
        res["pos"] = self.pos
        for a in particle_attributes:
            tmp = getattr(self, a)
            # Remove array type names from output
            if isinstance(tmp, array_locked):
                res[a] = tuple(tmp)
            else:
                res[a] = tmp

        # Get rid of OrderedDict in output
        return str(res).replace("OrderedDict(", "ParticleHandle(")

    # The individual attributes of a particle are implemented as properties.
    property id:
        """Integer particle id

        """

        def __get__(self):
            self.update_particle_data()
            return self._id

    # Particle Type
    property type:
        """
        The particle type for nonbonded interactions.

        type : :obj:`int`
            Nonbonded interactions act between different types of particles.

        .. note::
           The value of `type` has to be an integer >= 0.

        """

        def __set__(self, _type):
            if is_valid_type(_type, int) and _type >= 0:
                set_particle_type(self._id, _type)
            else:
                raise ValueError("type must be an integer >= 0")

        def __get__(self):
            self.update_particle_data()
            return self.particle_data.p.type

    # Particle MolId
    property mol_id:
        """
        The molecule id of the Particle.

        mol_id : :obj:`int`
            The particle ``mol_id`` is used to differentiate between
            particles belonging to different molecules, e.g. when virtual
            sites are used, or object-in-fuid cells. The default
            ``mol_id`` for all particles is 0.

        .. note::
           The value of ``mol_id`` has to be an integer >= 0.

        """

        def __set__(self, _mol_id):
            if is_valid_type(_mol_id, int) and _mol_id >= 0:
                set_particle_mol_id(self._id, _mol_id)
            else:
                raise ValueError("mol_id must be an integer >= 0")

        def __get__(self):
            self.update_particle_data()
            return self.particle_data.p.mol_id

    # Position
    property pos:
        """
        The unwrapped (not folded into central box) particle position.

        pos : (3,) array_like of :obj:`float`
            The particles's absolute position.

        """

        def __set__(self, _pos):
            cdef double mypos[3]
            check_type_or_throw_except(
                _pos, 3, float, "Position must be 3 floats")
            for i in range(3):
                mypos[i] = _pos[i]
            if place_particle(self._id, mypos) == -1:
                raise Exception("particle could not be set")

        def __get__(self):
            self.update_particle_data()
            return make_array_locked(unfolded_position( < Vector3d > self.particle_data.r.p, < Vector3i > self.particle_data.l.i, box_geo.length()))

    property pos_folded:
        """
        The wrapped (folded into central box) position vector of a particle.

        pos : (3,) array_like of :obj:`float`
            The particles's position.

        .. note::
           Setting the folded position is ambiguous and is thus not possible, please use `pos`.

        Examples
        --------
        >>> import espressomd
        >>>
        >>> system = espressomd.System()
        >>>
        >>> system.box_l=[10,10,10]
        >>> # add two bonded particles to particle 0
        >>> system.part.add(id=0, pos=(5, 0, 0))
        >>> system.part.add(id=1, pos=(10, 0, 0))
        >>> system.part.add(id=2, pos=(25, 0, 0))
        >>> for p in system.part:
        >>>     print(p.pos)
        [ 5.  0.  0.]
        [ 10.   0.   0.]
        [ 25.   0.   0.]
        >>>
        >>> for p in system.part:
        >>>     print(p.pos_folded)
        [5.0, 0.0, 0.0]
        [0.0, 0.0, 0.0]
        [5.0, 0.0, 0.0]

        """

        def __set__(self, pos_folded):
            raise AttributeError(
                "setting a folded position is not implemented")

        def __get__(self):
            self.update_particle_data()
            return make_array_locked(folded_position(Vector3d(self.particle_data.r.p), box_geo))

    property image_box:
        """
        The image box the particles is in.

        This is the number of times the particle position has been folded by
        the box length in each direction.
        """

        def __get__(self):
            self.update_particle_data()
            return array_locked([self.particle_data.l.i[0],
                                 self.particle_data.l.i[1],
                                 self.particle_data.l.i[2]])

    # Velocity
    property v:
        """
        The particle velocity in the lab frame.

        v : (3,) array_like of :obj:`float`
            The particles's velocity

        .. note::
           The velocity remains variable and will be changed during integration.

        """

        def __set__(self, _v):
            cdef double myv[3]
            check_type_or_throw_except(
                _v, 3, float, "Velocity has to be floats")
            for i in range(3):
                myv[i] = _v[i]
            set_particle_v(self._id, myv)

        def __get__(self):
            self.update_particle_data()
            return make_array_locked(self.particle_data.m.v)

    # Force
    property f:
        """
        The instantaneous force acting on this particle.

        f : (3,) array_like of :obj:`float`
            The current forces on the particle

        .. note::
           Whereas the velocity is modified with respect to the velocity you set
           upon integration, the force it recomputed during the integration step and any
           force set in this way is immediatly lost at the next integration step.

        """

        def __set__(self, _f):
            cdef Vector3d f
            check_type_or_throw_except(_f, 3, float, "Force has to be floats")
            for i in range(3):
                f[i] = _f[i]
            set_particle_f(self._id, f)

        def __get__(self):
            self.update_particle_data()
            return make_array_locked(self.particle_data.f.f)

    # Bonds
    property bonds:
        """
        The bonds stored by this particle. Note that bonds are only stored by
        one partner. You need to define a bonded interaction.

        bonds : list/tuple of tuples/lists
            a bond tuple is specified as a bond identifier associated with
            a particle `(bond_ID, part_ID)`. A single particle may contain
            multiple such tuples.

        See Also
        --------
        espressomd.particle_data.ParticleHandle.add_bond : Method to add bonds to a `Particle`
        espressomd.particle_data.ParticleHandle.delete_bond : Method to remove bonds from a `Particle`

        .. note::
           Bond ids have to be an integer >= 0.

        """

        def __set__(self, _bonds):
            # Assigning to the bond property means replacing the existing value
            # i.e., we delete all existing bonds
            delete_particle_bonds(self._id)

            # Empty list? ony delete
            if _bonds:
                nlvl = nesting_level(_bonds)
                if nlvl == 1:  # Single item
                    self.add_bond(_bonds)
                elif nlvl == 2:  # List of items
                    for bond in _bonds:
                        self.add_bond(bond)
                else:
                    raise ValueError(
                        "Bonds have to specified as lists of tuples/lists or a single list.")

        def __get__(self):
            self.update_particle_data()
            bonds = []
            # Go through the bond list of the particle
            i = 0
            while i < self.particle_data.bl.n:
                bond = []
                # Bond type:
                bond_id = self.particle_data.bl.e[i]
                bond.append(BondedInteractions()[bond_id])
                # Number of partners
                nPartners = bonded_ia_params[bond_id].num

                i += 1

                # Copy bond partners
                for j in range(nPartners):
                    bond.append(self.particle_data.bl.e[i])
                    i += 1
                bonds.append(tuple(bond))

            return tuple(bonds)

    property node:
        """
        The node the particle is on, identified by its MPI rank.
        """

        def __get__(self):
            return get_particle_node(self._id)

    # Properties that exist only when certain features are activated
    # MASS
    property mass:
        """
        Particle mass.

        mass : :obj:`float`
               The particle mass.

        See Also
        ----------
        :meth:`espressomd.thermostat.Thermostat.set_langevin` : Setting the parameters of the Langevin thermostat

        """

        def __set__(self, _mass):
            IF MASS == 1:
                check_type_or_throw_except(
                    _mass, 1, float, "Mass has to be 1 floats")
                set_particle_mass(self._id, _mass)
            ELSE:
                raise AttributeError("You are trying to set the particle mass \
                                     but the mass feature is not compiled in.")

        def __get__(self):
            self.update_particle_data()
            return self.particle_data.p.mass

    IF ROTATION:
        property omega_lab:
            """
            The particle angular velocity the lab frame.

            omega_lab : (3,) array_like of :obj:`float`
                The particle's angular velocity measured from the lab frame.

            .. note::
               This needs the feature ``ROTATION``.

               If you set the angular velocity of the particle in the lab
               frame, the orientation of the particle
               (:attr:`~espressomd.particle_data.ParticleHandle.quat`) must be
               set before setting ``omega_lab``, otherwise the conversion from
               lab to body frame will not be handled properly.

            See Also
            ---------
            :attr:`~espressomd.particle_data.ParticleHandle.omega_body`

            """

            def __set__(self, _o):
                cdef Vector3d myo
                check_type_or_throw_except(
                    _o, 3, float, "Omega_lab has to be 3 floats.")
                for i in range(3):
                    myo[i] = _o[i]
                set_particle_omega_lab(self._id, myo)

            def __get__(self):
                self.update_particle_data()
                return array_locked(self.convert_vector_body_to_space(self.omega_body))

        property quat:
            """
            Particle quaternion representation.

            quat : (4,) array_like of :obj:`float`
                Sets the quaternion representation of the
                rotational position of this particle.

            .. note::
               This needs the feature ``ROTATION``.

            """

            def __set__(self, _q):
                cdef double myq[4]
                check_type_or_throw_except(
                    _q, 4, float, "Quaternions has to be 4 floats.")
                for i in range(4):
                    myq[i] = _q[i]
                set_particle_quat(self._id, myq)

            def __get__(self):
                self.update_particle_data()

                cdef const double * x = NULL
                pointer_to_quat(self.particle_data, x)
                return array_locked([x[0], x[1], x[2], x[3]])

        # Director (z-axis in body fixed frame)
        property director:
            """
            Director.

            .. note::
               Setting the director is not implemented.
               This needs the feature ``ROTATION``.

            """

            def __set__(self, _q):
                raise AttributeError(
                    "Setting the director is not implemented in the C++-core of Espresso.")

            def __get__(self):
                self.update_particle_data()
                return make_array_locked(self.particle_data.r.calc_director())

        # ROTATIONAL_INERTIA
        property omega_body:
            """
            The particle angular velocity in body frame.

            omega_body : (3,) array_like of :obj:`float`

            This property sets the angular momentum of this particle in the
            particles co-rotating frame (or body frame).

            .. note::
               This needs the feature ``ROTATION``.

            """

            def __set__(self, _o):
                cdef Vector3d myo
                check_type_or_throw_except(
                    _o, 3, float, "Omega_body has to be 3 floats.")
                for i in range(3):
                    myo[i] = _o[i]
                set_particle_omega_body(self._id, myo)

            def __get__(self):
                self.update_particle_data()
                cdef const double * o = NULL
                pointer_to_omega_body(self.particle_data, o)
                return array_locked([o[0], o[1], o[2]])

        property torque_lab:
            """
            The particle torque in the lab frame.

            torque_lab : (3,) array_like of :obj:`float`

            This property defines the torque of this particle
            in the fixed frame (or laboratory frame).

            .. note::
               The orientation of the particle
               (:attr:`~espressomd.particle_data.ParticleHandle.quat`) must be
               set before setting this property, otherwise the conversion from
               lab to body frame will not be handled properly.

            """

            def __set__(self, _t):
                cdef Vector3d myt
                check_type_or_throw_except(
                    _t, 3, float, "Torque has to be 3 floats.")
                for i in range(3):
                    myt[i] = _t[i]
                set_particle_torque_lab(self._id, myt)

            def __get__(self):
                self.update_particle_data()
                cdef Vector3d torque_body
                cdef Vector3d torque_space
                torque_body = get_torque_body(self.particle_data[0])
                torque_space = convert_vector_body_to_space(
                    self.particle_data[0], torque_body)

                return make_array_locked(torque_space)

    IF ROTATIONAL_INERTIA:
        property rinertia:
            """
            The particle rotational inertia.

            rintertia : (3,) array_like of :obj:`float`

            Sets the diagonal elements of this particles rotational inertia
            tensor. These correspond with the inertial moments along the
            coordinate axes in the particle’s co-rotating coordinate system.
            When the particle's quaternions are set to 1 0 0 0, the co-rotating
            and the fixed (lab) frame are co-aligned.

            .. note::
               This needs the feature ``ROTATIONAL_INERTIA``.

            """

            def __set__(self, _rinertia):
                cdef double rinertia[3]
                check_type_or_throw_except(
                    _rinertia, 3, float, "Rotation_inertia has to be 3 floats.")
                for i in range(3):
                    rinertia[i] = _rinertia[i]
                set_particle_rotational_inertia(self._id, rinertia)

            def __get__(self):
                self.update_particle_data()
                cdef const double * rinertia = NULL
                pointer_to_rotational_inertia(
                    self.particle_data, rinertia)
                return array_locked([rinertia[0], rinertia[1], rinertia[2]])

    IF MEMBRANE_COLLISION:
        property out_direction:
            """OIF Outward direction"""

            def __set__(self, _out_direction):
                cdef double out_direction[3]
                check_type_or_throw_except(
                    _out_direction, 3, float, "out_direction has to be 3 floats")
                for i in range(3):
                    out_direction[i] = _out_direction[i]
                set_particle_out_direction(self.id, out_direction) 

            def __get__(self):
                self.update_particle_data()
                cdef const double * out_direction = NULL
                pointer_to_out_direction(self.particle_data, out_direction)
                return np.array([out_direction[0], out_direction[1], out_direction[2]])

    IF AFFINITY:
        property bond_site:
            """OIF bond_site"""

            def __set__(self, _bond_site):
                cdef double bond_site[3]
                check_type_or_throw_except(
                    _bond_site, 3, float, "bond_site has to be 3 floats")
                for i in range(3):
                    bond_site[i] = _bond_site[i]
                set_particle_affinity(self.id, bond_site) 

            def __get__(self):
                self.update_particle_data()
                cdef const double * bond_site = NULL
                pointer_to_bond_site(self.particle_data, bond_site)
                return array_locked([bond_site[0], bond_site[1], bond_site[2]])


# Charge
    property q:
        """
        Particle charge.

        q : :obj:`float`

        .. note::
           This needs the feature ``ELECTROSTATICS``.

        """

        def __set__(self, _q):
            cdef double myq
            check_type_or_throw_except(
                _q, 1, float, "Charge has to be floats.")
            myq = _q
            set_particle_q(self._id, myq) 

        def __get__(self):
            self.update_particle_data()
            cdef const double * x = NULL
            pointer_to_q(self.particle_data, x)
            return x[0]

    IF LB_ELECTROHYDRODYNAMICS:
        property mu_E:
            """
            Particle electrophoretic velocity.

            mu_E : :obj:`float`

            This effectivly acts as a velocity offset between
            a lattice-Boltzmann fluid and the particle. Has only
            an effect if LB is turned on.

            .. note::
               This needs the feature ``LB_ELECTROHYDRODYNAMICS``.

            """

            def __set__(self, mu_E):
                cdef double _mu_E[3]

                check_type_or_throw_except(
                    mu_E, 3, float, "mu_E has to be 3 floats.")

                _mu_E[0] = mu_E[0]
                _mu_E[1] = mu_E[1]
                _mu_E[2] = mu_E[2]

                set_particle_mu_E(self._id, _mu_E)

            def __get__(self):
                cdef double mu_E[3]
                get_particle_mu_E(self._id, mu_E)

                return array_locked([mu_E[0], mu_E[1], mu_E[2]])

    IF VIRTUAL_SITES:
        property virtual:
            """ Virtual flag.

            Declares the particles as virtual (1) or non-virtual (0, default).

            virtual : :obj:`int`

            .. note::
               This needs the feature ``VIRTUAL_SITES``

            """

            def __set__(self, _v):
                if is_valid_type(_v, int):
                    set_particle_virtual(self._id, _v) 
                else:
                    raise ValueError("virtual must be an integer >= 0.")

            def __get__(self):
                self.update_particle_data()
                cdef const int * x = NULL
                pointer_to_virtual(self.particle_data, x)
                return x[0]

    IF VIRTUAL_SITES_RELATIVE:
        property vs_quat:
            """ Virtual site quaternion.

            This quaternion describes the virtual particles orientation in the
            body fixed frame of the related real particle.

            vs_quat : (4,) array_like of :obj:`float`

            .. note::
               This needs the feature ``VIRTUAL_SITES_RELATIVE``.

            """

            def __set__(self, q):
                if len(q) != 4:
                    raise ValueError(
                        "vs_quat has to be an array-like of length 4.")
                cdef double _q[4]
                for i in range(4):
                    _q[i] = q[i]
                set_particle_vs_quat(self._id, _q)

            def __get__(self):
                self.update_particle_data()
                cdef const double * q = NULL
                pointer_to_vs_quat(self.particle_data, q)
                return np.array([q[0], q[1], q[2], q[3]])

        property vs_relative:
            """
            Virtual sites relative parameters.

            Allows for manual access to the attributes of virtual sites in the "relative"
            implementation. PID denotes the id of the particle to which this virtual site
            is related and distance the distance between non-virtual and virtual particle.
            The relative orientation is specified as a quaternion of 4 floats.

            vs_relative : tuple: (PID, distance, (q1,q2,q3,q4))

            .. note::
               This needs the feature ``VIRTUAL_SITES_RELATIVE``

            """

            def __set__(self, x):
                if len(x) < 3:
                    raise ValueError(
                        "vs_relative needs input like id,distance,(q1,q2,q3,q4).")
                _relto = x[0]
                _dist = x[1]
                q = x[2]
                cdef double _q[4]
                for i in range(4):
                    _q[i] = q[i]

                if is_valid_type(_relto, int) and is_valid_type(_dist, float) and all(is_valid_type(fq, float) for fq in q):
                    set_particle_vs_relative(self._id, _relto, _dist, _q) 
                else:
                    raise ValueError(
                        "vs_relative needs input like id<int>,distance<float>,(q1<float>,q2<float>,q3<float>,q4<float>).")

            def __get__(self):
                self.update_particle_data()
                cdef const int * rel_to = NULL
                cdef const double * dist = NULL
                cdef const double * q = NULL
                pointer_to_vs_relative(
                    self.particle_data, rel_to, dist, q)
                return (rel_to[0], dist[0], np.array((q[0], q[1], q[2], q[3])))

        # vs_auto_relate_to
        def vs_auto_relate_to(self, _relto):
            """
            Setup this particle as virtual site relative to the particle with the given id.

            """
            check_type_or_throw_except(
                _relto, 1, int, "Argument of vs_auto_relate_to has to be of type int.")
            if vs_relate_to(self._id, _relto):
                handle_errors("vs_relative setup failed.")

    IF DIPOLES:
        property dip:
            """
            The orientation of the dipole axis.

            dip : (3,) array_like of :obj:`float`

            .. note::
               This needs the feature ``DIPOLES``.

            """

            def __set__(self, _q):
                cdef double myq[3]
                check_type_or_throw_except(
                    _q, 3, float, "Dipole moment vector has to be 3 floats.")
                for i in range(3):
                    myq[i] = _q[i]
                set_particle_dip(self._id, myq) 

            def __get__(self):
                self.update_particle_data()

                return make_array_locked(self.particle_data.calc_dip())

        # Scalar magnitude of dipole moment
        property dipm:
            """
            The magnitude of the dipole moment.

            dipm : :obj:`float`

            .. note::
               This needs the feature ``DIPOLES``.

            """

            def __set__(self, _q):
                check_type_or_throw_except(
                    _q, 1, float, "Magnitude of dipole moment has to be 1 floats.")
                set_particle_dipm(self._id, _q) 

            def __get__(self):
                self.update_particle_data()
                cdef const double * x = NULL
                pointer_to_dipm(self.particle_data, x)
                return x[0]

    IF EXTERNAL_FORCES:
        property ext_force:
            """
            An additional external force applied to the particle.

            ext_force : (3,) array_like of :obj:`float`

            .. note::
               This needs the feature ``EXTERNAL_FORCES``.

            """

            def __set__(self, _ext_f):
                cdef Vector3d ext_f
                check_type_or_throw_except(
                    _ext_f, 3, float, "External force vector has to be 3 floats.")
                for i in range(3):
                    ext_f[i] = _ext_f[i]

                set_particle_ext_force(self._id, ext_f) 

            def __get__(self):
                self.update_particle_data()

                cdef const double * ext_f = NULL
                cdef const int * ext_flag = NULL
                pointer_to_ext_force(self.particle_data, ext_flag, ext_f)
                if (ext_flag[0] & PARTICLE_EXT_FORCE):
                    return array_locked([ext_f[0], ext_f[1], ext_f[2]])
                else:
                    return array_locked([0.0, 0.0, 0.0])

        property fix:
            """
            Fixes the particle motion in the specified cartesian directions.

            fix : (3,) array_like of :obj:`int`

            Fixes the particle in space. By supplying a set of 3 integers as
            arguments it is possible to fix motion in x, y, or z coordinates
            independently. For example::

                part[<INDEX>].fix = [0, 0, 1]

            will fix motion for particle with index ``INDEX`` only in z.

            .. note::
               This needs the feature ``EXTERNAL_FORCES``.

            """

            def __set__(self, _fixed_coord_flag):
                cdef int ext_flag = 0
                check_type_or_throw_except(
                    _fixed_coord_flag, 3, int, "Fix has to be 3 ints.")
                for i in map(long, range(3)):
                    if _fixed_coord_flag[i]:
                        ext_flag |= _COORD_FIXED(i)
                set_particle_fix(self._id, ext_flag) 

            def __get__(self):
                self.update_particle_data()
                fixed_coord_flag = np.array([0, 0, 0], dtype=int)
                cdef const int * ext_flag = NULL
                pointer_to_fix(self.particle_data, ext_flag)
                for i in map(long, range(3)):
                    if ext_flag[0] & _COORD_FIXED(i):
                        fixed_coord_flag[i] = 1
                return array_locked(fixed_coord_flag)

        IF ROTATION:
            property ext_torque:
                """
                An additional external torque is applied to the particle.

                ext_torque : (3,) array_like of :obj:`float`

                ..  note::
                    * This torque is specified in the laboratory frame!
                    * This needs features ``EXTERNAL_FORCES`` and ``ROTATION``.

                """

                def __set__(self, _ext_t):
                    cdef Vector3d ext_t
                    check_type_or_throw_except(
                        _ext_t, 3, float, "External force vector has to be 3 floats.")
                    for i in range(3):
                        ext_t[i] = _ext_t[i]

                    set_particle_ext_torque(self._id, ext_t) 

                def __get__(self):
                    self.update_particle_data()
                    cdef const double * ext_t = NULL
                    cdef const int * ext_flag = NULL
                    pointer_to_ext_torque(
                        self.particle_data, ext_flag, ext_t)
                    if (ext_flag[0] & PARTICLE_EXT_TORQUE):
                        return array_locked([ext_t[0], ext_t[1], ext_t[2]])
                    else:
                        return array_locked([0.0, 0.0, 0.0])

    IF LANGEVIN_PER_PARTICLE:
        IF PARTICLE_ANISOTROPY:
            property gamma:
                """
                The body-fixed frictional coefficient used in the the Langevin thermostat.

                gamma : `float` or (3,) array_like of :obj:`float`

                .. note::
                    This needs features ``LANGEVIN_PER_PARTICLE`` and
                    ``PARTICLE_ANISOTROPY``.

                See Also
                ----------
                :meth:`espressomd.thermostat.Thermostat.set_langevin` : Setting the parameters of the Langevin thermostat

                """

                def __set__(self, _gamma):
                    cdef Vector3d gamma

                    # We accept a single number by just repeating it
                    if not isinstance(_gamma, collections.Iterable):
                        _gamma = 3 * [_gamma]

                    check_type_or_throw_except(
                        _gamma, 3, float, "Friction has to be 3 floats.")
                    for i in range(3):
                        gamma[i] = _gamma[i]
                    set_particle_gamma(self._id, gamma) 

                def __get__(self):
                    self.update_particle_data()

                    cdef const double * gamma = NULL
                    pointer_to_gamma(self.particle_data, gamma)
                    return array_locked([gamma[0], gamma[1], gamma[2]])

        ELSE:
            property gamma:
                """
                The translational frictional coefficient used in the the Langevin thermostat.

                gamma : :obj:`float`

                .. note::
                   This needs the feature ``LANGEVIN_PER_PARTICLE``.

                See Also
                ----------
                :meth:`espressomd.thermostat.Thermostat.set_langevin.set_langevin` : Setting the parameters of the Langevin thermostat

                """

                def __set__(self, _gamma):
                    check_type_or_throw_except(
                        _gamma, 1, float, "Gamma has to be a float.")
                    set_particle_gamma(self._id, _gamma) 

                def __get__(self):
                    self.update_particle_data()
                    cdef const double * gamma = NULL
                    pointer_to_gamma(self.particle_data, gamma)
                    return gamma[0]
        IF ROTATION:
            IF PARTICLE_ANISOTROPY:
                property gamma_rot:
                    """
                    The particle translational frictional coefficient used in the Langevin thermostat.

                    gamma_rot : :obj:`float` of (3,) array_like of :obj:`float`

                    .. note::
                        This needs features ``LANGEVIN_PER_PARTICLE``,
                        ``ROTATION`` and ``PARTICLE_ANISOTROPY``.
                    """

                    def __set__(self, _gamma_rot):
                        cdef Vector3d gamma_rot
                        # We accept a single number by just repeating it
                        if not isinstance(_gamma_rot, collections.Iterable):
                            _gamma_rot = 3 * [_gamma_rot]

                        check_type_or_throw_except(
                            _gamma_rot, 3, float, "Rotational friction has to be 3 floats.")
                        for i in range(3):
                            gamma_rot[i] = _gamma_rot[i]
                        set_particle_gamma_rot(self._id, gamma_rot) 

                    def __get__(self):
                        self.update_particle_data()
                        cdef const double * gamma_rot = NULL
                        pointer_to_gamma_rot(
                            self.particle_data, gamma_rot)
                        return array_locked([gamma_rot[0], gamma_rot[1], gamma_rot[2]])
            ELSE:
                property gamma_rot:
                    """
                    The particle rotational frictional coefficient used in the Langevin thermostat.

                    gamma : :obj:`float`

                    """

                    def __set__(self, _gamma_rot):
                        check_type_or_throw_except(
                            _gamma_rot, 1, float, "gamma_rot has to be a float.")
                        set_particle_gamma_rot(self._id, _gamma_rot) 

                    def __get__(self):
                        self.update_particle_data()
                        cdef const double * gamma_rot = NULL
                        pointer_to_gamma_rot(
                            self.particle_data, gamma_rot)
                        return gamma_rot[0]

        property temp:
            """
            Particle's temperature in the Langevin thermostat.

            temp: :obj:`float`

            .. note::
                This needs the feature ``LANGEVIN_PER_PARTICLE``.

            """

            def __set__(self, _temp):
                check_type_or_throw_except(
                    _temp, 1, float, "temp has to be a float.")
                set_particle_temperature(self._id, _temp) 

            def __get__(self):
                self.update_particle_data()
                cdef const double * temp = NULL
                pointer_to_temperature(self.particle_data, temp)
                return temp[0]

    IF ROTATION:
        property rotation:
            """
            Switches the particle's rotational degrees of freedom in the
            Cartesian axes in the body-fixed frame. The content of the torque
            and omega variables are meaningless for the co-ordinates for which
            rotation is disabled.

            The default is not to integrate any rotational degrees of freedom.

            rotation : (3,) array_like of :obj:`int`

            .. note::
                This needs the feature ``ROTATION``.

            """

            def __set__(self, _rot):
                cdef int rot
                check_type_or_throw_except(
                    _rot, 3, int, "The rotation flag has to be given as a tuple of 3 integers.")

                rot = 0
                if _rot[0]:
                    rot += ROT_X
                if _rot[1]:
                    rot += ROT_Y
                if _rot[2]:
                    rot += ROT_Z
                set_particle_rotation(self._id, rot) 

            def __get__(self):
                self.update_particle_data()
                cdef const int * _rot = NULL
                pointer_to_rotation(self.particle_data, _rot)
                rot = _rot[0]
                res = np.zeros(3, dtype=int)
                if rot & ROT_X:
                    res[0] = 1
                if rot & ROT_Y:
                    res[1] = 1
                if rot & ROT_Z:
                    res[2] = 1
                return array_locked(res)

    IF EXCLUSIONS:
        property exclusions:
            """
            The exclusion list of particles where nonbonded interactions are ignored.

            .. note::
                This needs the feature ``EXCLUSIONS``.

            """

            def __set__(self, _partners):
                # Delete all
                for e in self.exclusions:
                    self.delete_exclusion(e)

                nlvl = nesting_level(_partners)

                if nlvl == 0:  # Single item
                    self.add_exclusion(_partners)
                elif nlvl == 1:  # List of items
                    for partner in _partners:
                        self.add_exclusion(partner)
                else:
                    raise ValueError(
                        "Exclusions have to be specified as a lists of partners or a single item.")

            def __get__(self):
                self.update_particle_data()
                cdef List[int] exclusions = self.particle_data.exclusions()

                py_partners = []
                for i in range(exclusions.n):
                    py_partners.append(exclusions.e[i])
                return array_locked(py_partners)

        def add_exclusion(self, _partner):
            """
            Excluding interaction with the given partner.

            Parameters
            -----------
            _partner : :obj:`int`
                partner

            """
            if _partner in self.exclusions:
                raise Exception("Exclusion id {} already in exclusion list of particle {}".format(
                    _partner, self._id))

            check_type_or_throw_except(
                _partner, 1, int, "PID of partner has to be an int.")
            if self._id == _partner:
                raise Exception(
                    "Cannot exclude of a particle with itself!\n->particle id %i, partner %i." % (self._id, _partner))
            if change_exclusion(self._id, _partner, 0) == 1:
                raise Exception("Particle with id " +
                                str(_partner) + " does not exist.")

        def delete_exclusion(self, _partner):
            check_type_or_throw_except(
                _partner, 1, int, "PID of partner has to be an int.")
            if not _partner in self.exclusions:
                raise Exception("Particle with id " +
                                str(_partner) + " is not in exclusion list.")
            if change_exclusion(self._id, _partner, 1) == 1:
                raise Exception("Particle with id " +
                                str(_partner) + " does not exist.")

    IF ENGINE:
        property swimming:
            """
            Set swimming parameters.

            This property takes a dictionary with a different number of
            entries depending whether there is an implicit fluid (i.e. with the
            Langevin thermostat) of an explicit fluid (with LB).

            Swimming enables the particle to be self-propelled in the direction
            determined by its quaternion. For setting the quaternion of the
            particle see :attr:`~espressomd.particle_data.ParticleHandle.quat`. The
            self-propulsion speed will relax to a constant velocity, that is specified by
            ``v_swim``. Alternatively it is possible to achieve a constant velocity by
            imposing a constant force term ``f_swim`` that is balanced by friction of a
            (Langevin) thermostat. The way the velocity of the particle decays to the
            constant terminal velocity in either of these methods is completely
            determined by the friction coefficient. You may only set one of the
            possibilities ``v_swim`` *or* ``f_swim`` as you cannot relax to constant force
            *and* constant velocity at the same time. The setting both ``v_swim`` and
            ``f_swim`` to 0.0 thus disables swimming. This option applies to all
            non-lattice-Boltzmann thermostats. Note that there is no real difference
            between ``v_swim`` and ``f_swim`` since the latter may aways be chosen such that
            the same terminal velocity is achieved for a given friction coefficient.


            Parameters
            ----------
            f_swim : :obj:`float`
                Achieve a constant velocity by imposing a constant
                force term ``f_swim`` that is balanced by friction of a
                (Langevin) thermostat. This exludes the option ``v_swim``.
            v_swim : :obj:`float`
                Achieve a constant velocity by imposing a constant terminal
                velocity ``v_swim``. This exludes the option ``f_swim``.
            mode : :obj:`str`, \{'pusher', 'puller'\}
                The LB flow field can be generated by a pushing or a
                pulling mechanism, leading to change in the sign of the
                dipolar flow field with respect to the direction of motion.
            dipole_length : :obj:`float`
                This determines the distance of the source of
                propulsion from the particle's center.
            rotational_friction : :obj:`float`
                This key can be used to set the friction that causes
                the orientation of the particle to change in shear
                flow. The torque on the particle is determined by
                taking the cross product of the difference between the
                fluid velocity at the center of the particle and at
                the source point and the vector connecting the center
                and source.

            Notes
            -----
            This needs the feature ``ENGINE``.  The keys ``'mode'``,
            ``'dipole_length'``, and ``'rotational_friction'`` are only
            available if ``ENGINE`` is used with LB or ``CUDA``.

            Examples
            --------
            >>> import espressomd
            >>>
            >>> system = espressomd.System()
            >>>
            >>> # Usage with Langevin
            >>> system.part.add(id=0, pos=[1,0,0],swimming={'f_swim':0.03})
            >>>
            >>> # Usage with LB
            >>> system.part.add(id=1, pos=[2,0,0], swimming={'f_swim': 0.01,
            ...     'mode': 'pusher', 'dipole_length': 2.0, 'rotational_friction': 20})

            """

            def __set__(self, _params):
                global swimming_particles_exist
                cdef particle_parameters_swimming swim
                swim.swimming = True
                swim.v_swim = 0.0
                swim.f_swim = 0.0
                swim.push_pull = 0
                swim.dipole_length = 0.0
                swim.rotational_friction = 0.0
 
                if type(_params) == type(True):
                    if _params:
                        raise Exception(
                            "To enable swimming supply a dictionary of parameters.")
                else:
                    if 'f_swim' in _params and 'v_swim' in _params:
                        if _params["f_swim"] == 0 or _params["v_swim"] == 0:
                            pass
                        else:
                            raise Exception(
                                "You can't set v_swim and f_swim at the same time.")
                    if 'f_swim' in _params:
                        check_type_or_throw_except(
                            _params['f_swim'], 1, float, "f_swim has to be a float.")
                        swim.f_swim = _params['f_swim']
                    if 'v_swim' in _params:
                        check_type_or_throw_except(
                            _params['v_swim'], 1, float, "v_swim has to be a float.")
                        swim.v_swim = _params['v_swim']

                    if 'mode' in _params:
                        if _params['mode'] == "pusher":
                            swim.push_pull = -1
                        elif _params['mode'] == "puller":
                            swim.push_pull = 1
                        elif _params['mode'] == "N/A":
                            swim.push_pull = 0
                        else:
                            raise Exception(
                                "'mode' has to be either 'pusher' or 'puller'.")

                    if 'dipole_length' in _params:
                        check_type_or_throw_except(
                            _params['dipole_length'], 1, float, "dipole_length has to be a float.")
                        swim.dipole_length = _params['dipole_length']

                    if 'rotational_friction' in _params:
                        check_type_or_throw_except(
                            _params['rotational_friction'], 1, float, "rotational_friction has to be a float.")
                        swim.rotational_friction = _params[
                            'rotational_friction']

                if swim.f_swim != 0 or swim.v_swim != 0:
                    swimming_particles_exist = True
                    mpi_bcast_parameter(FIELD_SWIMMING_PARTICLES_EXIST)

                set_particle_swimming(self._id, swim) 

            def __get__(self):
                self.update_particle_data()
                swim = {}
                mode = "N/A"
                cdef const particle_parameters_swimming * _swim = NULL
                pointer_to_swimming(self.particle_data, _swim)

                if _swim.push_pull == -1:
                    mode = 'pusher'
                elif _swim.push_pull == 1:
                    mode = 'puller'
                swim = {
                    'v_swim': _swim.v_swim,
                    'f_swim': _swim.f_swim,
                    'mode': mode,
                    'dipole_length': _swim.dipole_length,
                    'rotational_friction': _swim.rotational_friction
                }

                return swim

    def remove(self):
        """
        Delete the particle.

        See Also
        --------
        espressomd.particle_data.ParticleList.add
        espressomd.particle_data.ParticleList.clear

        """
        if remove_particle(self._id):
            raise Exception("Could not remove particle.")
        del self

    # Bond related methods
    # Does not work properly with 3 or more partner bonds!

    def add_verified_bond(self, bond):
        """
        Add a bond, the validity of which has already been verified.

        See Also
        --------
        add_bond : Delete an unverified bond held by the `Particle`.
        bonds : `Particle` property containing a list of all current bonds help by `Particle`.

        """

        # If someone adds bond types with more than four partners, this has to
        # be changed
        cdef int bond_info[5]
        bond_info[0] = bond[0]._bond_id
        for i in range(1, len(bond)):
            bond_info[i] = bond[i]
        if self._id in bond[1:]:
            raise Exception(
                "Bond partners {} include the particle {} itself.".format(bond[1:], self._id))

        add_particle_bond(self._id, make_const_span[int](bond_info, len(bond)))

    def delete_verified_bond(self, bond):
        """
        Delete a single bond from the particle. The validity of which has already been verified.

        Parameters
        ----------
        bond : :obj:`tuple`
            tuple where the first element is either a bond ID of a bond type,
            and the last element is the ID of the parter particle to be bonded
            to.


        See Also
        --------
        delete_bond : Delete an unverified bond held by the `Particle`.
        bonds : `Particle` property containing a list of all current bonds help by `Particle`.

        """

        cdef int bond_info[5]
        bond_info[0] = bond[0]._bond_id
        for i in range(1, len(bond)):
            bond_info[i] = bond[i]

        delete_particle_bond(
            self._id, make_const_span[int](bond_info, len(bond)))

    def check_bond_or_throw_exception(self, bond):
        """
        Checks the validity of the given bond:

            - If the bondtype is given as an object or a numerical id
            - If all partners are of type int
            - If the number of partners satisfies the bond
            - If the bond type used exists (is lower than n_bonded_ia)
            - If the number of bond partners fits the bond type

        Throws an exception if any of these are not met.

        """
        # Has it []-access
        if not hasattr(bond, "__getitem__"):
            raise ValueError(
                "Bond needs to be a tuple or list containing bond type and partners.")

        # Bond type or numerical bond id
        if not isinstance(bond[0], BondedInteraction):
            if is_valid_type(bond[0], int):
                bond[0] = BondedInteractions()[bond[0]]
            else:
                raise Exception(
                    "1st element of Bond has to be of type BondedInteraction or int.")

        # Check whether the bond has been added to the list of active bonded
        # interactions
        if bond[0]._bond_id == -1:
            raise Exception(
                "The bonded interaction has not yet been added to the list of active bonds in Espresso.")

        # Validity of the numeric id
        if bond[0]._bond_id >= bonded_ia_params.size():
            raise ValueError(
                "The bond type", bond[0]._bond_id, "does not exist.")

        bond_id = bond[0]._bond_id
        # Number of partners
        if bonded_ia_params[bond_id].num != len(bond) - 1:
            raise ValueError("Bond of type", bond[
                             0]._bond_id, "needs", bonded_ia_params[bond_id].num, "partners.")

        # Type check on partners
        for i in range(1, len(bond)):
            if not is_valid_type(bond[i], int):
                if not isinstance(bond[i], ParticleHandle):
                    raise ValueError(
                        "Bond partners have to be of type integer or ParticleHandle.")
                else:
                    # Put the particle id instead of the particle handle
                    bond[i] = bond[i].id

    def add_bond(self, _bond):
        """
        Add a single bond to the particle.

        Parameters
        ----------
        _bond : :obj:`tuple`
            tuple where the first element is either a bond ID of a bond type,
            and the last element is the ID of the parter particle to be bonded
            to.


        See Also
        --------
        bonds : `Particle` property containing a list of all current bonds help by `Particle`.

        Examples
        --------
        >>> import espressomd
        >>> from espressomd.interactions import HarmonicBond
        >>>
        >>> system = espressomd.System()
        >>>
        >>> # define a harmonic potential and add it to the system
        >>> harm_bond = HarmonicBond(r_0=1, k=5)
        >>> system.bonded_inter.add(harm_bond)
        >>>
        >>> # add two particles
        >>> system.part.add(id=0, pos=(1, 0, 0))
        >>> system.part.add(id=1, pos=(2, 0, 0))
        >>>
        >>> # bond them via the bond type
        >>> system.part[0].add_bond((harm_bond,1))
        >>> # or via the bond index (zero in this case since it is the first one added)
        >>> system.part[0].add_bond((0,1))

        """

        if tuple(_bond) in self.bonds:
            raise Exception("Bond {} already exists on particle {}.".format(
                tuple(_bond), self._id))

        bond = list(_bond)  # As we will modify it
        self.check_bond_or_throw_exception(bond)
        self.add_verified_bond(bond)

    def delete_bond(self, _bond):
        """
        Delete a single bond from the particle.

        Parameters
        ----------
        _bond : bond to be deleted

        See Also
        --------
        bonds :  Particle property, a list of all current bonds.


        Examples
        --------

        >>> import espressomd
        >>> from espressomd.interactions import HarmonicBond
        >>>
        >>> system = espressomd.System()

        define a harmonic potential and add it to the system

        >>> harm_bond = HarmonicBond(r_0=1, k=5)
        >>> system.bonded_inter.add(harm_bond)

        add two bonded particles to particle 0

        >>> system.part.add(id=0, pos=(1, 0, 0))
        >>> system.part.add(id=1, pos=(2, 0, 0))
        >>> system.part.add(id=2, pos=(1, 1, 0))
        >>> system.part[0].add_bond((harm_bond,1))
        >>> system.part[0].add_bond((harm_bond,2))
        >>>
        >>> bonds = system.part[0].bonds
        >>> print(bonds)
        ((HarmonicBond(0): {'r_0': 1.0, 'k': 5.0, 'r_cut': 0.0}, 1), (HarmonicBond(0): {'r_0': 1.0, 'k': 5.0, 'r_cut': 0.0}, 2))

        delete the bond between particle 0 and particle 1

        >>> system.part[0].delete_bond(bonds[0])
        >>> print(system.part[0].bonds)
        ((HarmonicBond(0): {'r_0': 1.0, 'k': 5.0, 'r_cut': 0.0}, 2),)

        """

        bond = list(_bond)  # as we modify it
        self.check_bond_or_throw_exception(bond)
        self.delete_verified_bond(bond)

    def delete_all_bonds(self):
        """
        Delete all bonds from the particle.

        See Also
        ----------
        delete_bond : Delete an unverified bond held by the particle.
        bonds : Particle property containing a list of all current bonds help by particle.

        """

        delete_particle_bonds(self._id)

    def update(self, P):
        if "id" in P:
            raise Exception("Cannot change particle id.")

        for k in P.keys():
            setattr(self, k, P[k])

    IF ROTATION:
        def convert_vector_body_to_space(self, vec):
            """Converts the given vector from the particle's body frame to the space frame"""
            cdef Vector3d res
            cdef Vector3d _v
            _v[0] = vec[0]
            _v[1] = vec[1]
            _v[2] = vec[2]
            self.update_particle_data()
            res = convert_vector_body_to_space(self.particle_data[0], _v)
            return np.array((res[0], res[1], res[2]))

        def convert_vector_space_to_body(self, vec):
            """Converts the given vector from the space frame to the particle's body frame"""
            cdef Vector3d res
            cdef Vector3d _v
            _v[0] = vec[0]
            _v[1] = vec[1]
            _v[2] = vec[2]
            self.update_particle_data()
            res = convert_vector_space_to_body(self.particle_data[0], _v)
            return np.array((res[0], res[1], res[2]))

        def rotate(self, axis=None, angle=None):
            """Rotates the particle around the given axis

            Parameters
            ----------
            axis : array-like

            angle : float

            """
            cdef Vector3d a
            a[0] = axis[0]
            a[1] = axis[1]
            a[2] = axis[2]

            rotate_particle(self._id, a, angle)

cdef class _ParticleSliceImpl:
    """Handles slice inputs.

    This base class should not be used directly. Use
    :class:`espressomd.ParticleSlice` instead, which contains all the particle
    properties.

    """

    def __cinit__(self, slice_, prefetch_chunk_size=10000):
        self._chunk_size = prefetch_chunk_size

        id_list = np.arange(max_seen_particle + 1)
        self.id_selection = id_list[slice_]
        mask = np.empty(len(self.id_selection), dtype=np.bool)
        cdef int i
        for i in range(len(self.id_selection) - 1, -1, -1):
            mask[i] = particle_exists(i)
        self.id_selection = self.id_selection[mask]

    def __iter__(self):
        return self._id_gen()

    def _id_gen(self):
        """Generator for chunked and prefetched iteration of particles.
        """
        for chunk in self.chunks(self.id_selection, self._chunk_size):
            prefetch_particle_data(chunk)
            for i in chunk:
                yield ParticleHandle(i)

    def chunks(self, l, n):
        """Generator returning chunks of length n from l.
        """
        for i in range(0, len(l), n):
            yield l[i:i + n]

    def __len__(self):
        return len(self.id_selection)

    property pos_folded:
        """
        Particle position (folded into central image).

        """

        def __get__(self):
            pos_array = np.zeros((len(self.id_selection), 3))
            for i in range(len(self.id_selection)):
                pos_array[i, :] = ParticleHandle(
                    self.id_selection[i]).pos_folded
            return pos_array

    IF EXCLUSIONS:
        def add_exclusion(self, _partner):
            for i in self.id_selection:
                ParticleHandle(i).add_exclusion(_partner)

        def delete_exclusion(self, _partner):
            for i in self.id_selection:
                ParticleHandle(i).delete_exclusion(_partner)

    def __str__(self):
        res = ""
        pl = ParticleList()
        for i in self.id_selection:
            if pl.exists(i):
                res += str(pl[i]) + ", "
        # Remove final comma
        return "ParticleSlice([" + res[:-2] + "])"

    def update(self, P):
        if "id" in P:
            raise Exception("Cannot change particle id.")

        for k in P.keys():
            setattr(self, k, P[k])

    # Bond related methods
    def add_bond(self, _bond):
        """
        Add a single bond to the particles.

        """
        for i in self.id_selection:
            ParticleHandle(i).add_bond(_bond)

    def delete_bond(self, _bond):
        """
        Delete a single bond from the particles.

        """
        for i in self.id_selection:
            ParticleHandle(i).delete_bond(_bond)

    def delete_all_bonds(self):
        for i in self.id_selection:
            ParticleHandle(i).delete_all_bonds()

    def remove(self):
        """
        Delete the particles.

        See Also
        --------
        add

        """
        for id in self.id_selection:
            ParticleHandle(id).remove()


class ParticleSlice(_ParticleSliceImpl):

    """
    Handles slice inputs e.g. part[0:2]. Sets values for selected slices or
    returns values as a single list.

    """

    def __setattr__(self, name, value):
        if name != "_chunk_size" and not hasattr(ParticleHandle, name):
            raise AttributeError(
                "ParticleHandle does not have the attribute {}.".format(name))
        super().__setattr__(name, value)


cdef class ParticleList:
    """
    Provides access to the particles via [i], where i is the particle id.
    Returns a :class:`ParticleHandle` object.

    """

    # Retrieve a particle

    def __getitem__(self, key):
        if isinstance(key, slice):
            return ParticleSlice(key)

        try:
            if isinstance(key, range):
                return ParticleSlice(key)
        except:
            pass

        if isinstance(key, tuple) or isinstance(key, list) or isinstance(key, np.ndarray):
            return ParticleSlice(np.array(key))

        return ParticleHandle(key)

    # __getstate__ and __setstate__ define the pickle interaction
    def __getstate__(self):
        """Attributes to pickle.
        Content of particle_attributes, minus a few exceptions dip, director:
        Setting only the director will overwrite the orientation of the
        particle around the axis parallel to dipole moment/director.
        Quaternions contain the full info id: The particle id is used as the
        storage key when pickling all particles via ParticleList, and the
        interface (rightly) does not support changing of the id after the
        particle was created.

        """

        pickle_attr = copy(particle_attributes)
        for i in ["director", "dip", "id", "image_box", "node"]:
            if i in pickle_attr:
                pickle_attr.remove(i)
        IF MASS == 0:
            pickle_attr.remove("mass")
        odict = {}
        key_list = [p.id for p in self]
        for particle_number in key_list:
            pdict = {}

            for property_ in pickle_attr:
                pdict[property_] = ParticleHandle(
                    particle_number).__getattribute__(property_)
            odict[particle_number] = pdict
        return odict

    def __setstate__(self, params):
        exclusions = collections.OrderedDict()
        for particle_number in params.keys():
            params[particle_number]["id"] = particle_number
            IF EXCLUSIONS:
                exclusions[particle_number] = params[particle_number][
                    "exclusions"]
                del params[particle_number]["exclusions"]
            self._place_new_particle(params[particle_number])
        IF EXCLUSIONS:
            for pid in exclusions:
                self[pid].exclusions = exclusions[pid]

    def __len__(self):
        return n_part

    def add(self, *args, **kwargs):
        """
        Adds one or several particles to the system

        Parameters
        ----------
        Either a dictionary or a bunch of keyword args.

        Returns
        -------
        Returns an instance of :class:`espressomd.particle_data.ParticleHandle` for each added particle.

        See Also
        --------
        :meth:`espressomd.particle_data.ParticleHandle.remove`

        Examples
        --------

        >>> import espressomd
        >>>
        >>> system = espressomd.System()
        >>>
        >>> # add two particles
        >>> system.part.add(id=0, pos=(1, 0, 0))
        >>> system.part.add(id=1, pos=(2, 0, 0))

        Pos is mandatory, id can be omitted, in which case it is assigned automatically.
        Several particles can be added by passing one value per particle to each property::

            system.part.add(pos=((1,2,3),(4,5,6)),q=(1,-1))

        """

        # Did we get a dictionary
        if len(args) == 1:
            if hasattr(args[0], "__getitem__"):
                P = args[0]
        else:
            if len(args) == 0 and len(kwargs.keys()) != 0:
                P = kwargs
            else:
                raise ValueError(
                    "add() takes either a dictionary or a bunch of keyword args.")

        # Check for presence of pos attribute
        if not "pos" in P:
            raise ValueError(
                "pos attribute must be specified for new particle")

        if len(np.array(P["pos"]).shape) == 2:
            return self._place_new_particles(P)
        else:
            return self._place_new_particle(P)

    def _place_new_particle(self, P):
        # Handling of particle id
        if not "id" in P:
            # Generate particle id
            P["id"] = max_seen_particle + 1
        else:
            if particle_exists(P["id"]):
                raise Exception("Particle %d already exists." % P["id"])

        # Prevent setting of contradicting attributes
        IF DIPOLES:
            if 'dip' in P and 'dipm' in P:
                raise ValueError("Contradicting attributes: dip and dipm. Setting \
dip is sufficient as the length of the vector defines the scalar dipole moment.")
            IF ROTATION:
                if 'dip' in P and 'quat' in P:
                    raise ValueError("Contradicting attributes: dip and quat. \
Setting dip overwrites the rotation of the particle around the dipole axis. \
Set quat and scalar dipole moment (dipm) instead.")

        # The ParticleList[]-getter ist not valid yet, as the particle
        # doesn't yet exist. Hence, the setting of position has to be
        # done here. the code is from the pos:property of ParticleHandle
        cdef double mypos[3]
        check_type_or_throw_except(
            P["pos"], 3, float, "Position must be 3 floats.")
        for i in range(3):
            mypos[i] = P["pos"][i]
        if place_particle(P["id"], mypos) == -1:
            raise Exception("particle could not be set.")

        # Pos is taken care of
        del P["pos"]
        id = P["id"]
        del P["id"]

        if P != {}:
            self[id].update(P)

        return self[id]

    def _place_new_particles(self, Ps):
        # Check if all entries have the same length
        n_parts = len(Ps["pos"])
        if not all(np.shape(Ps[k]) and len(Ps[k]) == n_parts for k in Ps):
            raise ValueError(
                "When adding several particles at once, all lists of attributes have to have the same size")

        # Place new particles and collect ids
        ids = []
        for i in range(n_parts):
            P = {}
            for k in Ps:
                P[k] = Ps[k][i]
            ids.append(self._place_new_particle(P).id)

        return self[ids]

    # Iteration over all existing particles
    def __iter__(self):
        for i in range(max_seen_particle + 1):
            if particle_exists(i):
                yield self[i]

    def exists(self, idx):
        if is_valid_type(idx, int):
            return particle_exists(idx)
        if isinstance(idx, slice) or isinstance(idx, tuple) or isinstance(idx, list) or isinstance(idx, np.ndarray):
            tf_array = np.zeros(len(idx), dtype=np.bool)
            for i in range(len(idx)):
                tf_array[i] = particle_exists(idx[i])
            return tf_array

    def clear(self):
        """
        Removes all particles.

        See Also
        --------
        add
        :meth:`espressomd.particle_data.ParticleHandle.remove`

        """

        remove_all_particles()

    def __str__(self):
        res = ""
        for i in range(max_seen_particle + 1):
            if self.exists(i):
                res += str(self[i]) + ", "
        # Remove final comma
        return "ParticleList([" + res[:-2] + "])"

    def writevtk(self, fname, types='all'):
        """
        Write the positions and velocities of particles with specified
        types to a VTK file.

        Parameters
        ----------
        fname: :obj:`str`
            Filename of the target output file
        types: list of :obj:`int` or the string 'all', optional (default: 'all')
            A list of particle types which should be output to 'fname'

        Examples
        --------

        >>> import espressomd
        >>>
        >>> system = espressomd.System()
        >>>
        >>> # add several particles
        >>> system.part.add(pos=.5*system.box_l,v=[1,0,0],type=0)
        >>> system.part.add(pos=.4*system.box_l,v=[0,2,0],type=1)
        >>> system.part.add(pos=.7*system.box_l,v=[2,0,1],type=1)
        >>> system.part.add(pos=.1*system.box_l,v=[0,0,1],type=2)
        >>>
        >>> # write to VTK
        >>> system.part.writevtk("part_type_0_1.vtk", types=[0,1])
        >>> system.part.writevtk("part_type_2.vtk", types=[2])
        >>> system.part.writevtk("part_all.vtk")

        .. todo:: `move to ./io/writer/`

        """

        global box_l
        if not hasattr(types, '__iter__'):
            types = [types]

        n = 0
        for p in self:
            for t in types:
                if p.type == t or t == "all":
                    n += 1

        with open(fname, "w") as vtk:
            vtk.write("# vtk DataFile Version 2.0\n")
            vtk.write("particles\n")
            vtk.write("ASCII\n")
            vtk.write("DATASET UNSTRUCTURED_GRID\n")
            vtk.write("POINTS {} floats\n".format(n))
            for p in self:
                for t in types:
                    if p.type == t or t == "all":
                        vtk.write("{} {} {}\n".format(*(p.pos_folded)))

            vtk.write("POINT_DATA {}\n".format(n))
            vtk.write("SCALARS velocity float 3\n")
            vtk.write("LOOKUP_TABLE default\n")
            for p in self:
                for t in types:
                    if p.type == t or t == "all":
                        vtk.write("{} {} {}\n".format(*p.v))

    property highest_particle_id:
        """
        Largest particle id.

        """

        def __get__(self):
            return max_seen_particle

    property n_part_types:
        """
        Number of particle types.

        """

        def __get__(self):
            return max_seen_particle_type

    property n_rigidbonds:
        """
        Number of rigid bonds.

        """

        def __get__(self):
            return n_rigidbonds

    # property max_part:
    #     def __get__(self):
    #         return max_seen_particle

    # # property n_part:
    #     def __get__(self):
    #         return n_part

    def pairs(self):
        """
        Generator returns all pairs of particles.

        """

        cdef int i
        cdef int j
        for i in range(max_seen_particle + 1):
            for j in range(i + 1, max_seen_particle + 1, 1):
                if not (particle_exists(i) and particle_exists(j)):
                    continue

                yield (self[i], self[j])

    def select(self, *args, **kwargs):
        """Generates a particle slice by filtering particles via a user-defined criterion

        Parameters:

        Either: a keyword arguments in which the keys are names of particle
        properties and the values are the values to filter for. E.g.,::

            system.part.select(type=0, q=1)

        Or: a function taking a ParticleHandle as argument and returning True if
        the particle is to be filtered for. E.g.,::

            system.part.select(lambda p: p.pos[0] < 0.5)

        Returns
        -------
        :class:`ParticleSlice` :
            An instance of :class:`ParticleSlice` containing the selected particles

        """

        # Ids of the selected particles
        ids = []
        # Did we get a function as argument?
        if len(args) == 1 and len(kwargs) == 0 and isinstance(args[0], types.FunctionType):
            # Go over all particles and pass them to the user-provided function
            for p in self:
                if args[0](p):
                    ids.append(p.id)
            return ParticleSlice(ids)

        # Did we get a set of keyword args?
        elif len(args) == 0:
            for p in self:
                select = True
                # Check, if the particle fails any required criteria
                for k in kwargs:
                    # Fetch user-provided value and value in particle
                    val1 = kwargs[k]
                    val2 = getattr(p, k)
                    # Get tolerance from numerical accuracy limits
                    tol = max(
                        np.amax(np.spacing(val1)), np.amax(np.spacing(val2)))

                    # Compare
                    if not np.allclose(val1, val2, atol=tol):
                        select = False
                        break
                if select:
                    ids.append(p.id)
            return ParticleSlice(ids)
        else:
            raise Exception(
                "select() takes either selection function as positional argument or a set of keyword arguments.")


def set_slice_one_for_all(particle_slice, attribute, values):
    for i in particle_slice.id_selection:
        setattr(ParticleHandle(i), attribute, values)


def set_slice_one_for_each(particle_slice, attribute, values):
    for i, v in zip(particle_slice.id_selection, values):
        setattr(ParticleHandle(i), attribute, v)


def _add_particle_slice_properties():
    """
    Automatically add all of ParticleHandle's properties to ParticleSlice.

    """

    def seta(particle_slice, values, attribute):
        """
        Setter function that sets attribute on every member of particle_slice.
        If values contains only one element, all members are set to it. If it
        contains as many elements as there are members, each of them gets set
        to the corresponding one. For attributes that are lists of various length,
        (bonds, exclusions) the nesting level decides if it is one-for-all or one-for-each.

        """

        N = len(particle_slice.id_selection)

        # Special attributes
        if attribute == "bonds":
            nlvl = nesting_level(values)
            if nlvl == 1 or nlvl == 2:
                set_slice_one_for_all(particle_slice, attribute, values)
            elif nlvl == 3 and len(values) == N:
                set_slice_one_for_each(particle_slice, attribute, values)
            else:
                raise Exception("Failed to set bonds for particle slice.")

            return

        elif attribute == "exclusions":
            nlvl = nesting_level(values)
            if nlvl == 0 or nlvl == 1:
                set_slice_one_for_all(particle_slice, attribute, values)
            elif nlvl == 2 and len(values) == N:
                set_slice_one_for_each(particle_slice, attribute, values)
            else:
                raise Exception("Failed to set exclusions for particle slice.")

            return

        elif attribute == "vs_relative":
            nlvl = nesting_level(values)
            if nlvl in [1, 2]:
                set_slice_one_for_all(particle_slice, attribute, values)
            elif nlvl == 3 and len(values) == N:
                set_slice_one_for_each(particle_slice, attribute, values)
            else:
                raise Exception(
                    "Failed to set vs_relative for particle slice.")

            return

        else:
            target = getattr(
                ParticleHandle(particle_slice.id_selection[0]), attribute)
            target_shape = np.shape(target)

            if not target_shape:  # scalar quantity
                if not np.shape(values):
                    set_slice_one_for_all(particle_slice, attribute, values)
                elif np.shape(values)[0] == N:
                    set_slice_one_for_each(particle_slice, attribute, values)
                else:
                    raise Exception("Shape of value (%s) does not broadcast to shape of attribute (%s)." % (
                        np.shape(values), target_shape))

                return

            else:  # fixed length vector quantity

                if target_shape == np.shape(values):
                    set_slice_one_for_all(particle_slice, attribute, values)
                elif target_shape == tuple(np.shape(values)[1:]) and np.shape(values)[0] == N:
                    set_slice_one_for_each(particle_slice, attribute, values)
                else:
                    raise Exception("Shape of value (%s) does not broadcast to shape of attribute (%s)." % (
                        np.shape(values), target_shape))

                return

    def geta(particle_slice, attribute):
        """
        Getter function that copies attribute from every member of
        particle_slice into an array (if possible).
        For special properties, a tuple of tuples is used.

        """

        N = len(particle_slice.id_selection)
        if N == 0:
            return np.empty(0, dtype=type(None))

        # get first slice member to determine its type
        target = getattr(ParticleHandle(
            particle_slice.id_selection[0]), attribute)
        if type(target) is array_locked:  # vectorial quantity
            target_type = target.dtype
        else:  # scalar quantity
            target_type = type(target)

        if attribute in ["exclusions", "bonds", "vs_relative", "swimming"]:
            values = []
            for part in particle_slice._id_gen():
                values.append(getattr(part, attribute))
        else:
            values = np.empty((N,) + np.shape(target), dtype=target_type)
            i = 0
            for part in particle_slice._id_gen():
                values[i] = getattr(part, attribute)
                i += 1

        return values

    for attribute_name in particle_attributes:
        if attribute_name in dir(ParticleSlice):
            continue

        # synthesize a new property
        new_property = property(functools.partial(geta, attribute=attribute_name), functools.partial(
            seta, attribute=attribute_name), doc=getattr(ParticleHandle, attribute_name).__doc__)
        # attach the property to ParticleSlice
        setattr(ParticleSlice, attribute_name, new_property)


_add_particle_slice_properties()
