#' consolidate.tracksom.result
#' 
#' Given a tracksom object and a dataframe containing cells from all time points, 
#' extract all the meta clusters, clusters, and tracking id and combine them with the cell information.
#' 
#' @param tracksom.result NO DEFAULT. TrackSOM object as generated by [TrackSOM()]
#' @param dat NO DEFAULT. Dataframe containing all cells from all time points
#' @param divide.by NO DEFAULT. Character showing column name of the dataframe that indicate the time point of each cell
#' 
#' @return Dataframe containing the meta cluster, cluster, and tracking ids for all cells

consolidate.tracksom.result <- function(tracksom.result, dat, divide.by) {
  time.points <- names(tracksom.result$metaclustering$mcPerCell)
  
  cell.dat <- data.frame(dat)
  
  # append the cluster/code to cell.dat
  fsom.code.colname <- 'TrackSOM_cluster'
  fsom.codes <- data.frame(tracksom.result$FlowSOM$map$mapping[, 1])
  names(fsom.codes) <- fsom.code.colname
  cell.dat <- cbind(cell.dat, fsom.codes)
  
  # divide cell.dat based on the time point
  divide.list <- unique(cell.dat[[divide.by]])
  cell.dat.list <- lapply(divide.list, function(i) {
    return(subset(cell.dat, cell.dat[[divide.by]] == i))
  })
  
  metacluster.per.tp <- lapply(time.points, function(time.point) {
    return(tracksom.result$metaclustering$mcPerCell[[time.point]])
  })
  
  # get the lineage id for all cells in all time point
  lineage.ids.per.tp <- lapply(seq_along(cell.dat.list), function(i) {
    # isolate just the code for all cells in a time point
    c.dat <- cell.dat.list[[i]]
    fsom.code <- c.dat[[fsom.code.colname]]
    
    # get the lineage id for each code. TrackSOM output the lineage id of the code instead of metacluster, thus complicated.
    lineage.fsom.code.mapping <- tracksom.result$tracking$lineage[[paste("timestep", i, sep = "_")]]
    
    # code mapping is stored in numerical order i.e. vector index 1 is for code 1.
    # thus to get the lineage id for code 1, just accesss the value inside vector index 1
    fsom.code.mapped <- sapply(fsom.code, function(code){
      return(lineage.fsom.code.mapping[code])
    })
    
    return(fsom.code.mapped)
  })
  
  # get the proximity id for all cells in all time point
  proximity.ids.per.tp <- lapply(seq_along(cell.dat.list), function(i) {
    # isolate just the code for all cells in a time point
    c.dat <- cell.dat.list[[i]]
    fsom.code <- c.dat[[fsom.code.colname]]
    
    # get the lineage id for each code. TrackSOM output the lineage id of the code instead of metacluster, thus complicated.
    proximity.fsom.code.mapping <- tracksom.result$tracking$proximity[[i]]
    
    if(is.null(proximity.fsom.code.mapping)) {
      # very first time point
      return(rep(NA, dim(c.dat)[1]))
    }
    
    # code mapping is stored in numerical order i.e. vector index 1 is for code 1.
    # thus to get the lineage id for code 1, just accesss the value inside vector index 1
    fsom.code.mapped <- sapply(fsom.code, function(code){
      return(proximity.fsom.code.mapping[code])
    })
    
    return(fsom.code.mapped)
  })
  
  # append the meta cluster and lineage id onto the cells dataframe
  complete.cell.dat.list <- lapply(seq_along(cell.dat.list), function(i) {
    c.dat <- data.frame(cell.dat.list[[i]])
    
    metaclusters <- data.frame(metacluster.per.tp[[i]])
    names(metaclusters) <- "TrackSOM_metacluster"
    
    lineage.ids <- data.frame(lineage.ids.per.tp[[i]])
    names(lineage.ids) <- "TrackSOM_metacluster_lineage_tracking"
    
    proximity.ids <- data.frame(proximity.ids.per.tp[[i]])
    names(proximity.ids) <- "TrackSOM_metacluster_proximity_tracking"
    
    c.dat <- cbind(c.dat, metaclusters, lineage.ids, proximity.ids)
    
    return(c.dat)
    
  })
  
  # combine everything into 1 giant data frame
  complete.cell.dat <- do.call(rbind, complete.cell.dat.list)
  
  return(complete.cell.dat)  
}
