#################################################################################
#
# Tutorial file how to read in hdf5 data from SP-Wind TotalControl Farmconners simulations.
#
#
# author: Ishaan Sood
# date: 01 Feb 2021
# email: sood.ishaan26@gmail.com
#
#################################################################################


import numpy as np
import matplotlib.pyplot as plt
import fatigue as ft
import h5py as hp

plt.close("all")


###########################################################################################################################
#
# FILE: SingleFullWake.h5/SinglePartialWake.h5/MultipleFullWake.h5
#
#   group: normal_op
#       description:    Normal performance data
#       dataset: performance_time_array - Time series for power and speed data, from 900 to 4500 in step of 1 s
#       dataset: structural_time_array - Time series for structural data, from 900 to 4500 in step of 0.01 s
#       subgroups: Power (P_i), Speed (REWS_i), Flapwise bending moment (FlapM_i), Tower bottom bending moment (TBBM_i)
#
#   group: controlled_op
#       description:    Controlled performance data
#       dataset: performance_time_array - Time series for power and speed data, from 900 to 4500 in step of 1 s
#       dataset: structural_time_array - Time series for structural data, from 900 to 4500 in step of 0.01 s
#       subgroups: Power (P_i), Speed (REWS_i), Flapwise bending moment (FlapM_i), Tower bottom bending moment (TBBM_i)
#
#   Turbine index 'i' is case dependant. Full description at https://farmconners.readthedocs.io/en/latest/test_cases.html :
#
#   SingleFullWake : Turbine 29 and 32 are yawed by 30 and 20 degrees respectively (counter-clockwise from top).
#                    Turbine 25 and 28 are the respective downstream turbines
#
#   MultipleFullWake : Turbine 29 and 32 are yawed by 30 and 20 degrees respectively (counter-clockwise from top).
#                      Turbine 25,21,17,13,9,5,1 and 28,24,20,16,12,8,4 are the respective downstream turbines
#
#   SinglePartiallWake : Turbine 5 and 25 are yawed by 20 and 30 degrees respectively (counter-clockwise from top).
#                        Turbine 10,6 and 30,26 are the respective downstream turbines
#
###########################################################################################################################

######## Data loading and analysis example for SingleFullWake

# open the file

meta_data = hp.File('SingleFullWake.h5', 'r')

perf_time = np.array(meta_data['performance_time_array'])
struct_time = np.array(meta_data['structural_time_array'])

######################### Quantities of interest examples

#### Power gain

P_29_n = np.array(meta_data['normal_op']['Power']['P_29'])
P_32_n = np.array(meta_data['normal_op']['Power']['P_32'])
P_25_n = np.array(meta_data['normal_op']['Power']['P_25'])
P_28_n = np.array(meta_data['normal_op']['Power']['P_28'])

P_29_c = np.array(meta_data['controlled_op']['Power']['P_29'])
P_32_c = np.array(meta_data['controlled_op']['Power']['P_32'])
P_25_c = np.array(meta_data['normal_op']['Power']['P_25'])
P_28_c = np.array(meta_data['controlled_op']['Power']['P_28'])

P_normal = P_29_n.mean(axis=1)+P_32_n.mean(axis=1)+P_25_n.mean(axis=1)+P_28_n.mean(axis=1)
P_wfc = P_29_c.mean(axis=1)+P_32_c.mean(axis=1)+P_25_c.mean(axis=1)+P_28_c.mean(axis=1)

del_P = P_wfc/P_normal

print('Power gain is =',del_P)

#### Wake added TI

U_25_n = np.array(meta_data['normal_op']['Speed']['REWS_25'])

U_25_c = np.array(meta_data['controlled_op']['Speed']['REWS_25'])

del_TI = U_25_n.std(axis=1)/U_25_n.mean(axis=1) - U_25_c.std(axis=1)/U_25_c.mean(axis=1)

print('Reduction in TI is =',U_25_n.std(axis=1))

### Load alleviation

N = 1e6 ## cycles
m = 10  ## slope of SN curve

## Blade root flapwise damage equivalent loads

FlapM_29_n = np.array(meta_data['normal_op']['FlapM']['FlapM_29'])

FlapM_29_c = np.array(meta_data['controlled_op']['FlapM']['FlapM_29'])

DEL_n = ft.EqLoad(FlapM_29_n[0],N,m)
DEL_c = ft.EqLoad(FlapM_29_c[0],N,m)

del_DEL = 1 - DEL_c/DEL_n

print('Flapwise loading change is =',del_DEL)

## Tower bottom damage equivalent loads

TBBM_29_n = np.array(meta_data['normal_op']['TBBM']['TBBM_29'])

TBBM_29_c = np.array(meta_data['controlled_op']['TBBM']['TBBM_29'])

DEL_n = ft.EqLoad(TBBM_29_n[0],N,m)
DEL_c = ft.EqLoad(TBBM_29_c[0],N,m)

del_DEL = 1 - DEL_c/DEL_n

print('Tower bottom loading change is =',del_DEL)