module types
implicit none
integer, parameter      :: dp = selected_real_kind(P=15)
end module types

module const
use types
integer,parameter       :: nang   = 91                ! angle mesh
integer,parameter       :: namax  = 1000              ! number of grain rad. mesh
real(kind=dp),parameter :: mic2cm = 1.0e-4_dp         ! micron to cm
real(kind=dp),parameter :: pi     = 3.1415926535897932384_dp
real(kind=dp),parameter :: twopi  = 2.0_dp * pi
real(kind=dp),parameter :: halfpi = 0.5_dp * pi
real(kind=dp),parameter :: r2d    = 180.0_dp / pi
integer :: k
real(kind=dp)           ::lmdip(1:3),angip(1:5)       ! sampling wavelengths and angles
data (lmdip(k),k=1,3) / 2.5_dp, 3.07_dp, 3.5_dp /
data (angip(k),k=1,5) / 30.0_dp, 60.0_dp, 90.0_dp, 120.0_dp, 150.0_dp /
end module const

!--------------------------------------------------------------------------------
!
!     average optical properties of spheres with power-law distribution
!
!--------------------------------------------------------------------------------
subroutine mie_mrn_opacities(amax,amin,p,nwl,fn,fnice,allwavel,pangout,LP)
use types; use const
use omp_lib
implicit none
!--------------------------------------------------------------------------------
real(kind=dp) :: amax                   ! maximum grain radius (micron)
real(kind=dp) :: amin                   ! minimum grain radius (micron)
real(kind=dp) :: p                      ! power law index of size distribution
real(kind=dp) :: rhod                   ! material density (g/cm^3)
integer       :: nwl                    ! number of wavelengths
character     :: fn*50                  ! output file name
character     :: fnice*8                ! file name of optical constant
!--------------------------------------------------------------------------------
integer       :: i,j,ia,kk
real(kind=dp) :: lnamin,lnamax,dlna,norm,norm4g
real(kind=dp) :: x,ext,sca,back,gs,dang
real(kind=dp) :: wn,S11,S12,S33,S34,ang(2*nang-1)
real(kind=dp) :: kabsip,kscaip,gip,LP(1:3,1:5)
real(kind=dp) :: Z11ip2,Z12ip2
complex(kind=dp)::refrel
complex(kind=dp),dimension(2*nang-1)::S1,S2
real(kind=dp),dimension(nwl)           :: wl,refre,refim,kabsave,kscaave,gsave
real(kind=dp),dimension(namax)         :: aa,ma,mass
real(kind=dp),dimension(namax,nwl)     :: kabs,ksca,g
real(kind=dp),dimension(nwl,2*nang-1)  :: Z11ave,Z12ave,Z22ave,Z33ave,Z34ave,Z44ave
real(kind=dp),dimension(namax,2*nang-1):: Z11,Z12,Z22,Z33,Z34,Z44
real(kind=dp),allocatable,dimension(:) :: Z11ip,Z12ip,Z22ip,Z33ip,Z34ip,Z44ip,POL
logical::allwavel,pangout

!--------------------------------------------------------------------------------
!       some safety checks...
!--------------------------------------------------------------------------------
if(amax .lt. amin) then 
    print *, 'error : amax < amin' 
    print *, 'stop'
    stop
endif
if(p .lt. 0.0_dp) then
    print *, 'warning : q is supposed to be positive.'
endif
if(pangout .and. nwl .ne. 3) then
    print *, 'pangout=.true. is only valid when nwl=3'
    print *, 'stop'
    stop
endif

!--------------------------------------------------------------------------------
!       Load optical constant
!--------------------------------------------------------------------------------
if(allwavel) then
    open(50,file="IceSilicate_Iceband_"//trim(fnice)//".lnk",status="old")
else
    open(50,file="IceSilicate_3bands_"//trim(fnice)//".lnk",status="old")
endif
do i=1,11 
    read(50,'()') !skip header
end do
read(50,*) rhod
do i=1,6
    read(50,'()') !skip header
end do
do i=1,nwl
    read(50,*) wl(i),refre(i),refim(i)
end do
close(50)

!--------------------------------------------------------------------------------
!       make size distribution bin
!--------------------------------------------------------------------------------
lnamin = log(amin)
lnamax = log(amax)
dlna   = log(amax/amin) / real(namax-1,kind=dp)
do ia=1,namax
    aa(ia)   = exp(lnamin+real(ia-1,kind=dp)*dlna)
    ma(ia)   = 3.0_dp/(4.0_dp*rhod*(aa(ia)*mic2cm))
    mass(ia) = 4.0_dp*pi*(aa(ia)*mic2cm)**3.0_dp*rhod/3.0_dp
end do

!--------------------------------------------------------------------------------
!       make angle mesh
!--------------------------------------------------------------------------------
dang = halfpi/real(nang-1,kind=dp)
do j=1,2*nang-1
    ang(j)=dang*real(j-1,kind=dp)*r2d
end do

!--------------------------------------------------------------------------------
!       perform distribution average of optical properties
!--------------------------------------------------------------------------------
Z11ave  = 0.0_dp
Z12ave  = 0.0_dp 
Z22ave  = 0.0_dp 
Z33ave  = 0.0_dp 
Z34ave  = 0.0_dp 
Z44ave  = 0.0_dp 
kabsave = 0.0_dp
kscaave = 0.0_dp
gsave   = 0.0_dp
norm    = 0.0_dp
do ia=1,namax-1 
    norm = norm + 0.5_dp*(aa(ia)**(4.0_dp-p) + aa(ia+1)**(4.0_dp-p))
end do

do i=1,nwl
    Z11 = 0.0_dp
    Z12 = 0.0_dp
    Z22 = 0.0_dp
    Z33 = 0.0_dp
    Z34 = 0.0_dp
    Z44 = 0.0_dp
    refrel  = cmplx(refre(i),refim(i))
    wn      = twopi/(wl(i)*mic2cm)

    !$OMP parallel do schedule (static,1)                     &
    !$OMP default(none)                                       &
    !$OMP shared(refrel,wn,aa,wl,ma,mass,i)                   &
    !$OMP shared(kabs,ksca,g,Z11,Z12,Z22,Z33,Z34,Z44)         &
    !$OMP private(x,ext,sca,back,gs,S1,S2,S11,S12,S33,S34)      
    do ia=1,namax
         x = twopi*aa(ia)/wl(i)
         call bhmie_rt(x,refrel,nang,S1,S2,ext,sca,back,gs)
         kabs(ia,i) = ma(ia)*(ext-sca)
         ksca(ia,i) = ma(ia)*sca
         g(ia,i)    = gs
         do j=1,2*nang-1
         S11  =       0.5_dp*abs(S2(j))*abs(S2(j))
         S11  = S11 + 0.5_dp*abs(S1(j))*abs(S1(j))
         S12  =       0.5_dp*abs(S2(j))*abs(S2(j))
         S12  = S12 - 0.5_dp*abs(S1(j))*abs(S1(j))       
         S33  = dble(S1(j)*conjg(S2(j)))
         S34  = aimag(S2(j)*conjg(S1(j)))
         Z11(ia,j) = S11 / mass(ia) / wn / wn 
         Z12(ia,j) = S12 / mass(ia) / wn / wn 
         Z22(ia,j) = Z11(ia,j)
         Z33(ia,j) = S33 / mass(ia) / wn / wn 
         Z34(ia,j) = S34 / mass(ia) / wn / wn 
         Z44(ia,j) = Z33(ia,j)
         end do
    end do
    !$OMP end parallel do
    
    do j=1,2*nang-1
         do ia=1,namax-1 
         Z11ave(i,j) = Z11ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z11(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z11(ia+1,j))
         Z12ave(i,j) = Z12ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z12(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z12(ia+1,j))
         Z22ave(i,j) = Z22ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z22(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z22(ia+1,j))
         Z33ave(i,j) = Z33ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z33(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z33(ia+1,j))
         Z34ave(i,j) = Z34ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z34(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z34(ia+1,j))
         Z44ave(i,j) = Z44ave(i,j) + 0.5_dp*(aa(ia)**(4.0_dp-p)*Z44(ia,j)&
                         & + aa(ia+1)**(4.0_dp-p)*Z44(ia+1,j))
         end do
   end do

   norm4g = 0.0_dp
   do ia=1,namax-1 
         kabsave(i) = kabsave(i) + 0.5_dp*(aa(ia)**(4.0_dp-p)*kabs(ia,i)&
                         & + aa(ia+1)**(4.0_dp-p)*kabs(ia+1,i))
         kscaave(i) = kscaave(i) + 0.5_dp*(aa(ia)**(4.0_dp-p)*ksca(ia,i)&
                         & + aa(ia+1)**(4.0_dp-p)*ksca(ia+1,i))
         gsave(i) = gsave(i) + 0.5_dp*(aa(ia)**(4.0_dp-p)*ksca(ia,i)*g(ia,i)&
                         & + aa(ia+1)**(4.0_dp-p)*ksca(ia+1,i)*g(ia+1,i))
         norm4g = norm4g + 0.5_dp*(aa(ia)**(4.0_dp-p)*ksca(ia,i)&
                         & + aa(ia+1)**(4.0_dp-p)*ksca(ia+1,i))
    end do
    gsave(i) = gsave(i) / norm4g 
end do
Z11ave  = Z11ave  / norm
Z12ave  = Z12ave  / norm
Z22ave  = Z22ave  / norm
Z33ave  = Z33ave  / norm
Z34ave  = Z34ave  / norm
Z44ave  = Z44ave  / norm
kabsave = kabsave / norm
kscaave = kscaave / norm

!--------------------------------------------------------------------------------
!       Output 1 : fixed scattering angle ( P vs wavelength )
!--------------------------------------------------------------------------------
if(allwavel) then
write(*,*) "Writing...:","pwavel_"//TRIM(fn)//".dat"
open(20,file="./data/pwavel_"//TRIM(fn)//".dat",status="unknown")
allocate(Z11ip(1:nwl),Z12ip(1:nwl),Z22ip(1:nwl),Z33ip(1:nwl),&
        Z34ip(1:nwl),Z44ip(1:nwl),POL(1:nwl))
do k=1,5
    ! Interpolation
    do j=1,2*nang-2
        if(ang(j) .le. angip(k) .and. ang(j+1) .gt. angip(k)) then
            do i=1,nwl
            Z11ip(i) = (Z11ave(i,j+1)-Z11ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z11ave(i,j)
            Z12ip(i) = (Z12ave(i,j+1)-Z12ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z12ave(i,j)
            Z22ip(i) = (Z22ave(i,j+1)-Z22ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z22ave(i,j)
            Z33ip(i) = (Z33ave(i,j+1)-Z33ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z33ave(i,j)
            Z34ip(i) = (Z34ave(i,j+1)-Z34ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z34ave(i,j)
            Z44ip(i) = (Z44ave(i,j+1)-Z44ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z44ave(i,j)
            POL(i)   = -Z12ip(i)/Z11ip(i)
            enddo
        endif
    enddo
    
    ! interpolation at wavelength
    do kk=1,3 
    do i=1,nwl-1
        if(wl(i) .le. lmdip(kk) .and. wl(i+1) .gt. lmdip(kk)) then
             Z11ip2 = (Z11ip(i+1)-Z11ip(i))/(wl(i+1)-wl(i))*(lmdip(kk)-wl(i))+Z11ip(i)
             Z12ip2 = (Z12ip(i+1)-Z12ip(i))/(wl(i+1)-wl(i))*(lmdip(kk)-wl(i))+Z12ip(i)
             exit
        endif
    enddo
    LP(kk,k) = -Z12ip2 / Z11ip2
    enddo
    write(20,2000) angip(k)," = scattering angle (deg)"
    write(20,2000) amax," = maximum grain radius (um)"
    write(20,2000) amin," = minimum grain radius (um)"
    write(20,2000) p," = power law index q"
    write(20,2200) trim(fnice)," = ice-silicate model"
    write(20,2100) namax," = number of grain radius mesh"
    write(20,3000) "lambda (um)","kabs (cm2/g)","ksca (cm2/g)","<cos>",&
            "Z11(cm2/g/str)","Z12(cm2/g/str)","-Z12/Z11"
    do i=1,nwl
        write(20,1000) wl(i),kabsave(i),kscaave(i),gsave(i),Z11ip(i),Z12ip(i),POL(i)
    enddo
    write(20,*)
enddo
close(20)
deallocate(Z11ip,Z12ip,Z22ip,Z33ip,Z34ip,Z44ip,POL)

else    ! allwavel = .false.
!--------------------------------------------------------------------------------
!       Output 3 : fixed wavelength  (P vs theta)
!--------------------------------------------------------------------------------
do i=1,nwl
    do k=1,5
        do j=1,2*nang-2
        if(ang(j) .le. angip(k) .and. ang(j+1) .gt. angip(k)) then
            Z11ip2 = (Z11ave(i,j+1)-Z11ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z11ave(i,j)
            Z12ip2 = (Z12ave(i,j+1)-Z12ave(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z12ave(i,j)
        endif
        enddo
    LP(i,k) = -Z12ip2 / Z11ip2
    enddo
enddo
endif

if(pangout) then
write(*,*) "Writing...:","pangle_"//TRIM(FN)//".dat"
do i=1,nwl
open(20,file="./data/pangle_"//TRIM(FN)//".dat",status="unknown")
write(20,2000) amax," = maximum grain radius (um)"
write(20,2000) amin," = minimum grain radius (um)"
write(20,2000) p," = power law index q"
write(20,2200) trim(fnice)," = ice-silicate model"
write(20,2100) namax," = number of grain radius mesh"
write(20,2000) wl(i)," = wavelength (micron)"
write(20,2000) kabsave(i)," = absorption opacity (cm^2/g)"
write(20,2000) kscaave(i)," = scattering opacity (cm^2/g)"
write(20,2000) gsave(i)," = <cos> : asymmetry parameter"
write(20,3000) "angle (deg)","Z11(cm2/g/str)","Z12(cm2/g/str)","Z22(cm2/g/str)",&
    "Z33(cm2/g/str)","Z34(cm2/g/str)","Z44(cm2/g/str)"
do j=1,2*nang-1
write(20,1000) ang(j),Z11ave(i,j),Z12ave(i,j),Z22ave(i,j),&
        Z33ave(i,j),Z34ave(i,j),Z44ave(i,j)
enddo
write(20,1000)
enddo
close(20)
endif


1000 format(' ',1P10E15.5)
2000 format('#',1PE15.5,A)
2100 format('#',I15,A)
2200 format('#',A15,A)
3000 format('#',7A15)

return
end subroutine mie_mrn_opacities

!--------------------------------------------------------------------------------
!
!     compute optical properties of single size grains
!
!--------------------------------------------------------------------------------
subroutine mie_single_opacities(a,nwl,fn,fnice,allwavel,pangout,LP)
use types; use const
use omp_lib
implicit none
!--------------------------------------------------------------------------------
real(kind=dp) :: a                      ! grain radius (micron)
real(kind=dp) :: rhod                   ! material density (g/cm^3)
integer       :: nwl                    ! number of wavelengths
character     :: fn*50                  ! output file name
character     :: fnice*8                ! file name of optical constant
!--------------------------------------------------------------------------------
integer       :: i,j,kk
real(kind=dp) :: x,ext,sca,back,gs,dang
real(kind=dp) :: wn,S11,S12,S33,S34,ang(2*nang-1)
real(kind=dp) :: kabsip,kscaip,gip,LP(1:3,1:5)
real(kind=dp) :: ma,mass,Z11ip2,Z12ip2
complex(kind=dp)::refrel
complex(kind=dp),dimension(2*nang-1)::S1,S2
real(kind=dp),dimension(nwl)           :: wl,refre,refim,kabs,ksca,g
real(kind=dp),dimension(nwl,2*nang-1)  :: Z11,Z12,Z22,Z33,Z34,Z44
real(kind=dp),allocatable,dimension(:) :: Z11ip,Z12ip,Z22ip,Z33ip,Z34ip,Z44ip,POL
logical::allwavel,pangout

if(pangout .and. nwl .ne. 3) then
        print *, 'pangout=.true. is only valid when nwl=3'
        print *, 'stop'
        stop
endif

!--------------------------------------------------------------------------------
!       Load optical constant
!--------------------------------------------------------------------------------
if(allwavel) then
        open(50,file="IceSilicate_Iceband_"//trim(fnice)//".lnk",status="old")
else
        open(50,file="IceSilicate_3bands_"//trim(fnice)//".lnk",status="old")
endif
do i=1,11 
        read(50,'()') !skip header
end do
read(50,*) rhod
do i=1,6
        read(50,'()') !skip header
end do
do i=1,nwl
        read(50,*) wl(i),refre(i),refim(i)
end do
close(50)

ma   = 3.0_dp/(4.0_dp*rhod*a*mic2cm)
mass = 4.0_dp*pi*(a*mic2cm)**3.0_dp*rhod/3.0_dp

!--------------------------------------------------------------------------------
!       make angle mesh
!--------------------------------------------------------------------------------
dang = halfpi/real(nang-1,kind=dp)
do j=1,2*nang-1
        ang(j)=dang*real(j-1,kind=dp)*r2d
end do

!--------------------------------------------------------------------------------
!       compute optical properties
!--------------------------------------------------------------------------------
Z11 = 0.0_dp
Z12 = 0.0_dp
Z22 = 0.0_dp
Z33 = 0.0_dp
Z34 = 0.0_dp
Z44 = 0.0_dp

!$OMP parallel do schedule (static,1)                     &
!$OMP default(none)                                       &
!$OMP shared(a,wl,refre,refim,ma,mass,nwl)                &
!$OMP shared(kabs,ksca,g,Z11,Z12,Z22,Z33,Z34,Z44)         &
!$OMP private(x,wn,refrel,ext,sca,back,gs,S1,S2,S11,S12,S33,S34)      
do i=1,nwl
    refrel  = cmplx(refre(i),refim(i))
    wn      = twopi/(wl(i)*mic2cm)
    x       = twopi*a/wl(i)
    call bhmie_rt(x,refrel,nang,S1,S2,ext,sca,back,gs)
    kabs(i) = ma*(ext-sca)
    ksca(i) = ma*sca
    g(i)    = gs
    do j=1,2*nang-1
    S11  =       0.5_dp*abs(S2(j))*abs(S2(j))
    S11  = S11 + 0.5_dp*abs(S1(j))*abs(S1(j))
    S12  =       0.5_dp*abs(S2(j))*abs(S2(j))
    S12  = S12 - 0.5_dp*abs(S1(j))*abs(S1(j))       
    S33  = dble(S1(j)*conjg(S2(j)))
    S34  = aimag(S2(j)*conjg(S1(j)))
    Z11(i,j) = S11 / mass / wn / wn 
    Z12(i,j) = S12 / mass / wn / wn 
    Z22(i,j) = Z11(i,j)
    Z33(i,j) = S33 / mass / wn / wn 
    Z34(i,j) = S34 / mass / wn / wn 
    Z44(i,j) = Z33(i,j)
    end do
end do
!$OMP end parallel do

!--------------------------------------------------------------------------------
!       Output 1 : fixed scattering angle ( P vs wavelength )
!--------------------------------------------------------------------------------
if(allwavel) then
write(*,*) "Writing...:","pwavel_"//TRIM(fn)//".dat"
open(20,file="./data/pwavel_"//TRIM(fn)//".dat",status="unknown")
allocate(Z11ip(1:nwl),Z12ip(1:nwl),Z22ip(1:nwl),Z33ip(1:nwl),&
        Z34ip(1:nwl),Z44ip(1:nwl),POL(1:nwl))
do k=1,5
    ! Interpolation
    do j=1,2*nang-2
        if(ang(j) .le. angip(k) .and. ang(j+1) .gt. angip(k)) then
            do i=1,nwl
            Z11ip(i) = (Z11(i,j+1)-Z11(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z11(i,j)
            Z12ip(i) = (Z12(i,j+1)-Z12(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z12(i,j)
            Z22ip(i) = (Z22(i,j+1)-Z22(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z22(i,j)
            Z33ip(i) = (Z33(i,j+1)-Z33(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z33(i,j)
            Z34ip(i) = (Z34(i,j+1)-Z34(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z34(i,j)
            Z44ip(i) = (Z44(i,j+1)-Z44(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z44(i,j)
            POL(i)   = -Z12ip(i)/Z11ip(i)
            enddo
            exit
        endif
    enddo
    ! interpolation at wavelength
    do kk=1,3 
    do i=1,nwl-1
        if(wl(i) .le. lmdip(kk) .and. wl(i+1) .gt. lmdip(kk)) then
             Z11ip2 = (Z11ip(i+1)-Z11ip(i))/(wl(i+1)-wl(i))*(lmdip(kk)-wl(i))+Z11ip(i)
             Z12ip2 = (Z12ip(i+1)-Z12ip(i))/(wl(i+1)-wl(i))*(lmdip(kk)-wl(i))+Z12ip(i)
        endif
    enddo
    LP(kk,k) = -Z12ip2 / Z11ip2
    enddo
    write(20,2000) angip(k)," = scattering angle (deg)"
    write(20,2200) trim(fnice)," = ice-silicate model"
    write(20,3000) "lambda (um)","kabs (cm2/g)","ksca (cm2/g)","<cos>",&
            "Z11(cm2/g/str)","Z12(cm2/g/str)","-Z12/Z11"
    do i=1,nwl
        write(20,1000) wl(i),kabs(i),ksca(i),g(i),Z11ip(i),Z12ip(i),POL(i)
    enddo
    write(20,*)
enddo
close(20)
deallocate(Z11ip,Z12ip,Z22ip,Z33ip,Z34ip,Z44ip,POL)
else
    do i=1,nwl
        do k=1,5
            do j=1,2*nang-2
            if(ang(j) .le. angip(k) .and. ang(j+1) .gt. angip(k)) then
            Z11ip2 = (Z11(i,j+1)-Z11(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z11(i,j)
            Z12ip2 = (Z12(i,j+1)-Z12(i,j))/(ang(j+1)-ang(j))*(angip(k)-ang(j))+Z12(i,j)
            endif
            enddo
        LP(i,k) = - Z12ip2 / Z11ip2
        enddo
    enddo
endif

!--------------------------------------------------------------------------------
!       Output 3 : fixed wavelength  (P vs theta)
!--------------------------------------------------------------------------------
if(pangout) then
write(*,*) "Writing...:","pangle_"//TRIM(FN)//".dat"
open(20,file="./data/pangle_"//TRIM(FN)//".dat",status="unknown")
write(20,2200) trim(fnice)," = ice-silicate model"
write(20,2000) a," = grain radius (micron)"
write(20,2000) wl(i)," = wavelength (micron)"
write(20,2000) kabs(i)," = absorption opacity (cm^2/g)"
write(20,2000) ksca(i)," = scattering opacity (cm^2/g)"
write(20,2000) g(i)," = <cos> : asymmetry parameter"
write(20,3000) "angle (deg)","Z11(cm2/g/str)","Z12(cm2/g/str)","Z22(cm2/g/str)",&
    "Z33(cm2/g/str)","Z34(cm2/g/str)","Z44(cm2/g/str)"
do j=1,2*nang-1
write(20,1000) ang(j),Z11(i,j),Z12(i,j),Z22(i,j),Z33(i,j),Z34(i,j),Z44(i,j)
enddo
endif

1000 format(' ',1P10E15.5)
2000 format('#',1PE15.5,A)
2100 format('#',I15,A)
2200 format('#',A15,A)
3000 format('#',7A15)

return
end subroutine mie_single_opacities
