!--------------------------------------------------------------------------------
!
!       BRUGGEMAN MIXING RULE
!
!--------------------------------------------------------------------------------
!       Some type and constants
!--------------------------------------------------------------------------------
module types
implicit none
integer, parameter      :: dp = selected_real_kind(P=15)
end module types
module const
use types
implicit none
integer,      parameter :: itermax    = 100       ! maximum iteration in newton raphson
integer,      parameter :: nfilemax   = 5000      ! max number of lines of loaded files
integer,      parameter :: nwl        = 250       ! number of wavelength grids
integer,      parameter :: nspec      = 4         ! number of dust species
real(kind=dp),parameter :: wlmin      = 2.00_dp   ! minimum wavelength 
real(kind=dp),parameter :: wlmax      = 4.25_dp   ! maximum wavelength
complex(kind=dp),parameter :: vac     = cmplx(1.0_dp,0.0_dp) ! vacuum 
!--------------------------------------------------------------------------------
!       Mass abundances in Pollack et al. (1994)
!--------------------------------------------------------------------------------
real(kind=dp),parameter :: zeta_olivine_P94       = 2.64e-3_dp  ! olivine
real(kind=dp),parameter :: zeta_orthopyroxene_P94 = 7.70e-4_dp  ! orthopyroxene
real(kind=dp),parameter :: zeta_reforg_P94        = 3.53e-3_dp  ! ref. organics
real(kind=dp),parameter :: zeta_volorg_P94        = 6.02e-4_dp  ! vol. organics
real(kind=dp),parameter :: zetaice_P94            = 5.55e-3_dp  ! ice
real(kind=dp),parameter :: zetatro_P94            = 7.68e-4_dp  ! troilite
!--------------------------------------------------------------------------------
!       Mass abundances in D'Alessio (2001)
!--------------------------------------------------------------------------------
real(kind=dp),parameter :: zeta_sil_D01           = 3.4e-3_dp   ! silicate
real(kind=dp),parameter :: zeta_ice_D01           = 5.6e-3_dp   ! ice
real(kind=dp),parameter :: zeta_tro_D01           = 7.68e-4_dp  ! troilite
real(kind=dp),parameter :: zeta_org_D01           = 4.1e-3_dp   ! organics
end module const

!--------------------------------------------------------------------------------
!       define structure
!--------------------------------------------------------------------------------
module defstruct
use types; use const
implicit none
type inclusions
integer::nlam
real(kind=dp)   ,dimension(nfilemax) :: wavel,eps1,eps2
complex(kind=dp),dimension(nfilemax) :: eps
end type inclusions
end module defstruct

!--------------------------------------------------------------------------------
!       main program
!--------------------------------------------------------------------------------
program mixture
use types; use defstruct; use const
implicit none
type(inclusions) sil,ice,organics,tro,mix 
integer::ios,ispec,iter,i,iwl
real(kind=dp),dimension(2,2)::JI,J
real(kind=dp),dimension(nspec)::F
real(kind=dp)::wl,dwl,det,g,h,cmre,cmim,ea1,ea2
real(kind=dp)::rhosil,rhoorg,rhoice,rhotro,rhod
real(kind=dp)::zetasil,zetaorg,zetaice,zetatro,zeta_all,xi_ice
real(kind=dp)::fnorm,fsil,forg,fice,ftro,norm,cdum
complex(kind=dp)::silip,iceip,orgip,troip
complex(kind=dp),dimension(nspec)::EPSI
complex(kind=dp)::EPSAVE
character::fn*50

!--------------------------------------------------------------------------------
!       output file name
!--------------------------------------------------------------------------------

fn      =       "IceSilicate_Iceband_fice1.0.lnk"

!--------------------------------------------------------------------------------
!
!       load optical constant data
!
!--------------------------------------------------------------------------------
! 1. astronomical silicate
!--------------------------------------------------------------------------------
ios = 0
open(10,file="./refdata/callindex.out_silD03",status="unknown")
do i=1,5 
        read(10,'()') ! skip header
end do
do i=1,nfilemax
        read(10,*,iostat=ios) sil%wavel(i),sil%eps1(i),sil%eps2(i),cdum,cdum
        if(ios .ne. 0) then
        sil%nlam=i-1 
        exit
        end if
end do
close(10)
sil%eps1 = sil%eps1+1.0_dp  ! This line is necessary for callindex.out_silD03 
sil%eps  = cmplx(sil%eps1,sil%eps2)
call order(sil)
!--------------------------------------------------------------------------------
! 2. water ice
!--------------------------------------------------------------------------------
ios = 0
open(20,file="./refdata/Warren08_waterice.lnk",status="unknown")
do i=1,nfilemax
        read(20,*,iostat=ios) ice%wavel(i),ice%eps1(i),ice%eps2(i)
        if(ios .ne. 0) then
        ice%nlam=i-1 
        exit
        end if
end do
close(20)
ice%eps = cmplx(ice%eps1,ice%eps2)
! converting refractive index into dielectric function
do iwl=1,ice%nlam
        ice%eps(iwl) = ice%eps(iwl) * ice%eps(iwl)
end do
ice%eps1 = real(ice%eps)
ice%eps2 = aimag(ice%eps)
call order(ice)
!--------------------------------------------------------------------------------
! 3. organics
!--------------------------------------------------------------------------------
ios = 0
open(30,file="./refdata/organicsk.lnk",status="unknown")
do i=1,nfilemax
        read(30,*,iostat=ios) organics%wavel(i),organics%eps1(i),&
        organics%eps2(i)
        if(ios .ne. 0) then
        organics%nlam=i-1 
        exit
        end if
end do
close(30)
! converting refractive index into dielectric function
organics%eps=cmplx(organics%eps1,organics%eps2)
do iwl=1,organics%nlam
        organics%eps(iwl) = organics%eps(iwl) * organics%eps(iwl)
end do
organics%eps1 = real(organics%eps)
organics%eps2 = aimag(organics%eps)
call order(organics)
!--------------------------------------------------------------------------------
! 4. troilite
!--------------------------------------------------------------------------------
ios = 0
open(40,file="./refdata/troilitek.lnk",status="unknown")
do i=1,nfilemax
        read(40,*,iostat=ios) tro%wavel(i),tro%eps1(i),tro%eps2(i)
        if(ios .ne. 0) then
        tro%nlam=i-1 
        exit
        end if
end do
close(40)
! converting refractive index into dielectric function
tro%eps=cmplx(tro%eps1,tro%eps2)
do iwl=1,sil%nlam
 tro%eps(iwl) = tro%eps(iwl) * tro%eps(iwl)
end do
tro%eps1 = real(tro%eps)
tro%eps2 = aimag(tro%eps)
call order(tro)

!--------------------------------------------------------------------------------
!       set mass abundance of each dust component 
!--------------------------------------------------------------------------------
! 1. D'Alessio et al. (2001) ( nspec = 4 ) 
!--------------------------------------------------------------------------------
! (a) Reconstructed from Pollack et al. (1994)
!--------------------------------------------------------------------------------
!zeta_all     =  zeta_olivine_P94 + zeta_orthopyroxene_P94 + &
!                 zeta_reforg_P94 + zeta_volorg_P94 + &
!                 zetaice_P94 + zetatro_P94
!zetasil      = (zeta_olivine_P94 + zeta_orthopyroxene_P94) / zeta_all
!zetaorg      = (zeta_reforg_P94 + zeta_volorg_P94) / zeta_all
!zetaice      = zetaice_P94 /zeta_all
!zetatro      = zetatro_P94 /zeta_all
!--------------------------------------------------------------------------------
! (b) Just use the values in D'Alessio et al. 2001 (truncated at few digits)
!--------------------------------------------------------------------------------
!zeta_all     = zeta_sil_D01+zeta_org_D01+zeta_tro_D01+zeta_ice_D01
!zetasil      = zeta_sil_D01/zeta_all
!zetaorg      = zeta_org_D01/zeta_all
!zetaice      = zeta_ice_D01/zeta_all
!zetatro      = zeta_tro_D01/zeta_all
!--------------------------------------------------------------------------------
!rhosil       = 3.30_dp
!rhoorg       = 1.50_dp
!rhoice       = 0.92_dp
!rhotro       = 4.83_dp
!
!--------------------------------------------------------------------------------
! 2. Birnstiel et al. (2018) (DSHARP) ( nspec = 4 )
! Mass fraction of water ice is set as 20%, and the remaining 80% is occupied by
! silicate, organics, and troilite based on the recipe of D'Alessio et al. (2001).
!--------------------------------------------------------------------------------
!zeta_all     = zeta_sil_D01 + zeta_org_D01 + zeta_tro_D01 ! "All" except for ice.
!zetaice      = 0.2_dp 
!zetasil      = (1.0_dp - zetaice) * zeta_sil_D01/zeta_all
!zetaorg      = (1.0_dp - zetaice) * zeta_org_D01/zeta_all
!zetatro      = (1.0_dp - zetaice) * zeta_tro_D01/zeta_all
!rhosil       = 3.30_dp
!rhoorg       = 1.50_dp
!rhoice       = 0.92_dp
!rhotro       = 4.83_dp
!
!--------------------------------------------------------------------------------
! 3. Tazaki et al. (2021) ( nspec = 4, but no carbon, no troilite )
! Silicate and ice mixture with parameterized Pollack abundance.
! xi_ice : ice abundance parameter [ corresponds to "fice" in the paper]
!--------------------------------------------------------------------------------
xi_ice        = 1.0_dp        ! fice parameter in Tazaki et al. (2021)
zetaorg       = 0.0_dp        ! just put 0 because no carbon
zetatro       = 0.0_dp        ! just put 0 because no troilite
zeta_all      = zetaice_P94 * xi_ice + zeta_olivine_P94 + zetaorg + zetatro
zetaice       = zetaice_P94 * xi_ice / zeta_all
zetasil       = zeta_olivine_P94     / zeta_all
rhosil        = 3.5_dp
rhoice        = 0.92_dp
rhoorg        = 1.5_dp
rhotro        = 4.83_dp

!--------------------------------------------------------------------------------
!       compute volume filling factor of each component
!--------------------------------------------------------------------------------
fnorm   =  zetasil*rhosil**(-1.0_dp)+zetaorg*rhoorg**(-1.0_dp)+&
        zetaice*rhoice**(-1.0_dp)+zetatro*rhotro**(-1.0_dp)
fsil    =  zetasil*rhosil**(-1.0_dp) / fnorm
forg    =  zetaorg*rhoorg**(-1.0_dp) / fnorm 
fice    =  zetaice*rhoice**(-1.0_dp) / fnorm
ftro    =  zetatro*rhotro**(-1.0_dp) / fnorm

! check normalization
!if(fsil+forg+fice+ftro .ne. 1.0_dp) then
!        print *, 'Error'
!        print *, 'Normalization check:',fsil+forg+fice+ftro
!        print *, "filling Factor is not normalized to unity!"
!        stop
!endif

!--------------------------------------------------------------------------------
!       compute mean dust density
!--------------------------------------------------------------------------------
rhod=fsil*rhosil+forg*rhoorg+fice*rhoice+ftro*rhotro

!--------------------------------------------------------------------------------
!               BRUGGEMAN MIXING RULE
!               VIA NEWTON-RAPHSON METHOD
!--------------------------------------------------------------------------------
F(1) = fsil ! silicate
F(2) = fice ! water ice
F(3) = forg ! organics
F(4) = ftro ! troilite

dwl=(wlmax/wlmin)**(1.0/dble(nwl-1))
mix%nlam=nwl
do iwl=1,nwl
    wl=wlmin*dwl**dble(iwl-1)

    !Interpolate
    call interpolate(wl,sil,silip)
    EPSI(1) = silip

    call interpolate(wl,ice,iceip)
    EPSI(2) = iceip

    call interpolate(wl,organics,orgip)
    EPSI(3) = orgip

    call interpolate(wl,tro,troip)
    EPSI(4) = troip

    ! set initial guess for newton-raphson
    EPSAVE = EPSI(2)
    !EPSAVE = vac
    !
    ! Begin itegration for Bruggeman-Mixing rule
    DO ITER=1,ITERMAX
        G  =  0.0_dp
        H  =  0.0_dp
        J  =  0.0_dp
        JI =  0.0_dp
        DO ISPEC=1,NSPEC
          CALL JACOBIAN(EPSI(ISPEC),EPSAVE,F(ISPEC),JI)
          J  =  J  +  JI
          G  =  G  +  CMRE(EPSI(ispec),EPSAVE,F(ISPEC))
          H  =  H  +  CMIM(EPSI(ispec),EPSAVE,F(ISPEC))
        END DO

          DET = J(1,1)*J(2,2)-J(1,2)*J(2,1)
          EA1 = REAL(EPSAVE)-(1.0_dp/DET)*(J(2,2)*G-J(1,2)*H)
          EA2 = AIMAG(EPSAVE)-(1.0_dp/DET)*(-J(2,1)*G+J(1,1)*H)
 
        IF(ABS(REAL(EA1)-REAL(EPSAVE)) .LT. 1.0e-5_dp &
                .AND. ABS(REAL(EA2)-AIMAG(EPSAVE)) .LT. 1.0e-5_dp ) EXIT
        IF(ITER .EQ. ITERMAX) THEN
                write(*,*) "Error!"
                write(*,*) "|------------------------------|"
                write(*,*) "|  ITERATION IS NOT CONVERGED. |"
                write(*,*) "|  CALCULATION IS ABORTED!     |"
                write(*,*) "|------------------------------|"
                STOP
        END IF
        EPSAVE = CMPLX(EA1,EA2)
     END DO

 mix%wavel(iwl)  =  wl
 mix%eps1(iwl)   =  EA1
 mix%eps2(iwl)   =  EA2
 mix%eps(iwl)    =  CMPLX(EA1,EA2)

 !---------------------
 ! CHECK SOLUTION
 ! 
 ! Does the average dielectric function satisfy the basic equation of
 ! Bruggeman mixing rule?
 ! Basic equation:
 !
 ! \Sum_{j=1}^{N} f_j * (eps_j - eps_m)/(eps_j + 2.0 * eps_m) = 0
 ! 
 ! where, eps_j and eps_m are the dielectric function of 
 ! j-th species and matrix, respectively.
 ! 
 !-----------------------
 norm=cmplx(0.0_dp,0.0_dp)
 do ispec=1,nspec
         norm = norm + f(ispec) * (epsi(ispec)-mix%eps(iwl))/(epsi(ispec)+2.0*mix%eps(iwl))
 enddo
 if(abs(norm) .ge. 1.0e-5_dp) then
         print *, 'Obtained solution does not satisfy the original equation!'
         print *, 'Something strange. Stop calculation.'
         stop
 endif
end do

!--------------------------------------------------------------------------------
!                       OUTPUT
!--------------------------------------------------------------------------------
open(10,file=trim(fn),status="unknown")
write(10,2000) mix%nlam," = number of wavelength"
write(10,2000) nspec," = number of species"
write(10,2100) "Mass fraction and material density:"
write(10,2200) zetasil*1.e2," = mass fraction of silicate [%]" 
write(10,2200) zetaorg*1.e2," = mass fraction of amo. Carbon [%]" 
write(10,2200) zetaice*1.e2," = mass fraction of water ice [%]" 
write(10,2200) zetatro*1.e2," = mass fraction of troilite [%]" 
write(10,2200) rhosil," = mass density of silicate (g/cc)" 
write(10,2200) rhoorg," = mass density of organics (g/cc)" 
write(10,2200) rhoice," = mass density of water ice (g/cc)" 
write(10,2200) rhotro," = mass density of troilite (g/cc)" 
write(10,2200) rhod," = mean mass density of dust (g/cc)" 
write(10,2100) "Volume fraction:"
write(10,2200) fsil*1.e2," = vol. fraction of silicate [%]" 
write(10,2200) forg*1.e2," = vol. fraction of organics [%]" 
write(10,2200) fice*1.e2," = vol. fraction of water ice [%]" 
write(10,2200) ftro*1.e2," = vol. fraction of troilite [%]" 
write(10,3000) "lambda [um]","Re(m)","Im(m)"
do iwl=1,mix%nlam
write(10,1000) mix%wavel(iwl),real(sqrt(mix%eps(iwl))),aimag(sqrt(mix%eps(iwl)))
end do
close(10)
write(*,*) trim(fn)//" was created!"

1000 format(1P3E15.5)
2000 format('#',I5,A)
2100 format('# ',A)
2200 format(1PE15.5,A)
3000 format('#',3A15)

stop
end program mixture

!--------------------------------------------------------------------------------
!       check the wavelength order of the array
!--------------------------------------------------------------------------------
subroutine order(mat)
use defstruct
implicit none
type(inclusions) mat,update
integer::i
update = mat
if(mat%wavel(1) .lt. mat%wavel(mat%nlam)) then
        do i=mat%nlam,1,-1       
        update%wavel(mat%nlam-i+1) = mat%wavel(i)
        update%eps1(mat%nlam-i+1)  = mat%eps1(i)
        update%eps2(mat%nlam-i+1)  = mat%eps2(i)
        update%eps(mat%nlam-i+1)   = mat%eps(i)
        end do
end if
mat = update
return
end subroutine order

!---------------------------
function cmre(epsi,epsave,f)
use types
implicit none
real(kind=dp)::ei1,ei2,ea1,ea2,cmre,f
complex(kind=dp)::epsi,epsave

ei1 = real(epsi)
ei2 = aimag(epsi)
ea1 = real(epsave)
ea2 = aimag(epsave)

cmre = f*((ei1-ea1)*(ei1+2.0_dp*ea1)+(ei2-ea2)*(ei2+2.0_dp*ea2))&
        &/((ei1+2.0_dp*ea1)**2.0_dp+(ei2+2.0_dp*ea2)**2.0_dp)

return
end function cmre
!-----------------------------
function cmim(epsi,epsave,f)
use types
implicit none
real(kind=dp)::ei1,ei2,ea1,ea2,cmim,f
complex(kind=dp)::epsi,epsave

ei1 = real(epsi)
ei2 = aimag(epsi)
ea1 = real(epsave)
ea2 = aimag(epsave)

cmim = f*((ei2-ea2)*(ei1+2.0_dp*ea1)-(ei1-ea1)*(ei2+2.0_dp*ea2))&
        &/((ei1+2.0_dp*ea1)**2.0_dp+(ei2+2.0_dp*ea2)**2.0_dp)

return
end function cmim
!--------------------------------------------------------------------------------
!       compute Jacobian 
!--------------------------------------------------------------------------------
subroutine jacobian(epsi,epsave,f,j)
use types
implicit none
real(kind=dp),dimension(2,2)::j
real(kind=dp)::ei1,ei2,ea1,ea2
real(kind=dp)::theta,theta2,xi,zeta,lmd,dxi,dzeta,dtheta1,dtheta2
real(kind=dp)::dlmd1,dlmd2,f
complex(kind=dp)::epsi,epsave

ei1 = real(epsi)
ei2 = aimag(epsi)
ea1 = real(epsave)
ea2 = aimag(epsave)

theta   = (ei1+2.0_dp*ea1)**2.0_dp+(ei2+2.0_dp*ea2)**2.0_dp
theta2  = theta * theta
xi      = (ei1-ea1)*(ei1+2.0_dp*ea1)
zeta    = (ei2-ea2)*(ei2+2.0_dp*ea2)
lmd     = (ei2-ea2)*(ei1+2.0_dp*ea1)-(ei1-ea1)*(ei2+2.0_dp*ea2)
dxi     = ei1 - 4.0_dp * ea1
dzeta   = ei2 - 4.0_dp * ea2
dtheta1 = 4.0_dp*(ei1+2.0_dp*ea1)
dtheta2 = 4.0_dp*(ei2+2.0_dp*ea2)
dlmd1   = 3.0_dp * ei2
dlmd2   = - 3.0_dp * ei1

!elements of jacobian
j(1,1) = f * (dxi*theta-(xi+zeta)*dtheta1)/theta2
j(1,2) = f * (dzeta*theta-(xi+zeta)*dtheta2)/theta2
j(2,1) = f * (dlmd1*theta-lmd*dtheta1)/theta2
j(2,2) = f * (dlmd2*theta-lmd*dtheta2)/theta2

return
end subroutine jacobian

subroutine interpolate(lmd,mat,epsip)
use types; use defstruct
implicit none
integer::i
real(kind=dp)::lmd,ipre,ipim
complex(kind=dp)::epsip
type(inclusions) mat
if(mat%wavel(1) .ge. mat%wavel(mat%nlam)) then
        if(lmd .gt. mat%wavel(1) .or. lmd .lt. mat%wavel(mat%nlam)) then
                print *, 'error: dielectric function not found!' 
                print *, 'set wmin or wmax correctly'
                stop
        else
                do i=1,mat%nlam
                if(mat%wavel(i) .eq. lmd) then
                        ipre = mat%eps1(i)
                        ipim = mat%eps2(i)
                elseif(mat%wavel(i) .gt. lmd .and. mat%wavel(i+1) .lt. lmd) then
                        !real part is linearly interpolated in linear-space
                        !imaginary part is linearly interpolated in log-log space
                        ipre=(mat%eps1(i)-mat%eps1(i+1))/(mat%wavel(i)-mat%wavel(i+1))&
                                *(lmd-mat%wavel(i+1))+mat%eps1(i+1)
                        ipim=log10(mat%eps2(i)/mat%eps2(i+1))/log10(mat%wavel(i)/mat%wavel(i+1))&
                                *log10(lmd/mat%wavel(i+1))+log10(mat%eps2(i+1))
                        ipim=10.0**(ipim)
                end if
                end do
        end if
        else
        print *, 'error: order of wavelength of data should be large to samll'
        stop
end if
epsip=cmplx(ipre,ipim)
return
end subroutine interpolate
