#######################################################################################################
##### Fine-scale changes in speed and altitude suggest protean movements in homing pigeon flights #####
#######################################################################################################
# Baptiste Garde, Rory P. Wilson, Emmanouil Lempidakis, Luca Borger, Steven J. Portugal
# Anders Hedenstrom; Giacomo Dell'Omo, Michael Quetting, Martin Wikelski, Emily L. C. Shepard.

#### Start of Script ####

#### Load the appropriate list of flights ####

## Start
# Packages:
library(afpt)
library(zoo)
library(ggplot2)
library(move)
# Function to fill in missing character variables:
repeat.before = function(x) {
  ind = which(!is.na(x))
  if(is.na(x[1]))
    ind = c(1,ind)
  rep(x[ind], times = diff(c(ind, length(x) + 1) ))
}

## Load data summary file:
# Load files from both seasons
datsum <- read.table("Data_Summary_2018.txt", header = T, sep = "\t")
datsum2 <- read.table("Data_Summary_2019.txt", header = T, sep = "\t")
datsum2$Wind <- NULL
datsum$Folder.Date <- paste("3 - Radolfzell 07-2018/", datsum$Folder.Date, sep = "")
datsum2$Folder.Date <- paste("6 - Radolfzell 04-2019/Releases/", datsum2$Folder.Date, sep = "")
# Merge files
datsum <- rbind(datsum,datsum2)
rm(datsum2)

## Select windspeeds < 2 m/s:
# Create empty columns for new variables
datsum$Wind.Speed <- NA
datsum$TripDuration <- NA
datsum$TripLength <- NA
datsum$SlopeMod1 <- NA
# Fill in with values from 1 Hz data files
for (i in 1:nrow(datsum)){
  if(file.exists(paste("Processed files/",datsum$ACC.File[i], "_1Hz.txt", sep = ""))){
    dat <- read.table(paste("Processed files/",datsum$ACC.File[i], "_1Hz.txt", sep = ""), sep = "\t", header = T)
    datsum$Wind.Speed[i] <- mean(na.omit(dat$Wind.Speed))
    datsum$TripDuration[i] <- nrow(dat)
    datsum$TripLength[i] <- sum(dat$speed)
    if(all(is.na(dat$airspeed)) == F){
      mod1 <- lm(airspeed ~ Climb, data = dat)
      datsum$SlopeMod1[i] <- coefficients(mod1)[[2]]}
  }}
# Select wind speeds < 2 m/s
datsum <- datsum[datsum$Wind.Speed <= 2,]
# Remove errors
datsum <- datsum[!is.na(datsum$Flight),]
# Remove stopovers
datsum <- datsum[is.na(datsum$Stops),]
# Convert mass into grams
datsum$Mass <- datsum$Mass/1000

## Save processed file
write.table(datsum, "Processed files/ClimbRate_DatSum.txt", row.names = F, sep = "\t")


#### Open data files ####

## Batch opening 
for (i in 1:nrow(datsum)){
  if(file.exists(paste("Processed files/",datsum$ACC.File[i], "_1Hz.txt", sep = ""))){
    dat <- read.table(paste("Processed files/",datsum$ACC.File[i], "_1Hz.txt", sep = ""), sep = "\t", header = T)
    dat$TagID <- as.character(dat$TagID)
    dat$Season <- substr(datsum$Folder.Date, 1,1)[i]
    
    mod1 <- lm(airspeed ~ Climb, data = dat)
    datsum$SlopeMod1[i] <- coefficients(mod1)[[2]]
    
    if (i == 1) {bird <- dat} else {bird <- rbind(bird, dat)}
  }}

## Rename columns appropriately
bird$FlightID <- bird$BirdID
bird$BirdID <- substr(bird$BirdID, 8, 12)
bird$Va <- sqrt(bird$airspeed^2 + bird$Climb^2)
bird$Vz <- bird$Climb; bird$Climb <- NULL



#### Distribution of speed and power curve ####
## Load package afpt (Klein Heerenbrink, 2015)
library(afpt)

## Open file with individual measurements: 
# Open for both seasons
birdinfo <- read.table("D:/Flight costs/3 - Radolfzell 07-2018/Measurements.txt", header = T, sep = "\t")
birdinfo2 <- read.table("D:/Flight costs/6 - Radolfzell 04-2019/Measurements.txt", header = T, sep = "\t")
birdinfo$season <- 3
birdinfo2$season <- 6
# Merge files
birdinfo <- rbind(birdinfo, birdinfo2)
rm(birdinfo2)

## Create data frame to store power values
powercurves <- data.frame(matrix(ncol = 23, nrow = 0))
colnames(powercurves) <- c("bird.name","speed","power","strokeplane","amplitude","frequency","flags.redFreqLo",
                           "flags.redFreqHi","flags.thrustHi","flags.speedLo","kD.ind","kD.pro0","kD.pro2","kP.ind",
                           "kP.pro0","kP.pro2","CDpro0","ReynoldsNumber","Dnf.ind","Dnf.pro0","Dnf.pro2","Dnf.par","L")
## Create additional column to receive values
birdinfo$Vmp <- NA
birdinfo$Vmr <- NA
birdinfo$Vmax <- NA
birdinfo$maxpower <- NA

## Calculate the power curves based on each bird's measurements, using "apft":
for (i in 1:nrow(birdinfo)){
  # Select a flight
  dat <- subset(bird, subset = as.character(bird$BirdID) == as.character(birdinfo$Bird.ID[i]) &
                  bird$Season == birdinfo$season[i])
  # Verify that the flight file is not empty
  if(nrow(dat) != 0 & !is.na(birdinfo$mass[i]) & !is.na(birdinfo$wing.area[i])){
    # Create a "Bird" object specific to the bird
    birdI <- Bird(
      massTotal = birdinfo$mass[i]/1000, #  weight in kg
      wingSpan = birdinfo$wingspan[i],  #  wingspan in m
      wingArea = birdinfo$wing.area[i], #  wing area in m2
      massLoad = 0.0173,
      name = 'Rock dove',
      name.scientific = 'Columba livia',
      type = 'passerine',
      source = 'July 2018')
    
    # Fill in the prepared dataset
    speed <- seq(6,30,length.out=nrow(dat)) #  airspeed in m/s
    powercurve <- computeFlappingPower(birdI,speed)
    powercurve$Vz <- dat$Vz
    powercurve$Va <- dat$Va
    powercurve$BirdID <- as.character(birdinfo$Bird.ID[i])
    powercurves <- rbind(powercurves,powercurve)
    
    # Calculate optimal speeds
    maxpower <- computeFlightPerformance(birdI)
    birdinfo$Vmp[i] <- maxpower$table[2,2]
    birdinfo$Vmr[i] <- maxpower$table[3,2]
    birdinfo$Vmax[i] <- maxpower$table[4,2]
    birdinfo$maxpower[i] <- maxpower$table[1,3]
    if(exists("bird2") == F){bird2 <- dat}else{bird2 <- rbind(bird2, dat)}
  }}

rm(birdI, bird2) 

## Summary of optimal speeds:
mean(na.omit(birdinfo$Vmp)) # 12.43 m s-1
mean(na.omit(birdinfo$Vmr)) # 16.38 m s-1

## Create "Meanbird" >D to simulate a power curve for the average measurement of tested birds
# Create data frame
Meanbird <- data.frame(t(data.frame(colMeans(birdinfo[,2:length(birdinfo)], na.rm = T))))
# Create Bird object for "afpt"
Meanbird <- Bird(
  massTotal = Meanbird$mass/1000, # weight in kg
  wingSpan = Meanbird$wingspan, # wingspan in m
  wingArea = Meanbird$wing.area, # wing area in m2
  massLoad = 0.0173, # logger mass
  name = 'Rock dove',
  name.scientific = 'Columba livia',
  type = 'passerine',
  source = 'July 2018')

# Create a vector with speed values between 6 and 30
speed <- seq(6,30,length.out = 50) #  airspeed in m/s
# Calculate power curve
powercurve <- computeFlappingPower(Meanbird,speed)

## Calculate mean speed and SD for level, ascent and descent:
# Mean speed and SD for level flight
mean(na.omit(bird$Va[bird$Vz < 0.5 & bird$Vz > -0.5])) #19.77
sd(na.omit(bird$Va[bird$Vz < 0.5 & bird$Vz > -0.5])) #2.13

# Mean speed and SD for descent
mean(na.omit(bird$Va[bird$Vz < -0.5])) #21.70
sd(na.omit(bird$Va[bird$Vz < -0.5])) #2.43

# Mean speed and SD for ascent
mean(na.omit(bird$Va[bird$Vz > 0.5])) #18.52
sd(na.omit(bird$Va[bird$Vz > 0.5])) #2.24


#### Calculate power (Not used anymore) ####

## Prepare empty object to receive the data
# Create data frame to store power values
powercurves <- data.frame(matrix(ncol = 23, nrow = 0))
colnames(powercurves) <- c("bird.name","speed","power","strokeplane","amplitude","frequency","flags.redFreqLo",
                           "flags.redFreqHi","flags.thrustHi","flags.speedLo","kD.ind","kD.pro0","kD.pro2","kP.ind",
                           "kP.pro0","kP.pro2","CDpro0","ReynoldsNumber","Dnf.ind","Dnf.pro0","Dnf.pro2","Dnf.par","L")
# Create empty columns
bird$Power <- NA
bird$Powerk <- NA
bird2 <- bird[0,]

## Use the package afpt to calculate the power:
for (i in 1:nrow(birdinfo)){
  dat <- subset(bird, subset = as.character(bird$BirdID) == as.character(birdinfo$Bird.ID[i]) &
                  bird$Season == birdinfo$season[i])
  dat <- dat[!is.na(dat$Va) & !is.na(dat$Vz) & !is.na(dat$wbf) & dat$wbf > 2,]
  if(nrow(dat) != 0){
    if(!is.na(birdinfo$mass[i]) & !is.na(birdinfo$wing.area[i])){
      birdI <- Bird(
        massTotal = birdinfo$mass[i]/1000, #  weight in kg
        wingSpan = birdinfo$wingspan[i],  #  wingspan in m
        wingArea = birdinfo$wing.area[i], #  wing area in m2
        massLoad = 0.0173, # logger weight
        wingbeatFrequency = mean(dat$wbf),
        name = 'Rock dove',
        name.scientific = 'Columba livia',
        type = 'passerine',
        source = 'July 2018')
      
      # Create "powercurve" object containing all values used to calculate power:
      # powercurve <- computeFlappingPower(bird = birdI, speed = dat$Va, frequency = dat$wbf)
      powercurve <- computeFlappingPower(bird = birdI, speed = dat$Va)
      powercurves <- rbind(powercurves, powercurve)
      powercurve$Drag <- powercurve$Dnf.ind + powercurve$Dnf.pro0 + powercurve$Dnf.pro2 + powercurve$Dnf.par
      dat$Power <- dat$Va*powercurve$Drag # Calculate power based on the equation
      dat$Powerk <- powercurve$power # Calculate power with k = drag factor for induced drag and both profile drags
      bird2 <- rbind(bird2, dat)
    }else{
      dat$Power <- NA
      dat$Powerk <- NA
      bird2 <- rbind(bird2, dat)
    }}}

## Write files with Estimations of power
write.table(bird2, "Processed files/ClimbRate_alldata.txt", col.names = T, row.names = F, sep = "\t")
write.table(birdinfo, "Processed files/ClimbRate_BirdInfo_Complete.txt", col.names = T, row.names = F, sep = "\t")
rm(bird2, birdI)

#### Extract environmental data (elevation, land cover) ####

## Load packages to extract env data
library(zoo)
library(nlme)
library(MASS)
library(raster)
library(dplyr)

## Import...
# Load previously saved dataset:
bird <- read.table("Processed files/ClimbRate_alldata.txt", header = T, sep = "\t")
if(length(bird$Climb) != 0) {bird$Vz <- bird$Climb; bird$Climb <- NULL}

# Digital surface model:
srtm <- raster("GIS/srtm_germany_DSM(30m Resolution).tif")
# Digital elevation model:
dem <- raster("GIS/DEM25m_reproj.tif")
# Land cover
clc <- raster("GIS/CLC_raster_Radolfzell_georef.tif")

## Extract data for each flight:
for (i in 1:nrow(datsum)){
  # Open each flight
  #dat <- read.table(paste("Processed files/",datsum$ACC.File[i], "_50Hz.txt", sep = ""), sep = "\t", header = T)
  dat <- bird[as.character(bird$FlightID) == as.character(datsum$ACC.File[i]),]
  if (nrow(dat) > 0){
    
    # Extract elevation from DEM and DSM
    dat$elev <- extract(dem, cbind(dat$location.long, dat$location.lat)) # DEM
    dat$elev <- rollapply(dat$elev, width = 3, by = 1, align = "center", FUN = mean, fill = NA)
    dat$dsm <- extract(srtm, cbind(dat$location.long, dat$location.lat)) # DSM
    dat$dsm <- rollapply(dat$dsm, width = 3, by = 1, align = "center", FUN = mean, fill = NA)
    dat$agl <- dat$Altitude2s - dat$dsm # Altitude above ground based on DSM
    
    # Extract CLC
    bird.coord <- cbind(dat$location.long, dat$location.lat)
    clc.ref <- extract(clc,bird.coord)
    rm(bird.coord)
    dat$clc <- clc.ref
    dat$Land.Cover <- "Land"
    dat$Land.Cover[dat$clc == 22 | dat$clc == 23 | dat$clc == 24 | dat$clc == 25 | dat$clc == 29] <- "Woodlands"
    rm(clc.ref)
    
    row.names(dat) <- c() # Reset row names
    dat$Route <- datsum$Route[i] # Set Route label
    
    # Delete short passage over LC (<=5s):
    dat$LC.sections <- NA
    dat$LC.sections[dat$Land.Cover == "Woodlands"] <- 2
    dat$LC.sections[dat$Land.Cover == "Land"] <- 1
    y2 <- rle(dat$Land.Cover)$lengths
    dat$count <- rep(y2,y2)
    dat$Land.Cover[dat$count <= 5] <- NA
    dat$Land.Cover <- repeat.before(dat$Land.Cover)
    dat$count <- NULL
    rm(y2)
    
    #Calculate rate of change of altitude:
    dat$ddsm <- c(NA, dat$dsm[2:nrow(dat)] - dat$dsm[1:(nrow(dat)-1)])
    
    #assign(x = as.character(datsum$ACC.File[i]), value = dat) # Save flight as object (not used)
    if(exists("bird2") == F) {bird2 <- dat[0,]}
    bird2 <- rbind(bird2, dat)
  }}

## Write files
bird <- bird2
write.table(bird2, "Processed files/ClimbRate_alldata_topo.txt", col.names = T, row.names = F, sep = "\t")
write.table(birdinfo, "Processed files/Birdinfo_Pigeons.txt", col.names = T, row.names = F, sep = "\t")
rm(bird2)
bird <- unique(bird)


#### Obtain summary statistics for the data summary file ####

## Load data:
bird <- read.table("Processed files/ClimbRate_alldata_topo.txt", header = T, sep = "\t")
birdinfo <- read.table("Processed files/ClimbRate_BirdInfo_Complete.txt", header = T, sep = "\t")
datsum <- read.table("Processed files/ClimbRate_DatSum.txt", header = T, sep = "\t")


## Prepare datsum:
# Remove unused columns
datsum$FlightID <- datsum$ACC.File
datsum$ACC.File <- NULL
datsum$ACC.File2 <- NULL
datsum$Stops <- NULL
datsum$date <- substr(datsum$Folder.Date, 1, 1)

# Create columns for summary statistics
# Mean
datsum$MeanVeDBA <- NA
datsum$MeanAirspeed <- NA
datsum$MeanWBf <- NA
datsum$MeanVz <- NA
# Standard deviation
datsum$sdVeDBA <- NA
datsum$sdAirspeed <- NA
datsum$sdWBf <- NA
datsum$sdVz <- NA
# Range of values (max - min)
datsum$RangeVeDBA <- NA
datsum$RangeAirspeed <- NA
datsum$RangeWBf <- NA
datsum$RangeVz <- NA
# Other variables
datsum$Wind.Speed <- NA
datsum$Wind.Dir <- NA
datsum$Start.Time <- NA



## Loop to calculate summary variables for each flight:
for (i in 1:nrow(datsum)){
  dat <- bird[bird$FlightID == datsum$FlightID[i],]
  if(nrow(dat > 0)){
    # Mean
    datsum$MeanVeDBA[i] <- mean(na.omit(dat$SmVeDBA2s))
    datsum$MeanAirspeed[i] <- mean(na.omit(dat$Va))
    datsum$MeanWBf[i] <- mean(na.omit(dat$wbf))
    datsum$MeanVz[i] <- mean(na.omit(dat$Vz))
    # Stanard deviation
    datsum$sdVeDBA[i] <- sd(na.omit(dat$SmVeDBA2s))
    datsum$sdAirspeed[i] <- sd(na.omit(dat$Va))
    datsum$sdWBf[i] <- sd(na.omit(dat$wbf))
    datsum$sdVz[i] <- sd(na.omit(dat$Vz))
    # Range
    datsum$RangeVeDBA[i] <- max(dat$SmVeDBA2s, na.rm = T) - min(dat$SmVeDBA2s, na.rm = T)
    datsum$RangeAirspeed[i] <- max(dat$Va, na.rm = T) - min(dat$Va, na.rm = T)
    datsum$RangeWBf[i] <- max(dat$wbf, na.rm = T) - min(dat$wbf, na.rm = T)
    datsum$RangeVz[i] <- max(dat$Vz, na.rm = T) - min(dat$Vz, na.rm = T)
    # Other variables
    datsum$Wind.Speed[i] <- mean(na.omit(dat$Wind.Speed))
    datsum$Wind.Dir[i] <- mean(na.omit(dat$Direction.True))
    datsum$Start.Time[i] <- dat$Tnum[1]/3600
    
  }}

## Summary stats for speed
# For the whole flight
mean(datsum$RangeAirspeed, na.rm = T) # 10.41
max(datsum$MeanAirspeed, na.rm = T) # 22.89
# Separately for hill and valley flight
mean(datsum$sdVz[datsum$Route == "Valley"], na.rm = T) # 0.99
mean(datsum$sdVz[datsum$Route == "Hill"], na.rm = T) # 1.28


#### Results: Summary Statistics ####

## Prepare data summary:
# Create season column
datsum$Season <- NA
datsum$Season[substr(datsum$Folder.Date, 1,1) =="3"] <- "July"
datsum$Season[substr(datsum$Folder.Date, 1,1) =="6"] <- "April"
# Add empty columns
datsum$Ptot <- NA
datsum$Altitude <- NA
datsum$Elevation <- NA
datsum$AGL <- NA
datsum$Va <- NA
datsum$range <- NA
datsum$Temperature <- NA
datsum$Wind.Dir <- NA
# Fill in empty columns
for (i in 1:nrow(datsum)){
  dat <- subset(bird, subset = as.character(bird$FlightID) == as.character(datsum$FlightID[i]))
  datsum$Altitude[i] <- mean(na.omit(dat$Altitude2s))
  datsum$AGL[i] <- mean(na.omit(dat$agl))
  datsum$Va[i] <- mean(na.omit(dat$Va))
  datsum$Elevation[i] <- mean(na.omit(dat$dsm))
  datsum$Temperature[i] <- mean(na.omit(dat$Temperature))
  datsum$Wind.Dir[i] <- mean(na.omit(dat$Direction.True))
  datsum$range[i] <- max(na.omit(dat$Va))-min(na.omit(dat$Va))
  datsum$Ptot[i] <- sum(na.omit(dat$Pp)+na.omit(dat$Pk))
}
# Calculate total power (not used)
datsum$Ptot[datsum$Ptot == 0] <- NA
datsum$Ptotpertime <- datsum$Ptot/datsum$TripDuration


## Difference of airspeed between the two session (based on average speed of each bird)
# Calculate average airspeed per individual in each season
birdinfo$Va <- NA
for (i in 1:nrow(birdinfo)){
  dat <- subset(bird, subset = as.character(bird$BirdID) == as.character(birdinfo$Bird.ID[i])
                & as.character(bird$Season) == as.character(birdinfo$season[i]))
  birdinfo$Va[i] <- mean(na.omit(dat$Va))}
# Check normality of the variable airspeed
shapiro.test(birdinfo$Va[birdinfo$season == 3])
shapiro.test(birdinfo$Va[birdinfo$season == 6])
# Compare airspeed between seasons
t.test(Va ~ season, data = birdinfo[c(1,3,4,6,7,8,10,11,13,14),], paired = T) # t = 0.35709, df = 4, p-value = 0.7391

## Summary stats of airspeed 
# Average speed of each flight
mean(na.omit(datsum$Va))
sd(na.omit(datsum$Va))
max(na.omit(datsum$Va))

# All data
mean(bird$Va)
sd(bird$Va)
summary(bird$Va)

# Level flight
mean(bird$Va[bird$Vz > -0.5 & bird$Vz < 0.5]) # 19.77
sd(bird$Va[bird$Vz > -0.5 & bird$Vz < 0.5]) # 2.13
# Ascending flight
mean(bird$Va[bird$Vz >  0.5]) #18.50
sd(bird$Va[bird$Vz > 0.5]) #2.27
# Descending flight
mean(bird$Va[bird$Vz < -0.5]) #21.69
sd(bird$Va[bird$Vz < -0.5]) #2.50

# Flight duration
mean(datsum$TripDuration/60) # 6 minutes 7 seconds 
sd(datsum$TripDuration/60) # 1 minutes 3 seconds 
# Flight length
mean(datsum$TripLength) # 7.21 km
sd(datsum$TripLength) # 929 m

# Average mass:
mean(na.omit(birdinfo$mass)) # 454.98
sd(na.omit(birdinfo$mass)) # 14.70


## Extreme speeds of each flight:
# Create empty columns
datsum$VaMax <- NA
datsum$VaMin <- NA
datsum$VaRange <- NA
# Open each flight to identify extreme speeds
for (i in 1:nrow(datsum)){
  dat <- bird[as.character(bird$FlightID) == as.character(datsum$FlightID[i]),]
  datsum$VaMax[i] <- max(na.omit(dat$airspeed))
  datsum$VaMin[i] <- min(na.omit(dat$airspeed))
  datsum$VaRange[i] <- datsum$VaMax[i] - datsum$VaMin[i]
}
# Eiminate infinite values
datsum$VaMax[is.infinite(datsum$VaMax)] <- NA
datsum$VaMin[is.infinite(datsum$VaMin)] <- NA
datsum$VaRange[is.infinite(datsum$VaRange)] <- NA
# Mean max, min and range
mean(na.omit(datsum$VaRange)) # 10.34 m/s
mean(na.omit(datsum$VaMax)) # 25.22 m/s
mean(na.omit(datsum$VaMin)) # 14.87 m/s


## Counting number of flights over hill and valley:
length(which(datsum$Route == "Hill")) # 8
length(which(datsum$Route == "Valley")) # 20


## Comparison of mass between the two seasons (not used)
# Remove individuals that were not weighed/had wind measurements in both seasons
subbird <- birdinfo[birdinfo$Bird.ID != "p63" & birdinfo$Bird.ID != "p180" &
                      birdinfo$Bird.ID != "p735",,]
# Test normality of the variable mass
shapiro.test(birdinfo$mass[birdinfo$season == 3])
shapiro.test(birdinfo$mass[birdinfo$season == 6])
# Test homogeneity of variances
bartlett.test(mass ~ season, data = birdinfo)
# Compare mass between seasons
t.test(mass ~ season, data = birdinfo, paired = T) # t = 0.096006, df = 6, p-value = 0.9266

## Comparison of Wing loading between the two seasons
# Calculate Wing loading
birdinfo$wing.load <- birdinfo$mass/1000/birdinfo$wing.area # Convert mass in kg
# Test normality of wing loading
shapiro.test(birdinfo$wing.load[birdinfo$season == 3])
shapiro.test(birdinfo$wing.load[birdinfo$season == 6])
# Test homogeneity of variances
bartlett.test(wing.load ~ season, data = birdinfo)
# Compare wing loading between seasons
t.test(wing.load ~ season, data = subbird, paired = T) 
# t = 0.096006, df = 6, p-value = 0.9266


## Variation of speed (dVa)
# Calculate the variation of speed as the differential of airspeed at 1 and 5 s
for (i in 1:nrow(datsum)){
  dat <- bird[as.character(bird$FlightID) == as.character(datsum$FlightID[i]),]
  if (nrow(dat) > 0){
    # 1 s
    dat$dVa <- c(NA,dat$Va[2:nrow(dat)]-dat$Va[1:(nrow(dat)-1)])
    # 5 s
    dat$dVa5s <- c(NA, NA, NA, (dat$Va[6:nrow(dat)]-dat$Va[1:(nrow(dat)-5)])/5, NA, NA)
    # Measure the progress through the flight (in terms of distance)
    dat <- dat[!is.na(dat$location.lat),]
    dat$timestamp <- as.POSIXct(dat$timestamp)
    dat = dat[!duplicated(dat$Tnum),]
    lst <- rep(NA, nrow(dat))
    for (j in 1:(nrow(dat)-1)){
      lst[j+1] <- distm(c(dat$location.long[j], dat$location.lat[j]), 
                        c(dat$location.long[j+1], dat$location.lat[j+1]), fun = distHaversine)
    }
    # Re-calculate step length between two GPS fixes
    dat$step <- lst
    # Calculate progress
    dat$progress <- c(NA, cumsum(dat$step[2:nrow(dat)]))/sum(lst, na.rm = T)*100
    if(i == 1){bird2 <- dat} else {bird2 <- rbind(bird2, dat)}
    if(i == 1){plot(dVa ~ progress, data = dat, type = "l", ylim = c(-6,6))} else {lines(dVa ~ progress, data = dat)}
  }}
bird <- bird2

# Plot and summary statistics of dVa
hist(bird2$dVa)
IQR(bird2$dVa, na.rm = T) # 0.55
summary(bird2$dVa, na.rm = T)


## Summary stats of Climb rate
# Calculate climb angle
bird$VxVzAngle <- asin(bird$Vz/bird$Va)*180/pi
# Maximum climb angle
max(bird$VxVzAngle) # 14°
# Proportion of climb angles below 5°
nrow(bird[bird$Vz > 0 & bird$VxVzAngle < 5,])/nrow(bird[bird$Vz > 0,])*100 # 90.6 %

# Plot and summary statistics of Climb rate
for (i in 1:nrow(datsum)){
  #dat <- read.table(paste("Processed files/",datsum$ACC.File[i], "_50Hz.txt", sep = ""), sep = "\t", header = T)
  dat <- bird2[as.character(bird$FlightID) == as.character(datsum$FlightID[i]),]
  if (nrow(dat) > 0){
    if(i == 1){plot(Vz ~ progress, data = dat, type = "l", ylim = c(-12,9))} else {lines(Vz ~ progress, data = dat)}
  }}
hist(bird2$Vz)
IQR(bird2$Vz, na.rm = T)
summary(bird2$Vz)
# Median climb rate: 
median(bird2$Vz[bird2$Vz > 0], na.rm = T) # 0.63
IQR(bird2$Vz[bird2$Vz > 0], na.rm = T) # 0.75
# Median descent rate:
median(bird2$Vz[bird2$Vz < 0], na.rm = T) # -0.66
IQR(bird2$Vz[bird2$Vz < 0], na.rm = T) # 0.93

sd(bird2$Vz[bird2$Route == "Valley"], na.rm = T) # 0.96
sd(bird2$Vz[bird2$Route == "Hill"], na.rm = T) # 1.2


# Write data:
write.table(datsum, "Processed files/ClimbRate_DatSum_FULL.txt", col.names = T, row.names = T, sep = "\t")
write.table(birdinfo, "Processed files/ClimbRate_BirdInfo_FULL.txt", col.names = T, row.names = T, sep = "\t")
write.table(bird, "Processed files/ClimbRate_alldata_FULL.txt", col.names = T, row.names = T, sep = "\t")


#### Results: Mixed-effect models ####

## Load packages needed for models
library(lme4)
library(nlme)
library(MuMIn)
library(car)
library(ggplot2)

## Load data:
datsum <- read.table("Processed files/ClimbRate_DatSum_FULL.txt", header = T, sep = "\t")
birdinfo <- read.table("Processed files/ClimbRate_BirdInfo_FULL.txt", header = T, sep = "\t")
bird <- read.table("Processed files/ClimbRate_alldata_FULL.txt", header = T, sep = "\t")

# Rescaling variables:
bird2 <- bird
bird2$Vz <- scale(bird2$Vz)
bird2$Altitude2s <- scale(bird2$Altitude2s)

# Rename datsum columns for clarity
datsum$FlightID <- datsum$ACC.File
datsum$ACC.File <- NULL
datsum$ACC.File2 <- NULL
datsum$Stops <- NULL

# Model 1: Airspeed as a function of Climb rate, land cover, altitude and interactions 
# Model selection (All tested, but not shown)
mod1 <- lmer(Va ~ Vz * Land.Cover * Altitude2s + (1|FlightID), data = bird2) 
mod1b <- lmer(Va ~ Vz + Land.Cover + Altitude2s + Vz:Land.Cover + Vz:Altitude2s + Land.Cover:Altitude2s + (1|FlightID), data = bird2) 
mod1c <- lmer(Va ~ Vz + Land.Cover + Altitude2s + Vz:Land.Cover + Vz:Altitude2s + (1|FlightID), data = bird2) 
mod1e <- lmer(Va ~ Vz + Land.Cover + Altitude2s + Vz:Altitude2s + (1|FlightID), data = bird2) 
anova(mod1, mod1b, mod1c, mod1e) # Complex models do not improve on Model E
rm(mod1b, mod1c, mod1e)
# Results
mod1 <- lmer(Va ~ Vz + Land.Cover + Altitude2s + Vz:Altitude2s + (1|FlightID), data = bird2) 
summary(mod1)
Anova(mod1)
r.squaredGLMM(mod1) # R2m = 0.27; r2c = 0.62

# Unstandardise estimates:
  -1.32271/sd(bird$Vz)
  0.30989/sd(bird$Altitude2s)
  -0.18703/sd(bird$Vz)/sd(bird$Altitude2s)

## Model 2 two-sided: Va ~ Vz when climbing and descending
# Climbing
mod2a <- lmer(Va ~ Vz + (1|FlightID), data = bird[bird$Vz > 0,])
summary(mod2a) # Estimate = -1.09276; std = 0.04261; t-value = -25.64
Anova(mod2a) # Significant effect < 2.2e-16; X2 = 657.56
r.squaredGLMM(mod2a) # R2m = 0.0711; R2c = 0.533
# Descending
mod2b <- lmer(Va ~ Vz + (1|FlightID), data = bird[bird$Vz < 0,])
summary(mod2b) # Estimate = -1.15560; std = 0.03095; t-value = -37.34
Anova(mod2b) # Significant effect < 2.2e-16; X2 = 1394.3
r.squaredGLMM(mod2b)# R2m = 0.1445; R2c = 0.5915



#### Plots ####

### Double plot Va ~ Vz:
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/VaVz_SplitRegression.png", width = 1000, height = 600)
bob <- ggplot(data = bird, aes(x = Vz, y = Va, colour = BirdID)) + # Verify that the mass is in kg!
  geom_point(size = 2, alpha = 0.8) + theme_classic() + 
  theme(legend.position = "none", text = element_text(size = 40)) +
  geom_smooth(method = "lm", fill = NA, size = 2, data = subset(bird, Pp < 0)) +  
  geom_smooth(method = "lm", fill = NA, size = 2, data = subset(bird, Pp > 0)) +  
  xlab (expression(paste("Vertical velocity (m s"^"-1", ")"))) +
  ylab(expression(paste("Airspeed (m s"^"-1", ")"))) +  
  geom_vline(aes(xintercept = 0), lty = 2, size = 2) + 
  scale_color_manual(values = c("cornflowerblue", "chartreuse3", "darkorchid3", "orange", "red", "brown", "cyan3", " gray50"))
print(bob)
dev.off()


### Double plot Ptot ~ Vz:
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/PtotVz_SplitRegression.png", width = 1000, height = 600)
bob <- ggplot(data = bird, aes(x = Vz, y = Ptot, colour = FlightID)) +
  geom_point(size = 2) + theme_classic() + theme(legend.position = "none", text = element_text(size = 40)) +
  geom_smooth(method = "lm", fill = NA, size = 2, data = subset(bird, Pp < 0)) +  
  geom_smooth(method = "lm", fill = NA, size = 2, data = subset(bird, Pp > 0)) +  
  xlab (expression(paste("Climb rate (m s"^"-1", ")"))) + ylab("Total power (W)") +
  geom_vline(aes(xintercept = 0), lty = 2, size = 2) + scale_color_grey()
print(bob)
dev.off()


### Plot climb rate and variation of speed:
## Overall means:
# Create empty objects
MeanVz <- data.frame(cbind(c(1:100), NA))
MeanDVa <- data.frame(cbind(c(1:100), NA))
MeanDVa5s <- data.frame(cbind(c(1:100), NA))
# Calculate mean speed and Vz (all flights confounded)  
for (i in 1:100){
  MeanDVa$X2[i] <- mean(bird$dVa[bird$progress > i-1 & bird$progress < i], na.rm = T)
  MeanDVa5s$X2[i] <- mean(bird$dVa5s[bird$progress > i-1 & bird$progress < i], na.rm = T)
  MeanVz$X2[i] <- mean(bird$Vz[bird$progress > i-1 & bird$progress < i], na.rm = T)
}

## Progression plots
# Plot 1: Change of altitude through the flights
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/CR.png", width = 1000, height = 600)
p1 <- ggplot() + theme_classic() + ylim(min(bird$Vz, na.rm = T), max(bird$Vz, na.rm = T)) + 
  geom_line(data = bird, aes(x  = progress, y = Vz, colour = BirdID), size = 1) + xlab("% progression") +
  geom_line(data = MeanVz, aes(x  = X1, y = X2), col = "gray", size = 2) + 
  theme(text = element_text(size = 40)) +ylab(expression("Change in altitude" ~ (m ~ s^{-1})))
print(p1)
dev.off()

# Plot 2: Change in speed through the flights
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/dVa.png", width = 1000, height = 600)
p2 <- ggplot() + theme_classic() + ylim(min(bird$dVa, na.rm = T), max(bird$dVa, na.rm = T)) +
  geom_line(data = bird, aes(x  = progress, y = dVa, group = BirdID), size = 1) + xlab("% progression") +
  geom_line(data = MeanDVa, aes(x  = X1, y = X2), col = "gray", size = 2) +
  theme(text = element_text(size = 40)) + ylab(expression("Change in airspeed" ~ (m ~ s^{-2})))
print(p2)
dev.off()

## Violin plots
# Plot 1: Change of altitude through the flights
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/CR_Violin.png", width = 200, height = 600)
p1 <- ggplot(data = bird, aes(x  = 1, y = Vz)) + theme_classic() + 
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) + xlab("Density") +
  ylim(min(bird$Vz, na.rm = T), max(bird$Vz, na.rm = T)) + ylab(expression("Change in altitude" ~ (m ~ s^{-1}))) +
  scale_x_continuous(breaks = c(0.6,1,1.4)) + 
  theme(text = element_text(size = 40), axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p1)
dev.off()

# Plot 2: Change in speed through the flights
png(filename = "E:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/dVa_Violin.png", width = 200, height = 600)
p2 <- ggplot(data = bird, aes(x  = 1, y = dVa)) + theme_classic() +
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) + xlab("Density") +
  ylim(min(bird$dVa, na.rm = T), max(bird$dVa, na.rm = T)) + ylab(expression("Change in airspeed" ~ (m ~ s^{-2}))) +
  scale_x_continuous(breaks = c(0.6,1,1.4)) +
  theme(text = element_text(size = 40), axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p2)
dev.off()

# Clean up workspace 
rm(MeanVz, MeanDVa, MeanDVa5s)


### Elevation profile
## Calculate the mean elevation of valley and hill through the flight
# Prepare empty data frame for hill and valley
MeanValleyElev <- data.frame(cbind(c(1:100), NA, NA))
MeanHillElev <- data.frame(cbind(c(1:100),NA, NA))
# Complete datasets
for (i in 1:100){
  # Ground elevation valley
  MeanValleyElev$X2[i] <- mean(bird$elev[bird$progress > i-1 & bird$progress < i & bird$Route == "Valley"], na.rm = T)
  # Flight altitude valley
  MeanValleyElev$X3[i] <- mean(bird$Altitude2s[bird$progress > i-1 & bird$progress < i & bird$Route == "Valley"], na.rm = T)
  # Ground elevation hill
  MeanHillElev$X2[i] <- mean(bird$elev[bird$progress > i-1 & bird$progress < i & bird$Route == "Hill"], na.rm = T)
  # Flight altitude hill
  MeanHillElev$X3[i] <- mean(bird$Altitude2s[bird$progress > i-1 & bird$progress < i & bird$Route == "Hill"], na.rm = T)
}

## Elevation profile plots:
# Valley
p1 <- ggplot(data = MeanValleyElev, aes(x = X1, y = X3)) + theme_classic() +
  geom_ribbon(aes(ymin= 400 , ymax = MeanValleyElev$X2), alpha = 0.5) + theme(text = element_text(size=8)) +
  geom_line(data = bird[bird$Route == "Valley",], aes(x = progress, y = Altitude2s, group = FlightID), size = 0.25) +
  geom_line(data = MeanValleyElev, aes(x = X1, y = X3), colour = "gray", size = 1.5) +
  ylim(400, max(bird$Altitude2s, na.rm = T)) + xlab("% progression") + ylab("Altitude ASL")
# Hill
p2 <- ggplot(data = MeanHillElev, aes(x = X1, y = X3)) + theme_classic() + 
  geom_ribbon(aes(ymin= 400 , ymax = MeanHillElev$X2), alpha = 0.5) + theme(text = element_text(size=8)) +
  geom_line(data = bird[bird$Route == "Hill",], aes(x = progress, y = Altitude2s, group = FlightID), size = 0.25) +
  geom_line(data = MeanHillElev, aes(x = X1, y = X3), colour = "gray", size = 1.5) +
  ylim(400, max(bird$Altitude2s, na.rm = T)) + xlab("% progression") + ylab("Altitude ASL")
# Draw plots
print(p1)
print(p2)

## Violin plots
# Valley
p1 <- ggplot(data = bird[bird$Route == "Valley",], aes(x  = 1, y = Altitude2s)) + theme_classic() +
  geom_violin() + geom_boxplot(width = 0.2, outlier.shape = NA) + theme(text = element_text(size=8)) +
  ylab("Altitude ASL") + xlab("Density") + # ylim(400, max(bird$Altitude2s, na.rm = T)) +
  theme(axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
# Hill
p2 <- ggplot(data = bird[bird$Route == "Hill",], aes(x  = 1, y = Altitude2s)) + theme_classic() +
  geom_violin() + geom_boxplot(width = 0.2, outlier.shape = NA) + theme(text = element_text(size=8)) +
  ylab("Altitude ASL") + xlab("Density") + # ylim(400, max(bird$Altitude2s, na.rm = T)) +
  theme(axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
# Draw plots
print(p1)
print(p2)

# Clean up workspace
rm(MeanValleyElev, MeanHillElev)



#### Supplementary information: Comparison of variability with Ultralight data ####

### 1. Rate of change of altitude

## Data preparation
# Load raw data
UL <- read.table("SI_Ultralight_DD_050419.txt", header = T, sep = "\t")
pigeon <- read.table("SI_Pigeon736_DD_050419.txt", header = T, sep = "\t")

# Set timestamp
options(digits.secs = 3)
UL$timestamp <- as.POSIXct(strptime(paste("05/04/2019 ", UL$Time.hh.mm.ss.ddd, sep = ""), "%d/%m/%Y %H:%M:%OS"))
pigeon$timestamp <- as.POSIXct(strptime(paste("05/04/2019 ", pigeon$Time.hh.mm.ss.ddd, sep = ""), "%d/%m/%Y %H:%M:%OS"))
# Select flight
UL <- UL[UL$timestamp > as.numeric(as.POSIXct("2019-04-05 09:16:50")) & 
           UL$timestamp < as.numeric(as.POSIXct("2019-04-05 09:20:50")),]
pigeon <- pigeon[pigeon$timestamp > as.numeric(as.POSIXct("2019-04-05 09:18:00")) & 
                   pigeon$timestamp < as.numeric(as.POSIXct("2019-04-05 09:22:00")),]
# Set Numeric time starting at 0
UL$Tnum <- (as.numeric(UL$timestamp) - as.numeric(UL$timestamp[1]))/60
pigeon$Tnum <- (as.numeric(pigeon$timestamp) - as.numeric(pigeon$timestamp[1]))/60

# Create the Vz variable
UL$Vz <- UL$Diff_Pressure..R.200. ; UL$Diff_Pressure..R.200. <- NULL
pigeon$Vz <- pigeon$Diff_Pressure..R.200. ; pigeon$Diff_Pressure..R.200. <- NULL


## Figures for Vz:
library(ggplot2)
## Ultralight 
# Vz through time
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/UL_Vz_plotSI.png", width = 650, height = 400)
p1 <- ggplot() + theme_classic() + ylim(-2, 2) + 
  geom_line(data = UL, aes(x  = Tnum, y = Vz), size = 1) +
  theme(text = element_text(size = 30), axis.title.x = element_blank()) + ylab(expression("Change in altitude" ~ (m ~ s^{-1})))
print(p1)
dev.off()

# Vz Violin plot
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/UL_Vz_ViolinSI.png", width = 150, height = 400)
p2 <- ggplot(data = UL, aes(x  = 1, y = Vz)) + theme_classic() +
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) +
  ylim(-2, 2) + scale_x_continuous(breaks = c(0.6,1,1.4)) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank(),
        axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p2)
dev.off()


## Pigeon 
# Vz through time
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/pigeon_Vz_plotSI.png", width = 650, height = 400)
p1 <- ggplot() + theme_classic() + ylim(-2, 2) +  
  geom_line(data = pigeon, aes(x  = Tnum, y = Vz), size = 1) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank()) + ylab(expression("Change in altitude" ~ (m ~ s^{-1})))
print(p1)
dev.off()

# Vz Violin plot
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/pigeon_Vz_ViolinSI.png", width = 150, height = 400)
p2 <- ggplot(data = pigeon, aes(x  = 1, y = Vz)) + theme_classic() +
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) + 
  ylim(-2, 2) + scale_x_continuous(breaks = c(0.6,1,1.4)) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank(),
        axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p2)
dev.off()


### 2. Rate of change of speed

## Load raw data
library(zoo)
library(move)
UL <- read.table("SI_Ultralight_GPS_050419.txt", header = T, sep = "\t")
pigeon <- read.table("SI_Pigeon736_GPS_050419.txt", header = T, sep = "\t")

# Subsample from 5Hz to 1 Hz
UL <-   UL[!duplicated(UL$timestamp),]
pigeon <-   pigeon[!duplicated(pigeon$timestamp),]

# Timestamp
UL$timestamp <- as.POSIXct(strptime(UL$timestamp, "%d/%m/%Y %H:%M:%OS"))
pigeon$timestamp <- as.POSIXct(strptime(pigeon$timestamp, "%d/%m/%Y %H:%M:%OS"))
UL <- UL[UL$timestamp > as.numeric(as.POSIXct("2019-04-05 09:16:50")) & 
           UL$timestamp < as.numeric(as.POSIXct("2019-04-05 09:20:50")),]
pigeon <- pigeon[pigeon$timestamp > as.numeric(as.POSIXct("2019-04-05 09:18:00")) & 
                   pigeon$timestamp < as.numeric(as.POSIXct("2019-04-05 09:22:00")),]

## Calculate ground speed 
# Ultralight
bm <- move(x = UL$location.long, y = UL$location.lat, 
           time=UL$timestamp, 
           data=UL, 
           proj = CRS("+init=epsg:4326 +proj=longlat +ellps=WGS84 +datum=WGS84"), 
           animal=UL$TagID)
lst <- speed(bm)
lst <- c(NA,lst) 
UL$Vg <- unlist(lst)  

# Pigeon
bm <- move(x = pigeon$location.long, y = pigeon$location.lat, 
           time=pigeon$timestamp, 
           data=pigeon, 
           proj = CRS("+init=epsg:4326 +proj=longlat +ellps=WGS84 +datum=WGS84"), 
           animal=pigeon$TagID)
lst <- speed(bm)
lst <- c(NA,lst) 
pigeon$Vg <- unlist(lst)  

## Calculate dV
UL$dV <- c(NA,UL$Vg[2:nrow(UL)] - UL$Vg[1:(nrow(UL)-1)])
pigeon$dV <- c(NA,pigeon$Vg[2:nrow(pigeon)] - pigeon$Vg[1:(nrow(pigeon)-1)])

# Set Numeric time starting at 0
UL$Tnum <- (as.numeric(UL$timestamp) - as.numeric(UL$timestamp[1]))/60
pigeon$Tnum <- (as.numeric(pigeon$timestamp) - as.numeric(pigeon$timestamp[1]))/60

## Figures:
library(ggplot2)
## Ultralight 
# dV through time
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/UL_dV_plotSI.png", width = 650, height = 400)
p1 <- ggplot() + theme_classic() + ylim(-2, 2) +  
  geom_line(data = UL, aes(x  = Tnum, y = dV), size = 1) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank()) + ylab(expression("Change in speed" ~ (m ~ s^{-2})))
print(p1)
dev.off()

# dV Violin plot
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/UL_dV_ViolinSI.png", width = 150, height = 400)
p2 <- ggplot(data = UL, aes(x  = 1, y = dV)) + theme_classic() +
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) + 
  ylim(-2, 2) + scale_x_continuous(breaks = c(0.6,1,1.4)) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank(),
        axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p2)
dev.off()

## Pigeon 
# dV through time
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/pigeon_dV_plotSI.png", width = 650, height = 400)
p1 <- ggplot() + theme_classic() + ylim(-2, 2) + 
  geom_line(data = pigeon, aes(x  = Tnum, y = dV), size = 1) + 
  theme(text = element_text(size = 30), axis.title.x = element_blank()) + ylab(expression("Change in speed" ~ (m ~ s^{-2})))
print(p1)
dev.off()

# dV Violin plot
png(filename = "F:/Flight costs/Chapter 2 - Climb rate and speed selection/Figures/pigeon_dV_ViolinSI.png", width = 150, height = 400)
p2 <- ggplot(data = pigeon, aes(x  = 1, y = dV)) + theme_classic() +
  geom_violin(size = 1.2) + geom_boxplot(width = 0.2, outlier.shape = NA, size = 1.2) + xlab("Density") +
  ylim(-2, 2) + scale_x_continuous(breaks = c(0.6,1,1.4)) +
  theme(text = element_text(size = 30), axis.title.x = element_blank(),
        axis.title.y=element_blank(), axis.text.y=element_blank(), axis.ticks.y=element_blank())
print(p2)
dev.off()




#### END of Script ####
