#!/usr/bin/env nemesis
# ----------------------------------------------------------------------
#
# Brad T. Aagaard, U.S. Geological Survey
# Charles A. Williams, GNS Science
# Matthew G. Knepley, University of Chicago
#
# This code was developed as part of the Computational Infrastructure
# for Geodynamics (http://geodynamics.org).
#
# Copyright (c) 2010-2017 University of California, Davis
#
# See COPYING for license information.
#
# ----------------------------------------------------------------------
#
# PREREQUISITES: numpy

# ======================================================================
import numpy

# Define parameters.
demFile = "topobath_points.txt"
numProfiles = 4
pointsPerProfile = 3
journalFile = "topobath_netsurf.jou"
acisFile = "topobath_surf.sat"


# Read coordinates and reshape them.
demCoords = numpy.loadtxt(demFile, dtype=numpy.float64).reshape(numProfiles, pointsPerProfile, 3)

j = open(journalFile, 'w')
j.write("# CUBIT journal file generated by topobath_netsurf.py.\n" + \
            "#\n" + \
            "# Create an ACIS NURBS surface from intersecting lines.\n" + \
            "#\n" \
            "# ----------------------------------------------------------\n" + \
            "reset\n")

# Loop over profiles (u-lines).
for iProfile in range(numProfiles):
    points = demCoords[iProfile,:,:]
    for iPoint in range(pointsPerProfile):
        point = points[iPoint,:]
        j.write("create vertex x %10.2e y %10.2e z %10.2e\n" % \
                    (point[0], point[1], point[2]))
        if 0 == iPoint:
            j.write("${idBeg=Id('vertex')}\n")
    j.write("${idEnd=Id('vertex')}\n")
    j.write("create curve spline vertex {idBeg} to {idEnd} delete\n")
    if 0 == iProfile:
        j.write("${idPBeg=Id('curve')}\n")
j.write("${idPEnd=Id('curve')}\n\n")

# Loop over contours (v-lines).
for iContour in range(pointsPerProfile):
    points = demCoords[:,iContour,:]
    for iPoint in range(numProfiles):
        point = points[iPoint,:]
        j.write("create vertex x %10.2e y %10.2e z %10.2e\n" % \
                    (point[0], point[1], point[2]))
        if 0 == iPoint:
            j.write("${idBeg=Id('vertex')}\n")
    j.write("${idEnd=Id('vertex')}\n")
    j.write("create curve spline vertex {idBeg} to {idEnd} delete\n")
    if 0 == iContour:
        j.write("${idCBeg=Id('curve')}\n")
j.write("${idCEnd=Id('curve')}\n\n")

# Create net surface.
j.write("create surface net u curve {idCBeg} to {idCEnd} v curve {idPBeg} to {idPEnd}\n")

# Delete spline curves and export Acis file.
j.write("delete curve all\n")
j.write("export Acis '%s' overwrite\n" % acisFile)
j.close()

# End of file
