% WHISPERS 2014
% Tutorial: Image analysis of hyperspectral data using mathematical morphology
% Mauro DALLA MURA and Mathieu FAUVEL
% mauro.dalla-mura@gipsa-lab.grenoble-inp.fr, mathieu.fauvel@ensat.fr}

% Tutorial Part II

clear
close all
clc

%%
init

%%
% -----------------------------------------------------------
%       Toy 1 (discrimination based on the scale)
% -----------------------------------------------------------

%%
I = imread('rect.png');
class_cmap = [0 0 0; 1 0 0; 0 1 0; 0 0 1];

C = double(imread('rect_class.png'));

figure, imshow(C, class_cmap)

%% Classification

randn('seed',0)
rand('seed',0)

number_training_pixels_per_classe = 40;
[tr_idx, ts_idx] = gen_train_test(C, number_training_pixels_per_classe);
[nr, nc] = size(C);
class_cmap = [0 0 0; 1 0 0; 0 1 0; 0 0 1];

%% only spectral

L = svm_class(double(I(:)), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% MP without reconstruction

dorec = 0;
MP = morphological_profile(I, [10 20 30 40], 'disk', dorec);
MP_vec = reshape(MP, nr*nc, size(MP,3));

figure, % Show the complete profile
for i=1:size(MP,3)
    subplot(1, size(MP,3), i)
    imshow(MP(:,:,i), [])
end

L = svm_class(double(MP_vec), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% MP with reconstruction

dorec = 1;
MP = morphological_profile(I, [10 20 30 40], 'disk', dorec);
MP_vec = reshape(MP, nr*nc, size(MP,3));

figure, % Show the complete profile
for i=1:size(MP,3)
    subplot(1, size(MP,3), i)
    imshow(MP(:,:,i), [])
end

L = svm_class(double(MP_vec), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% AP with area

attr = 'area';
lambdas = [1000  2000 5000];
AP = ext_attribute_profile(I, attr, lambdas);

figure,
for i=1:size(AP,3)
    imshow(AP(:,:,i),[]), 
    pause(.1), 
end
AP_a = double(reshape(AP, nr*nc, size(AP,3)));

L = svm_class(double(AP_a), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)


%%
% -----------------------------------------------------------
%       Toy 2 (discrimination based on the shape)
% -----------------------------------------------------------

%%

I = imread('shapes.png');

class_cmap = [0 0 0; 1 0 0; 0 1 0; 0 0 1];

C = double(imread('shapes_class.png'));

figure, imshow(C, class_cmap)

%% Classification

randn('seed',0)
rand('seed',0)

number_training_pixels_per_classe = 40;
[tr_idx, ts_idx] = gen_train_test(C, number_training_pixels_per_classe);
[nr, nc] = size(C);
class_cmap = [0 0 0; 1 0 0; 0 1 0; 0 0 1];

%% MP with reconstruction

dorec = 1;
MP = morphological_profile(I, [1 2 3 4 5], 'disk', dorec);
MP_vec = reshape(MP, nr*nc, size(MP,3));

figure, % Show the complete profile
for i=1:size(MP,3)
    subplot(1, size(MP,3), i)
    imshow(MP(:,:,i), [])
end

L = svm_class(double(MP_vec), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% MP Directional profile

dorec = 1;
line_angle = [0 45 90 135];
MP = directional_profile(I, [10 15 20 25], line_angle, dorec);
MP_vec = reshape(MP, nr*nc, size(MP,3));

figure, % Show the complete profile
for i=1:size(MP,3)
    subplot(1, size(MP,3), i)
    imshow(MP(:,:,i), [])
end

L = svm_class(double(MP_vec), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% AP with area
% For convenience we report below the values of the attribute for all the
% regions in the image. So if you want to change the thresholds in the
% filtering you can consider those.
%           45
%           49
%           89
%          121
%          153
%          168
%          237
%          250
%          301
%          322
%          336
%          397
%        20000 -> background

attr = 'area';
lambdas = [100 150 200 300];
AP = ext_attribute_profile(I, attr, lambdas);

figure,
for i=1:size(AP,3)
    imshow(AP(:,:,i),[]), 
    pause(.1), 
end
AP_a = double(reshape(AP, nr*nc, size(AP,3)));

L = svm_class(double(AP_a), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%% AP with moment of inertia

% For convenience we report below the values of the attribute for all the
% regions in the image. So if you want to change the thresholds in the
% filtering you can consider those.
%     0.1580 -> circles
%     0.1591
%     0.1633 -> squares
%     0.1653
%     0.1653
%     0.2083 -> background
%     0.4176 -> lines
%     0.4203
%     0.4646
%     0.5039
%     0.5139
%     0.5249
%     0.5831
%     0.9953
attr = 'inertia';
lambdas = [0.15 0.16 0.2 0.25];
AP = ext_attribute_profile(I, attr, lambdas);

figure,
for i=1:size(AP,3)
    imshow(AP(:,:,i),[]), 
    pause(.1), 
end
AP_i = double(reshape(AP, nr*nc, size(AP,3)));

L = svm_class(double(AP_i), C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%%
% -----------------------------------------------------------
%       Toy 3 (discrimination based on the homogeneity)
% -----------------------------------------------------------

%%
randn('seed',0)
rand('seed',0)

C = double(imread('squares_class.png'));
lab = unique(C);
ncl = length(lab);
I = zeros(size(C));
for i=1:ncl
    idx = find(C==lab(i));
    vals = mvnrnd(125, 5^i, length(idx));
    I(idx) = vals;
end
I = uint8(reshape(I, size(C)));

figure, imshow(I)

%% Compute the profiles on the image
ap_s = attribute_profile(I, 's', [1 5 10 20]);
figure, for i=1:size(ap_s,3), imshow(ap_s(:,:,i),[]), pause(.5), end
ap_a = attribute_profile(I, 'a', [10 50 100 500]);
figure, for i=1:size(ap_a,3), imshow(ap_a(:,:,i),[]), pause(.5), end

%%
randn('seed',0)
rand('seed',0)
number_training_pixels_per_classe = 40;
[tr_idx, ts_idx] = gen_train_test(C, number_training_pixels_per_classe);
[nr, nc] = size(C);
class_cmap = [0 0 0; 1 0 0; 0 1 0; 0 0 1];

%%
ap = double(reshape(ap_a, numel(C), size(ap_a, 3)));
L = svm_class(ap, C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)

%%
ap = double(reshape(ap_s, numel(C), size(ap_s, 3)));
L = svm_class(ap, C(:), tr_idx);
acc = sum(L(ts_idx) == C(ts_idx)) / length(ts_idx);
fprintf('OA data: %f\n', acc);
figure, imshow(reshape(L,nr,nc), class_cmap)
