# Conceptual framework
# 
# We investigate how trait variation at different scales (of different quality) influence the observed functional structure and inferred assembly process.
# 
# 
# For this, we use 6 methods to measure community trait patterns; each method uses trait variation of different 'quality', from best to poorest:
  
# M1. TPD considering population level traits (maximum quality, maximum similarity with the real probabilistic trait structure) (Plants)
# M2. TPD using single average trait values (bandwidth method), population level (Plants)
# M3. Convex hulls, population level (Plants)
# M4. TPD considering species level traits (Plants and Ants)
# M5. TPD using single average trait values (bandwidth method), species level (Plants and Ants)
# M6. Convex hulls, global average (Plants and Ants)
# 
# We consider M1 to provide the most accurate/realistic representation of community trait patterns, and investigate how M2-6 compare with M1 in terms of (i) observed structure and (ii) inferred assembly process.
##############################################################################################
##############################################################################################

# Using Plant data: we calculate various indices and aspects of Functional Diversity using TPD (trait probabilities based on individuals) and also using other species-mean distance-based methods. 
# The steps include: 
# (1) Data preparation; 
# (2) Community-level FD indices using TPD (two ways: species vs populations) and dbfd; 
# (3) Data analysis
# 
#
# MKL Wong & CP Carmona
# Date created: 14 Dec 2020


############################################################
# (1) DATA PREPARATION
##########################################################
# SET WD, LOAD PACKAGES, LOAD DATA
#####


setwd("C:/Users/carlos/Dropbox/Manuscritos en marcha/Mark Wong/")
setwd("/Users/c.p.carmona/Dropbox/Manuscritos en marcha/Mark Wong/")
library(TPD)
library(FD)


# Load some functions
source("AnalysesFeb2020/Rao.for.dissim.R") # Function to estimate Rao from TPDc functions
source("AnalysesFeb2020/melodic.Rao.R") # Function to estimate Rao using trait dissimilarity and abundance matrix

# Load raw plant trait dataset. We have 49 plant spp. with 2 traits measured: Height and SLA.
traits.raw <- read.table('data/plant.traits.txt', header = TRUE)
# Log transform data scale trait data
scaled.values <- apply(log(traits.raw[,3:4]), MARGIN=2, scale) 
traits <- traits.raw
# put scaled values in traits dataset. From here onwards we will use this transformed data in 'traits'
traits[,3:4] <- scaled.values[,1:2] 
hist(traits$Height, plot =T) # check normality
hist(traits$SLA, plot = T) # check normality
cor(traits$Height, traits$SLA) # check correlation
plot(traits$Height, traits$SLA) # view correlation

# SUMMARISE SPECIES ABUNDANCES IN COMMUNITIES 
abun_plants.raw <- read.table("data/plant.freq.txt", header = TRUE) # load abundance dataset for the 49 plant spp. in 40 plots.
envData <- read.table("data/Water_content.txt", header = T)

#
#
#########################################################
# (2) COMMUNITY LEVEL FD INDICES (FRIC, FEVE, FDIV, RAO): using TPD (two ways: TPDs species and TPDs populations), and dbfd (same as with ant data)
########################################################
# 
#### M1: Traits at the population level. TPDs for each population, hence accounting for environment #### 
# TPDs ***by populations***
TPDs_plants_pop<- TPDs(species = traits$Species, traits[,3:4], samples = traits$Plot, alpha=0.99)
# TPDc for 40 communities ***from TPDs by populations***
TPDc_plants_pop <- TPDc(TPDs=TPDs_plants_pop, sampUnit = abun_plants.raw)
# TPD FD indices ***from TPDs by populations***
FDtpd_pop <- REND(TPDc = TPDc_plants_pop) 
M1_summary <- cbind.data.frame(
    FDtpd_pop$communities$FRichness,
    FDtpd_pop$communities$FEvenness,
    FDtpd_pop$communities$FDivergence) # make a summary of the data.
names(M1_summary) <- c("FRic", "FEve", "FDiv") # add names

dissim_plants_pop <- dissim(TPDs_plants_pop) #Estimate dissimilarity between populations
Rao_plants_pop <- Rao(diss = dissim_plants_pop, TPDc = TPDc_plants_pop, regional=F) 
M1_summary$Rao <- Rao_plants_pop$alpha_rao # put Rao in dataset

alpha <- list()
alpha$M1 <- M1_summary

############## #### #### #### #### #### #### #### #### 
####  M2. TPD using single average trait values (TPDsMean method), population level (Plants) #### 
# In this case, I will make the TPD using the TPDsMean function, assigning the same
# bandwidth to all the species in each community. I will use the ks::Hpi.diag function for this
populationsMeans <- as.data.frame(matrix(data = NA, nrow = length(unique(paste(traits$Species, traits$Plot))),
                                         ncol = ncol(traits), 
                                         dimnames = list(unique(paste(traits$Species, traits$Plot, sep=".")),
                                                         colnames(traits))))
populationsMeans[, "Species"] <- unlist(strsplit(rownames(populationsMeans), split = ".", fixed=T))[(1:(2*nrow(populationsMeans)) %% 2) == 1 ]
populationsMeans[, "Plot"] <- unlist(strsplit(rownames(populationsMeans), split = ".", fixed=T))[(1:(2*nrow(populationsMeans)) %% 2) == 0 ]
for(i in 1:nrow(populationsMeans)){
  popSel <- traits[traits$Plot == populationsMeans[i, "Plot"] & traits$Species == populationsMeans[i, "Species"], ]  
  populationsMeans[i, "Height"] <- mean(popSel$Height, na.rm=T)
  populationsMeans[i, "SLA"] <- mean(popSel$SLA, na.rm=T)
}
# now, for each community, define bandwidth (standard deviation)
populationsMeans$sdHeight <- populationsMeans$sdSLA <- rep(NA, nrow(populationsMeans))
for(i in 1:length(unique(populationsMeans$Plot))){
  rowsSel <- which(populationsMeans$Plot == unique(populationsMeans$Plot)[i])
  populationsMeans[rowsSel, "sdHeight"] <- sqrt(diag(Hpi.diag(populationsMeans[rowsSel, c("Height", "SLA")])))[1]
  populationsMeans[rowsSel, "sdSLA"] <- sqrt(diag(Hpi.diag(populationsMeans[rowsSel, c("Height", "SLA")])))[2]
}
TPDsMean_plants_pop<- TPDsMean(species = populationsMeans$Species, means=populationsMeans[,c("Height", "SLA")], 
                               samples = populationsMeans$Plot, sds = populationsMeans[,c("sdHeight", "sdSLA")],
                               alpha=0.99)
sapply(TPDsMean_plants_pop$TPDs, sum) #check all = 1

# TPDc for 40 communities from TPDs by species.
TPDc_plants_meanpop <- TPDc(TPDs = TPDsMean_plants_pop, sampUnit = abun_plants.raw) 
# TPD FD indices from TPDs by species.
FDtpd <- REND(TPDc = TPDc_plants_meanpop) 
M2_summary <- cbind.data.frame(FDtpd$communities$FRichness, FDtpd$communities$FEvenness, FDtpd$communities$FDivergence) # make a summary of the data.
names(M2_summary) <- c("FRic", "FEve", "FDiv") # add names

dissim_plants <- dissim(TPDsMean_plants_pop) # required for Rao
Rao_plants <- Rao(diss = dissim_plants, TPDc = TPDc_plants_meanpop, regional=F) # calculate Rao. 
M2_summary$Rao <- Rao_plants$alpha_rao # add Rao into summary
alpha$M2 <- M2_summary

##########################################################################
#### M3. Convex-hull and Rao using population means (no TPD) ####
# Now we use FD::dbfd to calculate FD indices
# First, we need to create a matrix of populations x comms and fill with the relative abudances:
popsxcomm <- matrix(0, nrow=nrow(abun_plants.raw), ncol=nrow(populationsMeans),
                    dimnames = list(rownames(abun_plants.raw), rownames(populationsMeans)))
for(i in 1:nrow(populationsMeans)){
  popsxcomm[populationsMeans[i, "Plot"], rownames(populationsMeans)[i]] <- 
    abun_plants.raw[populationsMeans[i, "Plot"], populationsMeans[i, "Species"]]
}
FDdbfd_pops <- dbFD(populationsMeans[, c("Height", "SLA")], popsxcomm, 
                    w.abun = T, stand.x = F, calc.FRic = T, print.pco = T) 
# Use melodic.rao to calculate Rao.
dist_mat <- compute_dist_matrix(populationsMeans[, c("Height", "SLA")], metric = 'euclidean') # make euclidean distance matrix.
dist_mat.scaled <- apply(dist_mat, MARGIN = 2, FUN = function(X) (X - min(X))/diff(range(X))) # scale distance matrix from 0 to 1.
Raopopsxcomm <- melodic.rao(samp=popsxcomm, dis=dist_mat.scaled)
#
# Make summary of FD indices from dbFD, add RaoM too. 
M3_summary <- cbind.data.frame(FDdbfd_pops$FRic, FDdbfd_pops$FEve, FDdbfd_pops$FDiv, Raopopsxcomm$rao) # make summary
names(M3_summary) <- c("FRic", "FEve", "FDiv", "Rao") # change names
alpha$M3 <- M3_summary

######################################################################
#### M4. TPDs using all individuals of species (no populations). ####
##########
# TPDs by species.
TPDs_plants<- TPDs(species = traits$Species, traits[,3:4], alpha = 0.99)
sapply(TPDs_plants$TPDs, sum) # check that all = 1
# TPDc for 40 communities from TPDs by species.
TPDc_plants <- TPDc(TPDs = TPDs_plants, sampUnit = abun_plants.raw) 
sapply(TPDc_plants$TPDc$TPDc, sum) #check all = 1
#
# TPD FD indices from TPDs by species.
FDtpd <- REND(TPDc = TPDc_plants) 
M4_summary <- cbind.data.frame(FDtpd$communities$FRichness, FDtpd$communities$FEvenness, FDtpd$communities$FDivergence) # make a summary of the data.
names(M4_summary) <- c("FRic", "FEve", "FDiv") # add names

dissim_plants <- dissim(TPDs_plants) # required for Rao
dissim_plants$species <- dissim_plants$populations #Required to match the names in the Rao function
dissim_plants$populations <- NULL
Rao_plants <- Rao(diss = dissim_plants, TPDc = TPDc_plants, regional=F) # calculate Rao. 
M4_summary$Rao <- Rao_plants$alpha_rao # add Rao into summary
alpha$M4 <- M4_summary

##############################################################################################
#### M5. TPD using single average trait values (TPDsMean method), species level (Plants) ####
# In this case, I will make the TPD using the TPDsMean function, assigning to each species and trait bandwidth
# value estimated using the ks::Hpi.diag
speciesMeans <- as.data.frame(matrix(data = NA, nrow = length(unique(traits$Species)),
                                         ncol = ncol(traits), 
                                         dimnames = list(unique(traits$Species),
                                                         colnames(traits))))
speciesMeans[, "Species"] <- unique(traits$Species)
heightAux <- tapply(traits$Height, traits$Species, mean, na.rm=T)
speciesMeans[, "Height"] <- heightAux[rownames(speciesMeans)]
SLAAux <- tapply(traits$SLA, traits$Species, mean, na.rm=T)
speciesMeans[, "SLA"] <- SLAAux[rownames(speciesMeans)]

speciesMeans[, "sdHeight"] <- speciesMeans[, "sdSLA"] <- NA
for(i in 1:length(unique(speciesMeans$Species))){
  spSel <-  as.character(unique(speciesMeans$Species)[i])
  traitsSel <- subset(traits, traits$Species == spSel)
  speciesMeans[spSel, c("sdHeight", "sdSLA")] <- sqrt(diag(Hpi.diag(traitsSel[, c("Height", "SLA")])))
}

TPDsMean_plants_sps<- TPDsMean(species = speciesMeans$Species, means=speciesMeans[,c("Height", "SLA")], 
                               sds = speciesMeans[,c("sdHeight", "sdSLA")],
                               alpha=0.99)
sapply(TPDsMean_plants_sps$TPDs, sum) #check all = 1

# TPDc for 40 communities from TPDs by species.
TPDc_plants_meansp <- TPDc(TPDs = TPDsMean_plants_sps, sampUnit = abun_plants.raw) 
sapply(TPDc_plants$TPDc$TPDc, sum) #check all = 1
# plotTPD(TPD=TPDc_plants_meansp) # see plots
#
# TPD FD indices from TPDs by species.
FDtpd <- REND(TPDc = TPDc_plants_meansp) 
M5_summary <- cbind.data.frame(FDtpd$communities$FRichness, FDtpd$communities$FEvenness, FDtpd$communities$FDivergence) # make a summary of the data.
names(M5_summary) <- c("FRic", "FEve", "FDiv") # add names

dissim_plants <- dissim(TPDsMean_plants_sps) # required for Rao
dissim_plants$species <- dissim_plants$populations
dissim_plants$populations <- NULL
Rao_plants <- Rao(diss = dissim_plants, TPDc = TPDc_plants_meansp, regional=F) # calculate Rao. 
M5_summary$Rao <- Rao_plants$alpha_rao # add Rao into summary
alpha$M5 <- M5_summary


##########
##### M6. Convex-hull and Rao using global means (no TPD) #####
traits.sp <- aggregate.data.frame(traits[, c("Height", "SLA")], by = traits['Species'], FUN = mean, na.rm = T) 
row.names(traits.sp) <- traits.sp[,1]
traits.sp <- traits.sp[,-1]
# Now we use dbfd to calculate FD indices
# First, we need to create a matrix of populations x comms and fill with the relative abudances:
FDdbfd_sps <- dbFD(traits.sp, abun_plants.raw, 
                    w.abun = T, stand.x = F, calc.FRic = T, print.pco = T) 
# Use melodic.rao to calculate Rao.
dist_mat <- compute_dist_matrix(traits.sp, metric = 'euclidean') # make euclidean distance matrix.
dist_mat.scaled <- apply(dist_mat, MARGIN = 2, FUN = function(X) (X - min(X))/diff(range(X))) # scale distance matrix from 0 to 1.
Raospsxcomm <- melodic.rao(samp=abun_plants.raw, dis=dist_mat.scaled) # i just call it RaoM to remind me it is your melodic.rao 
#
# Make summary of FD indices from dbFD, add RaoM too. 
M6_summary <- cbind.data.frame(FDdbfd_sps$FRic, FDdbfd_sps$FEve, FDdbfd_sps$FDiv, Raospsxcomm$rao) # make summary
names(M6_summary) <- c("FRic", "FEve", "FDiv", "Rao") # change names
alpha$M6 <- M6_summary


saveRDS(alpha, "alphaResultsPlants.rds")


##############################################################################
##############################################################################
########### ANALYSES
##############################################################################
##############################################################################
waterGrad <- envData$WaterCont
waterGrad2 <- waterGrad**2
waterGrad3 <- waterGrad**3

library(RColorBrewer)
library(mgcv)
library(vegan)
library(MuMIn)

Colores<-brewer.pal(7, "Set1")
Colores <- Colores[-6]
alphaPoints<-0.25
ColTransp<-c(rgb(t(col2rgb(Colores)/255),  alpha=alphaPoints, maxColorValue=1))
varPlot <- c("FRic"
             #,"FEve"
             #, "FDiv"
             , "Rao")
nombresVar <- c("FRic"
                # , "Functional evenness"
                # , "Functional divergence"
                , "Rao")
namesMethods <- c("M1. TPD; individuals\nwithin populations",
                  "M2. TPD;\npopulation mean",
                  "M3. Classic;\npopulation mean",
                  "M4. TPD; individuals\nacross populations",
                  "M5a. TPD;\nspecies mean",
                  "M6. Classic;\nspecies mean")
lwdLines <- 5

#### Patterns along the gradient####
pdf("Manuscript/Figures/Figure_plants_gradient_R1.pdf",  
    height = 0.8*14*(length(varPlot)), width = 0.8*14*length(namesMethods), pointsize=50)

par(mfrow=c(length(varPlot) , length(namesMethods)), 
    mgp=c(1, 0.1, 0), 
    mar = c(2, 3, 2, 0),
    oma=c(2,2,3,2))
#### #### 
plotNum <- 1
for(i in 1:(length(varPlot))){ #For each component of diversity (FRic, Rao, Beta), one row
  for(j in 1:length(alpha)){ # For each method, one column
    labX <- "Soil water content (%)"
    labY <- ifelse(j == 1, nombresVar[i], "")
    # alphaUse <- scale(alpha[[i]][, j])
    if(i != 3){ # FRichness and Rao
      alphaUse <- (alpha[[j]][, i])
      allPreds <- allPredsPlot <- c("waterGrad", "waterGrad2", "waterGrad3")
      modelo<-lm(alphaUse ~ waterGrad + waterGrad2 + waterGrad3, na.action="na.fail")
      modeloDredge<-dredge(modelo, rank = "AICc", subset = dc(waterGrad, waterGrad2, waterGrad3))
      selectedModel <- get.models(modeloDredge, 1)[[1]]
      if(j == 1){
        selectedVarsBestModel <- rownames(anova(selectedModel))
        selectedVarsBestModel <- selectedVarsBestModel[ - length(selectedVarsBestModel)]
        if(length(selectedVarsBestModel) > 0){
          nonSelectedVarsBestModel <- allPreds[- which(allPreds %in% selectedVarsBestModel)]
        }else{
          nonSelectedVarsBestModel <- allPreds
        }
        selectedVarsModel <- rownames(anova(selectedModel))
        selectedVarsModel <- selectedVarsModel[ - length(selectedVarsModel)]
        positionBestModel <- as.numeric(which(rowSums(!is.na(modeloDredge[, selectedVarsBestModel, drop=F])) == length(selectedVarsBestModel) &
                                     rowSums(!is.na(modeloDredge[, nonSelectedVarsBestModel, drop = F])) == 0 ))
        deltaBestModel <- modeloDredge[positionBestModel, "delta"]
      } else{
        selectedVarsModel <- rownames(anova(selectedModel))
        selectedVarsModel <- selectedVarsModel[ - length(selectedVarsModel)]
        if(length(selectedVarsBestModel) > 0){
          positionBestModel <- as.numeric(which(rowSums(!is.na(modeloDredge[, selectedVarsBestModel, drop = F])) == length(selectedVarsBestModel) &
                                       rowSums(!is.na(modeloDredge[, nonSelectedVarsBestModel, drop = F])) == 0))
        } else{
          positionBestModel <- which(rowSums(!is.na(modeloDredge[, nonSelectedVarsBestModel])) == 0)
        }
        deltaBestModel <- modeloDredge[positionBestModel, "delta"]
      }
      if(length(selectedVarsModel) == 0){
        leg <- vector("expression", 2) # One for R2, one for no predictors, one for deltaAIC
        leg[1]<- substitute(expression(R^2== MYVALUE), 
                            list( MYVALUE = max(0, round(summary(selectedModel)$r.sq,3))))[2]
        leg[2]<- substitute(expression(Delta[AICc]== MYVALUE), 
                            list( MYVALUE = round(deltaBestModel, 3)))[2]
      } else{
        leg <- vector("expression", 2 + length(selectedVarsModel))
        leg[1]<- substitute(expression(R^2== MYVALUE), 
                            list( MYVALUE = max(0, round(summary(selectedModel)$r.sq,3))))[2]
        leg[2]<- substitute(expression(Delta[AICc]== MYVALUE), 
                            list( MYVALUE = round(deltaBestModel, 3)))[2]
      }
      limY <- range(alphaUse); limY[2] <- limY[2] + 0.5 * (limY[2] - limY[1])
      plot(alphaUse ~ waterGrad, ylab="", axes=F, xlab="", pch = 16, col=Colores[j], 
           bg = "grey85" , ylim= limY, cex=1.5)
      mtext(side=1, adj=0.5, labX, outer=F, line=1.5)
      mtext(side=2, adj=0.5, labY, outer=F, line=1.8)
      axis(1, tcl=0.25, lwd=0.8)
      axis(2, las=1, tcl=0.25,lwd=0.8)
      box( bty = "l", lwd =0.8 ) 
      posLeg <- "topright"
      legend(posLeg, leg, adj=0, bty="n", cex=1.4)
      if(i==1){
        mtext(side=3, adj=0.5, paste0(namesMethods[j]), outer=F, line=0.5)
      }
      mtext(side=3, adj=0, paste0(letters[plotNum], ") "), outer=F, line=-0.5)
      plotNum <- plotNum + 1
      waterGradSeq <- seq(min(waterGrad), max(waterGrad), length.out = 100)
      waterGradSeq2 <- waterGradSeq**2
      waterGradSeq3 <- waterGradSeq**3
      
      predMat <- data.frame(waterGrad = waterGradSeq, waterGrad2 = waterGradSeq2,
                                waterGrad3 = waterGradSeq3)
      lines <- predict(selectedModel, newdata = predMat)
      lines(waterGradSeq, lines, lwd=lwdLines, col = Colores[j], lty= 1)
    }else{ ## Beta
      betaUse <- as.matrix(beta[[j]]$total)
      betaWater <- as.matrix(dist(waterGrad))
      plot(0, type="n", xlim=range(betaWater), ylim=range(betaUse), xlab="", ylab="", axes=F)
      mtext(side=1, adj=0.5, labX, outer=F, line=1.5)
      mtext(side=2, adj=0.5, labY, outer=F, line=1.8)
      axis(1, tcl=0.25, lwd=0.8)
      axis(2, las=1, tcl=0.25,lwd=0.8)
      box( bty = "l", lwd =0.8 ) 
      
      cexbetaPoints<-0.8
      mtext(side=2, adj=0.5, labY, outer=F, line=1.8)
      betaUse <- as.numeric(betaUse)
      betaWater <- as.numeric(betaWater)
      modeloBeta<-gam(betaUse~s(betaWater, k=4))

      points(betaWater, betaUse, pch = 16, col = ColTransp[j], bg = "grey85", cex = cexbetaPoints)
      xv<-seq(min(betaWater), max(betaWater), length.out = 100)
      predictors<-data.frame(betaWater=xv)
      yv<-predict(modeloBeta, newdata = predictors, type="response")
      lines(xv, yv, col=Colores[j], lwd=lwdLines, lty=1)
      mtext(side=3, adj=0, paste0(letters[plotNum], ") "), outer=F, line=-0.5)
      plotNum <- plotNum + 1
    }
  }
}
#### #### 
dev.off()




################## CORRELATIONS BETWEEN METHODS

pdf("Manuscript/Figures/Figure_plants_correlationsAlphaFRic_R1.pdf",  
    width = 30, height = 10, pointsize=23)


matFRicPlot <- matrix(1:36, nrow=6, ncol=6, byrow=T)
matRaoPlot <- matrix((1:36)+36, nrow=6, ncol=6, byrow=T)
matBetaPlot <- matrix((1:36)+36*2, nrow=6, ncol=6, byrow=T)
matBlank <- matrix(0, nrow=6, ncol=1)
layoutMat <- cbind(matFRicPlot, matBlank, matRaoPlot, matBlank, matBetaPlot)
widthsCols <- c(rep(1, 6), 0.5, rep(1, 6), 0.5, rep(1, 6))
widthsCols <- widthsCols/sum(widthsCols)
layout(layoutMat, widths = widthsCols)
par(mgp=c(1, 0.1, 0), 
    mar = c(0.1, 0.1, 0.1, 0.1),
    oma=c(1, 1, 3 , 1))

##### FUNCTIONAL RICHNESS ----

for(i in 1:length(alpha)){#Rows
  for(j in 1:length(alpha)){#Columns
    if(i == j){#Diag
      xAxis <- alpha[[i]]$FRic
      dens <- density(xAxis)
      plot(dens$x, dens$y, type="l", ylab="", axes=F, xlab="", 
           xlim=1.1*range(dens$x), ylim = 1.5 * range(dens$y), 
           col=Colores[i], lwd=lwdLines)
      points(xAxis, rep(0, length(xAxis)), bg=ColTransp[i], 
                        col=Colores[i], pch=21, cex=0.8)
      box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
      legend("topright", leg=paste0(names(alpha)[i]), bty="n")
      if(i==6){axis(1,tcl=0.25,lwd=0.8)}
      if(i == 1)  mtext(side=3, adj=0, "a) Functional richness", outer=F, line=0.5, cex=1.5)
      
    } else{
      if(j < i){#Lower trian
        xAxis <- (alpha[[j]]$FRic)
        yAxis <- (alpha[[i]]$FRic)
        densX <- density(xAxis)
        densY <- density(yAxis)
        plot(xAxis, yAxis, ylab="", axes=F, xlab="", 
             pch=21, bg="grey80", col="grey30",
             xlim=1.1*range(densX$x),
             ylim=1.1*range(densY$x))
        box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
        corr <-cor.test(xAxis, yAxis)
        leg<-vector("expression",2)
        leg[1]<- substitute(expression(rho== MYVALUE), 
                            list( MYVALUE = max(0, round(corr$estimate,2))))[2]
        pValor<-round(corr$p.value,3)
        leg[2] <- ifelse(pValor < 0.001, "p < 0.001", paste0("p = ", pValor))
        legend("topleft", leg=leg, bty="n")
        abline(lm(yAxis~xAxis), lty=2)
      } 
      if(j > i){#Upper trian
        plot(0, type="n", ylab="", axes=F, xlab="")
      }
      if(j==1){axis(2, las=1, tcl=0.25,lwd=0.8)}
      if(i==6){axis(1,tcl=0.25,lwd=0.8)}
    }
  }
}

##### RAO ----

for(i in 1:length(alpha)){#Rows
  for(j in 1:length(alpha)){#Columns
    if(i == j){#Diag
      xAxis <- alpha[[i]]$Rao
      dens <- density(xAxis)
      plot(dens$x, dens$y, type="l", ylab="", axes=F, xlab="", 
           xlim=range(dens$x), ylim = 1.5 * range(dens$y), 
           col=Colores[i], lwd=lwdLines)
      points(xAxis, rep(0, length(xAxis)), bg=ColTransp[i], 
             col=Colores[i], pch=21, cex=0.8)
      box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
      legend("topright", leg=paste0(names(alpha)[i]), bty="n")
      if(i==6){axis(1,tcl=0.25,lwd=0.8)}
      if(i == 1)  mtext(side=3, adj=0, "b) Rao", outer=F, line=0.5, cex=1.5)
      
    } else{
      if(j < i){#Lower trian
        xAxis <- (alpha[[j]]$Rao)
        yAxis <- (alpha[[i]]$Rao)
        densX <- density(xAxis)
        densY <- density(yAxis)
        plot(xAxis, yAxis, ylab="", axes=F, xlab="", 
             pch=21, bg="grey80", col="grey30",
             xlim=1.1*range(densX$x),
             ylim=1.1*range(densY$x))
        box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
        corr <-cor.test(xAxis, yAxis)
        leg<-vector("expression",2)
        leg[1]<- substitute(expression(rho== MYVALUE), 
                            list( MYVALUE = max(0, round(corr$estimate,2))))[2]
        pValor<-round(corr$p.value,3)
        leg[2] <- ifelse(pValor < 0.001, "p < 0.001", paste0("p = ", pValor))
        legend("topleft", leg=leg, bty="n")
        abline(lm(yAxis~xAxis), lty=2)
      } 
      if(j > i){#Upper trian
        plot(0, type="n", ylab="", axes=F, xlab="")
      }
      if(j==1){axis(2, las=1, tcl=0.25,lwd=0.8)}
      if(i==6){axis(1,tcl=0.25,lwd=0.8)}
    }
  }
}

##### BETA ----

for(i in 1:length(beta)){#Rows
  for(j in 1:length(beta)){#Columns
    if(i == j){#Diag
      xAxis <- beta[[i]]$total
      dens <- density(xAxis)
      plot(dens$x, dens$y, type="l", ylab="", axes=F, xlab="", 
           xlim=range(dens$x), ylim = 1.5 * range(dens$y), 
           col=Colores[i], lwd=lwdLines)
      points(xAxis, rep(0, length(xAxis)), bg=ColTransp[i], 
             col=Colores[i], pch=21, cex=0.8)
      box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
      legend("topright", leg=paste0(names(alpha)[i]), bty="n")
      if(i==6){axis(1,tcl=0.25,lwd=0.8)}
      if(i == 1)  mtext(side=3, adj=0, "c) Beta", outer=F, line=0.5, cex=1.5)
      
    } else{
      if(j < i){#Lower trian
        xAxis <- (beta[[j]]$total)
        yAxis <- (beta[[i]]$total)
        densX <- density(xAxis)
        densY <- density(yAxis)
        plot(xAxis, yAxis, ylab="", axes=F, xlab="", 
             pch=21, bg="grey80", col="grey30", cex=0.8,
             xlim=c(-0.1, 1.2),
             ylim=c(-0.1, 1.2))
        box(lwd =0.8) ##Esto hace que los ejes se toquen en el origen
        corr <-cor.test(xAxis, yAxis)
        leg<-vector("expression",2)
        leg[1]<- substitute(expression(rho== MYVALUE), 
                            list( MYVALUE = max(0, round(corr$estimate,2))))[2]
        pValor<-round(corr$p.value,3)
        leg[2] <- ifelse(pValor < 0.001, "p < 0.001", paste0("p = ", pValor))
        legend("topleft", leg=leg, bty="n")
        abline(lm(yAxis~xAxis), lty=2)
      } 
      if(j > i){#Upper trian
        plot(0, type="n", ylab="", axes=F, xlab="")
      }
      if(j==1){axis(2, las=1, tcl=0.25,lwd=0.8, at=seq(0, 1, by=0.25))}
      if(i==6){axis(1,tcl=0.25,lwd=0.8, at=seq(0, 1, by=0.5))}
    }
  }
}
#####----
dev.off()    







