import matplotlib.pyplot as plt
import numpy as np

# Here we define a new agent SineGeneratorAgent, and override the functions :
# init_parameters & agent_loop
# init_parameters() is used to setup the data stream and necessary parameters
# agent_loop() is an infinite loop, and will read from the stream continuously,
# then it sends the data to its output channel via send_output
# Each agent has internal current_state which can be used as a switch by the
# AgentNetwork
import torch
from sklearn import datasets, svm
from sklearn.metrics import confusion_matrix
from sklearn.model_selection import train_test_split
from sklearn.utils.multiclass import unique_labels

from agentMET4FOF.agentMET4FOF.agents import AgentMET4FOF, AgentNetwork, MonitorAgent


class Cuda_Agent(AgentMET4FOF):
    def init_parameters(self):
        y = torch.tensor([1.0, 2.0]).cuda()


class ConfusionMatrixAgent(AgentMET4FOF):
    def agent_loop(self):
        # import some data to play with
        iris = datasets.load_iris()
        X = iris.data
        y = iris.target
        class_names = iris.target_names

        # Split the data into a training set and a test set
        X_train, X_test, y_train, y_test = train_test_split(X, y, random_state=0)

        # Run classifier, using a model that is too regularized (C too low) to see
        # the impact on the results
        classifier = svm.SVC(kernel="linear", C=0.01)
        y_pred = classifier.fit(X_train, y_train).predict(X_test)
        # Plot normalized confusion matrix
        new_plot = plot_confusion_matrix(
            y_test,
            y_pred,
            classes=class_names,
            normalize=True,
            title="Normalized confusion matrix",
        )

        self.send_plot(new_plot)


def plot_confusion_matrix(
    y_true, y_pred, classes, normalize=False, title=None, cmap=plt.cm.Blues
):
    """
    This function prints and plots the confusion matrix.
    Normalization can be applied by setting `normalize=True`.
    """
    if not title:
        if normalize:
            title = "Normalized confusion matrix"
        else:
            title = "Confusion matrix, without normalization"

    # Compute confusion matrix
    cm = confusion_matrix(y_true, y_pred)
    # Only use the labels that appear in the data
    classes = classes[unique_labels(y_true, y_pred)]
    if normalize:
        cm = cm.astype("float") / cm.sum(axis=1)[:, np.newaxis]
        print("Normalized confusion matrix")
    else:
        print("Confusion matrix, without normalization")

    print(cm)

    fig, ax = plt.subplots()
    im = ax.imshow(cm, interpolation="nearest", cmap=cmap)
    ax.figure.colorbar(im, ax=ax)
    # We want to show all ticks...
    ax.set(
        xticks=np.arange(cm.shape[1]),
        yticks=np.arange(cm.shape[0]),
        # ... and label them with the respective list entries
        xticklabels=classes,
        yticklabels=classes,
        title=title,
        ylabel="True label",
        xlabel="Predicted label",
    )

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=45, ha="right", rotation_mode="anchor")

    # Loop over data dimensions and create text annotations.
    fmt = ".2f" if normalize else "d"
    thresh = cm.max() / 2.0
    for i in range(cm.shape[0]):
        for j in range(cm.shape[1]):
            ax.text(
                j,
                i,
                format(cm[i, j], fmt),
                ha="center",
                va="center",
                color="white" if cm[i, j] > thresh else "black",
            )
    fig.tight_layout()
    return fig


def main():
    # start agent network server
    agentNetwork = AgentNetwork()

    # init agents by adding into the agent network
    gen_agent = agentNetwork.add_agent(agentType=ConfusionMatrixAgent)
    monitor_agent = agentNetwork.add_agent(agentType=MonitorAgent)

    # connect agents by either way:
    # 1) by agent network.bind_agents(source,target)
    agentNetwork.bind_agents(gen_agent, monitor_agent)

    # 2) by the agent.bind_output()
    gen_agent.bind_output(monitor_agent)

    # set all agents states to "Running"
    agentNetwork.set_running_state()

    # allow for shutting down the network after execution
    return agentNetwork


if __name__ == "__main__":
    main()
