library(dplyr)
library(ggplot2)
library(survey)
library(lubridate)
library(tidyverse)

correctedPrevalence = function(smallthresh = FALSE, adjustspec = FALSE, agesexn = TRUE)
  #Returns the prevalence corrected by specificity, sensitivity and age-sex normalization
{
  prev_df <- read.csv("data/sero_consolidated_weights.csv", stringsAsFactors = F)
  if(smallthresh)
    prev_df <- prev_df %>% mutate(covid_resultado = ifelse(abbott_result_alt == "Positive", 1, 0))
  else
    prev_df <- prev_df %>% mutate(covid_resultado = ifelse(abbott_result == "Positive", 1, 0))
  
  
  if(smallthresh) {
    se = 0.925      #sensitivity
    sp = 0.967      #specificity
  }
  else {
    se = 0.84
    sp = 0.999
  }
  
  # Make weighted and unweighted survey object
  un_weighted_df <- svydesign(id = ~1, weights = ~1, data = prev_df)
  weighted_df <- survey::svydesign(id = ~1, weights = ~weights, data = prev_df)
  
  # Calculate prevlanece without weights 
  prev_unweighted <- svyby(~covid_resultado, 
                           by = ~location + donmo, 
                           design = un_weighted_df, 
                           FUN = svyciprop, method = "be", vartype = c("ci", "var")) %>% rename("var_unweighted" =  "var.as.numeric(covid_resultado)")
  
  # Calculate prevalence with weights
  prev_weighted <- svyby(~covid_resultado, 
                         by = ~location + donmo, 
                         design = weighted_df, 
                         FUN = svyciprop, method = "be", vartype = c("ci", "var")) %>% rename("var_weighted" =  "var.as.numeric(covid_resultado)")
  
  prevalence <- prev_unweighted %>% 
    rename(prev = covid_resultado) %>% 
    left_join(prev_weighted %>% 
                rename(
                  prev_weighted = covid_resultado,
                  ci_l_weighted = ci_l,
                  ci_u_weighted = ci_u
                ))
  
  # Sens/specificity correction (method from https://doi.org/10.1155/2011/608719)
  adjust_SensSpec <- function(p) {
    adj <- (p + sp - 1) / (se + sp - 1)
    adj <- ifelse(adj < 0, 0, adj)
    return(adj)
  }
  
  prevalence <- prevalence %>% 
    mutate(prev_weighted_adj = adjust_SensSpec(prev_weighted),
           ci_l_weighted_adj = adjust_SensSpec(ci_l_weighted),
           ci_u_weighted_adj = adjust_SensSpec(ci_u_weighted))
  
  prevalence <- prevalence %>% 
    mutate(prev_unweighted_adj = adjust_SensSpec(prev),
           ci_l_unweighted_adj = adjust_SensSpec(ci_l),
           ci_u_unweighted_adj = adjust_SensSpec(ci_u))
  
    prevalence = prev_df %>%  group_by(donmo, location) %>% summarize(n = n()) %>% merge(prevalence, by = c("donmo", "location"))
  
  if(adjustspec) {
    if(agesexn) {
      return(data.frame(donmo = prevalence$donmo, location = prevalence$location, n = prevalence$n, 
                        prev = prevalence$prev_weighted_adj, ci_l = prevalence$ci_l_weighted_adj, ci_u = prevalence$ci_u_weighted_adj,
                        var = prevalence$var_weighted))
    } else {
      return(data.frame(donmo = prevalence$donmo, location = prevalence$location, n = prevalence$n, 
                        prev = prevalence$prev_unweighted_adj, ci_l = prevalence$ci_l_unweighted_adj, ci_u = prevalence$ci_u_unweighted_adj,
                        var = prevalence$var_unweighted))
    }
  }
  else {
    if(agesexn) {
      return(data.frame(donmo = prevalence$donmo, location = prevalence$location, n = prevalence$n,  
                        prev = prevalence$prev_weighted, ci_l = prevalence$ci_l_weighted, ci_u = prevalence$ci_u_weighted, 
                        var = prevalence$var_weighted))
    }
    else{
      return(data.frame(donmo = prevalence$donmo, location = prevalence$location, n = prevalence$n,  
                        prev = prevalence$prev_unweighted, ci_l = prevalence$ci_l_unweighted, ci_u = prevalence$ci_u_unweighted,
                        var = prevalence$var_unweighted))
    }
  }
}

calculate.u = function(rho, alpha, delay = FALSE, p0 = 1)
{
  drho = diff(c(0, rho))
  A = matrix(0, nrow = length(drho), ncol = length(drho))
  for(i in 1:length(drho)) {
    A[i,i] = 1
    for(j in 1:length(drho)) {
      if(j < i - delay)
        A[i,j] = -(p0*(1-alpha)/alpha)*alpha^(i-j)
    }
  }
  u = solve(A, drho)
  return(u)
}

calculate.alphamin.p.SP = function(rho, delay = FALSE, alpha0, p0)
{
  alpha = linspace(1E-2, 1 - 1E-2, 101)
  p = linspace(1E-2, 1 - 1E-2, 101)
  iopt = 0
  jopt = 0
  J = Inf
  for(i in 1:length(p))
    for(j in 1:length(alpha))
    {
      u = calculate.u(rho, alpha[j], delay, p[i])
      J0 = abs(alpha[j] - alpha0) + abs(p[i] - p0)
      if(J0 < J && all(u >= 0))
      {
        J = J0
        iopt = i
        jopt = j
      }
    }
  
  return(data.frame(alpha = alpha[jopt], p = p[iopt]))
}


calculate.alphamin.p = function(rho, delay = FALSE)
{
  alpha = linspace(1E-2, 1 - 1E-2, 101)
  p = linspace(1E-2, 1 - 1E-2, 101)
  iopt = 0
  jopt = 0
  J = Inf
  for(i in 1:length(p))
    for(j in 1:length(alpha))
    {
      u = calculate.u(rho, alpha[j], delay, p[i])
      J0 = abs(u[length(u)-2]) + abs(u[length(u)-3])  #Minimizes the number of cases in July and August
      if(J0 < J && all(u >= 0))
      {
        J = J0
        iopt = i
        jopt = j
      }
    }
  
  return(data.frame(alpha = alpha[jopt], p = p[iopt]))
}

prev.bootstrap = function(posc)
{
  gamma = -1 + (posc$prev*(1-posc$prev))/posc$var
  alpha = posc$prev*gamma
  beta = gamma - alpha
  
  prev = sapply(1:length(alpha), function(i) ifelse(alpha[i] == 0, 0, rbeta(1, alpha[i], beta[i])))
  
  return( data.frame(donmo = posc$donmo, location = posc$location, n = posc$n, prev = prev))
}

