/*
    Copyright 2002-2013 CEA LIST

    This file is part of LIMA.

    LIMA is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    LIMA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with LIMA.  If not, see <http://www.gnu.org/licenses/>
*/
#ifndef ABSTRACTANALYSISHANDLER_H
#define ABSTRACTANALYSISHANDLER_H

#include <sstream>
#include "common/ProcessUnitFramework/AnalysisContent.h"

namespace Lima {

/**
 * The AbstractAnalysisHandler class defines the callback interface:
 * the functions that are to be defined to handle the events generated
 * by the LIMA linguistic analyzer.
 *
 * The interface includes functions that are used only in the analysis
 * of structured input
 */
class AbstractAnalysisHandler : public AnalysisData
{
public:

  virtual ~AbstractAnalysisHandler() {}

  /** 
   * function called by the LIMA analyzer on start of a new document
   *
   * @param props the document properties, which can contain some
   *              metadata that are automatically generated by the
   *              analyzer (such as language, date of analysis etc),
   *              and some metadata that are directly extracted from
   *              the document to analyze, in case of the analysis of
   *              an XML documents (in this case, the configuration of
   *              the analysis indicates in which tags the information
   *              can be found)
   */
//   virtual void startDocument(const Common::Misc::GenericDocumentProperties& props) = 0;

  /** 
   * function called by the LIMA analyzer on start at end of a document 
   */
//   virtual void endDocument() = 0;
  
  /** 
   * function called by the LIMA analyzer on the start of a text part
   * that is to be analyzed
   */
  virtual void startAnalysis() = 0;
  
//   virtual void startAnalysis(std::string bloc_type) {startAnalysis();};

  /** 
   * main function of the handler: handle data produced by the dumper
   * at end of the analysis of a given text part.
   *
   * This data is stored in a generic byte buffer (may be simple text or 
   * serialized binary data, according to the type of dumper used)
   *
   * @param buf    the buffer containing the result of the analysis  
   * @param length the size of the buffer 
   */
  virtual void handle(const char* buf,int length) =0;
  
  /** 
   * function called by the LIMA analyzer on the end of a text part
   * that is to be analyzed
   */
  virtual void endAnalysis() = 0;
  
  /** 
   * function called by the LIMA analyzer on the start of a new
   * hierarchy node, only used in case of a structured document
   * analysis
   */
//   virtual void startNode( const std::string& elementName, bool forIndexing ) = 0;
  
  /** 
   * function called by the LIMA analyzer on the end of a hierarchy
   * node, only used in case of a structured document analysis
   */
//   virtual void endNode( const Common::Misc::GenericDocumentProperties& props ) = 0;
  
};




class AnalysisHandlerContainer : public AnalysisData
{
  public:
    AnalysisHandlerContainer ( std::map<std::string, AbstractAnalysisHandler*>& handlers ) : AnalysisData(),m_handlers ( handlers ) {};
    AbstractAnalysisHandler* getHandler(const std::string& handlerId) const {
      if (m_handlers.find(handlerId) != m_handlers.end())
        return m_handlers[handlerId];
      else
        return 0;
    };
  private:
    AnalysisHandlerContainer& operator=(const AnalysisHandlerContainer&) {return *this;}
    std::map<std::string, AbstractAnalysisHandler*>& m_handlers;
};

} // Lima

#endif
