#' Interpolate LTER data to monthly observations 
#'
#' Interpolates dataframe of observations to monthly timestep at 1 m depth intervals.
#'
#' @param lakeAbr Lake identification, string
#' @param var Variable of interest. Use availableVars() to see available variables.
#' @param maxdepth Maximum depth of lake
#' @param printFigs Output individual profiles of interpolation? Options TRUE or FALSE (default)
#' @import dplyr
#' @import akima
#' @importFrom reshape2 melt
#' @importFrom lubridate decimal_date month
#' @importFrom ggforce facet_wrap_paginate
#' @export
#' 
monthlyInterpolate <- function(lakeAbr, dfin, var, maxdepth, printFigs = FALSE) {
  # Read in data

  obs = dfin %>%
    dplyr::filter(lakeid == lakeAbr, item == var) %>%
    dplyr::group_by(sampledate,depth) %>%
    dplyr::summarise(meanVar = mean(value,na.rm=TRUE)) %>%
    dplyr::mutate(decdate = decimal_date(sampledate)) %>%
    dplyr::mutate(month = month(sampledate)) %>%
    dplyr::arrange(sampledate,depth) # %>% mutate(meanVar = log(meanVar+1))
  ############## ############## ############## ############## ##############
  
  # Dates with single sample 
  removedates = obs %>% group_by(sampledate) %>% tally() %>% filter(n == 1)
  
  usedates = obs %>%
    dplyr::filter(!is.na(meanVar)) %>%
    dplyr::filter(depth > (maxdepth/2)) %>%
    dplyr::distinct(sampledate) %>% 
    filter(!sampledate %in% removedates$sampledate)
  
  # Interpolate to 1 m depth increments 
  f <- lapply(X = usedates$sampledate, FUN = interpData, observationDF = obs,
                     maxdepth = maxdepth)
  f = as.data.frame(do.call(cbind, f))
  names(f) = usedates$sampledate
  
  # Bind list into dataframe
  df = bind_cols(depth = 0:maxdepth,f) %>% 
    pivot_longer(-1, names_to = 'sampledate', values_to = 'var') %>% 
    arrange(sampledate,depth) %>% 
    mutate(sampledate = as.Date(sampledate))
  
  # Paginate interpolation figures
  if (printFigs == TRUE) {
    for (i in 1:ceiling(length(unique(df$sampledate))/36)) {
      p = ggplot(df) + geom_point(aes(x = var, y = depth), size = 3, shape = 21) +
        geom_point(data = obs, (aes(x = meanVar, y = depth)), size = 2) +
        # scale_fill_manual(values = c('red3','gold')) +
        scale_y_reverse() +
        xlab(var) + ylab('depth') +
        theme_bw() +
        theme(title = element_text(size = 8),
              text = element_text(size = 8)) +
        facet_wrap_paginate(vars(sampledate), nrow = 6, ncol = 6, page = i)
      print(p)
    }
  }
  
  # Interpolate to Monthly 
  full_y = seq(from = range(df$depth)[1],to = range(df$depth)[2], by = 1)
  full_x = seq.Date(from = round_date(df$sampledate[1], unit = 'month'),to = range(df$sampledate,na.rm = T)[2], by = 'month')
  # a = data.frame(x = full_x, y = full_y)
  a = expand.grid(sampledate = full_x, depth = full_y)
  
  interped = akima::interp(x = df$sampledate, y = df$depth, z = df$var, full_x, full_y,
                           duplicate = 'mean',linear=T, extrap = F)
  
  
  dimnames(interped$z) =  list(full_x, interped$y)
  df3 <- melt(interped$z, varnames = c('date', 'depth'), value.name = 'var')
  
  df3$date = as.Date(df3$date,origin = '1970-01-01') 
  df3 = arrange(df3, date, depth)
  
  return(list(observations = obs, weeklyInterpolated = df3))
}

# Vertical linear interpolation of water column concentrations 
interpData <- function(observationDF, date, maxdepth) {
  a = observationDF %>% filter(sampledate == date)
  if (sum(!is.na(a$meanVar)) == 0) {
    print('nothing')
    return(NULL)
  }
  
  b = a %>% filter(!is.na(meanVar))
  if (max(b$depth) < (maxdepth/2)) {
    print('too shallow')
    return(NULL)
  }
  
  yout = approx(x = a$depth, y = a$meanVar, xout = c(0:maxdepth), rule = 2)
  return(yout$y)
}


#' Decompose timeseries from monthly data
#'
#' @param df.load dataframe of montly estimates
#' @param lakeAbr Lake identification, string
#' @param var Variable of interest. Use availableVars() to see available variables.
#' @import dplyr
#' @importFrom tidyr pivot_longer
#' @export
decomposeTS2 <- function(df.load, lakeAbr, var, output.df = FALSE) {
  
  ## weights for moving avg
  fltr <- c(1/2, rep(1, times = 11), 1/2)/12
  ## create a time series (ts) object from the var data
  var.conc <- ts(data = df.load$intConc.mgL, frequency = 12, start = c(df.load$year[1],
                                                                df.load$month[1]))
  ## estimate of trend
  var.trend = stats::filter(var.conc, filter = fltr, method = 'convo', sides = 2)
  
  ## seasonal effect over time
  var.1T <- var.conc - var.trend
  
  ## We can obtain the overall seasonal effect by averaging the estimates
  # for each month and repeating this sequence over all years.
  ## length of ts
  ll <- length(var.1T)
  ## frequency (ie, 12)
  ff <- frequency(var.1T)
  ## number of periods (years); %/% is integer division
  periods <- ll%/%ff
  ## index of cumulative month
  index <- seq(1, ll, by = ff) - 1
  ## get mean by month
  mm <- numeric(ff)
  for (i in 1:ff) {
    mm[i] <- mean(var.1T[index + i], na.rm = TRUE)
  }
  ## subtract mean to make overall mean=0
  mm <- mm - mean(mm)

  # Finally, let’s create the entire time series of seasonal effects
  ## create ts object for season
  var.seas <- ts(rep(mm, periods + 1)[seq(ll)], start = start(var.1T),
                 frequency = ff)
  
  # The last step in completing our full decomposition model is obtaining the random errors which we can get via simple subtraction
  ## random errors over time
  var.err <- var.conc - var.trend - var.seas

  ## plot the obs ts, trend & seasonal effect
  output = as.data.frame(cbind(var.conc, var.trend, var.seas, var.err)) %>%
    mutate(date = df.load$date) %>%
    pivot_longer(cols = 1:4, names_to = 'decompose')
  output$decompose = factor(output$decompose, levels=c('var.conc', 'var.trend', 'var.seas', 'var.err'))
  
  return(output)
}
