#!/usr/bin/python

# Copyright (c) 2018 Hong Xu <hongx@usc.edu>.

# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
# associated documentation files (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge, publish, distribute,
# sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies or
# substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
# NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
# OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

# This script generates the dependency network of Debian packages. It generates two files: *.edges
# and *.desc. Simply run this file on a Debian (or Ubuntu) system.
#
# If you use this script in your published work, please cite "Shudan Zhong and Hong Xu. 2018.
# Learning Embeddings of Directed Networks with Text-Associated Nodes with Application in Software
# Package Dependency Networks. arXiv:1809.02270."

from __future__ import print_function

import platform
from apt import Cache


class Node:
    def __init__(self, id, name, virtual):
        self.id = id
        self.virtual = virtual
        self.name = name

class Nodes:
    def __init__(self):
        self.cache = Cache()
        self.nodes_id = {}
        self.nodes = []

    def get_node(self, node):
        """Obtain the node object from its name or ID."""
        if isinstance(node, str):
            if node in self.nodes_id:
                return self.get_node(self.nodes_id[node])
            else:
                self.nodes_id[node] = len(self.nodes) + 1
                new_node = Node(len(self.nodes) + 1, node, node not in self.cache)
                self.nodes.append(new_node)
                return new_node
        elif isinstance(node, int):
            return self.nodes[node-1]
        else:
            pass



dist_name, dist_version, _ = platform.linux_distribution()

description_file = open('{}-{}.desc'.format(dist_name, dist_version), 'w+')
edge_file = open('{}-{}.edges'.format(dist_name, dist_version), 'w+')


cache = Cache()
nodes = Nodes()
for package in cache:
    name = package.name
    version = package.versions[-1]
    p_id = nodes.get_node(str(name)).id
    print('[PACKAGE] {} {}'.format(p_id, name), file=description_file)
    print(package.versions[0].description, file=description_file)
    print('', file=description_file)
    for dependency in version.dependencies:
        for p in dependency:
            if p.rawtype in ('Depends', 'PreDepends', 'Recommends', 'Suggests'):
                print('{} {}'.format(
                    p_id, nodes.get_node(p.name).id), file=edge_file)
            elif p.rawtype in ('Enhances', 'Replaces'):
                print('{} {}'.format(nodes.get_node(p.name).id, p_id), file=edge_file)

edge_file.close()
