# -----------------------------------------------------------------------------
# calc.py
#
# A simple calculator with variables.   This is from O'Reilly's
# "Lex and Yacc", p. 63.
# -----------------------------------------------------------------------------

import sys
import re
import pdb
from lexerConcrete import lexer_main
from lexerConcrete import *

underApprox = False
getOverApprox = False
extractInv = False
names = set([])
isStem = True


def p_program(p):
    'program : varlist BEGIN lasso END'
    p[0] = "{}\nbegin\n{}\nend".format(p[1], p[3])


# def p_program_withComment(p):
#   'program : FIRSTLINE varlist BEGIN COMMENT lasso END'
#   p[0] = p[2] + '\n' + 'begin\n' + p[5] + '\nend'


def p_program_withComment(p):
    'program : varlist BEGIN COMMENT lasso END'
    p[0] = "{}\nbegin\n{}\nend".format(p[1], p[4])


def p_varlist(p):
    "varlist : VAR vars_decl ';' "
    p[0] = "var {};".format(p[2])


def p_vars_decl(p):
    "vars_decl : NAME ':' INT"
    p[0] = "{}:{}".format(p[1], str(p[3]))
    names.add(p[1])


def p_vars_decl_recurrent(p):
    "vars_decl : vars_decl ',' NAME ':' INT"
    p[0] = "{},{}:{}".format(p[1], p[3], str(p[5]))
    names.add(p[3])


def p_lasso(p):
    'lasso : stem cycle'
    # print("parsed concrete lasso")
    p[0] = "{}\n{}".format(p[1], p[2])


def p_stem(p):
    'stem : statement_list'
    global isStem
    # print("parsed stem")
    isStem = False
    p[0] = p[1][0]


def p_cycle(p):
    "cycle : WHILE boolexpression DO statement_list DONE ';' "
    # print("parsed cycle")

    if underApprox:
        underAprroxWrite = "\tassume({});".format(p[2][0])
        p[0] = "while {} do\n{}\n{}\ndone;".format(p[2][0],
                                                   p[4][0],
                                                   underAprroxWrite)
    elif getOverApprox:
        p[0] = "while {} do\n{}\n\ndone;".format(p[2][0], p[4][0])


def p_cycle_withComments(p):
    "cycle : WHILE boolexpression DO COMMENT statement_list DONE ';' COMMENT"
    # print "parsed cycle"

    if p[2][1] is True:
        print("Error. Found Nonlinear Guard. Not supported.")
        sys.exit()

    # replace symbol 'var' with 'var_post'
    substs = ["{}_post".format(name) for name in names]
    regexp = "|".join("({})".format(re.escape(name)) for name in names)
    regexp = re.compile(regexp)

    if extractInv:
        invComment = p[5][1]
        findex = invComment.rfind("[|")
        if findex == -1:
            line = "{} {} {} {}\n{} {} {}\n{}".format(*p[1:])
            print("Unable to find Invariant:\n`{}`".format(line))
            raise Exception("parserConcrete: "
                            "Unable to find Invariant `{}`".format(line))
        rindex = invComment.rfind("|]")
        inv = invComment[findex + 2:rindex]
        # print("inv = {}".format(inv))
        invFormulas = re.split(';', inv)
        # print("invFormulas = {}".format(invFormulas))
        invWrite = ''
        for invFormula in invFormulas:
            if invFormula:
                # Don't use variable names that may contain chars p, o, s and t.
                invFormula = regexp.sub(lambda m: substs[m.lastindex - 1],
                                        invFormula)
                if invWrite:
                    invWrite = "{} and {}".format(invWrite, invFormula)
                else:
                    invWrite = invFormula
        invWrite = "\tassume({});".format(invWrite)

        p[0] = "while {} do\n{}\n{}\ndone;".format(p[2][0], p[5][0], invWrite)
    else:
        p[0] = "while {} do\n{}\n\ndone;".format(p[2][0], p[5][0])


def p_statement_list_base(p):
    'statement_list : statement'
    p[0] = p[1]


def p_statement_list_recurrence(p):
    'statement_list : statement_list statement'
    p[0] = (p[1][0] + '\n' + p[2][0], p[2][1])


def p_statement_skip(p):
    "statement : SKIP ';'"
    p[0] = ('\tskip;', '')


def p_statement_comment(p):
    "statement : statement COMMENT"
    p[0] = (p[1][0], p[2])


def p_statement_assign(p):
    "statement : NAME '=' expression ';' "
    if p[3][1] is True and getOverApprox:
        p[0] = ('\t' + p[1] + '=' + 'random;', '')
    else:
        p[0] = ('\t' + p[1] + '=' + p[3][0] + ';', '')


def p_statement_random(p):
    "statement : NAME '=' RANDOM ';' "
    p[0] = ('\t' + p[1] + '=' + 'random;', '')


def p_statement_interval(p):
    "statement : NAME '=' INTERVAL ';' "
    p[0] = ('\t' + p[1] + '=' + 'random;', '')


def p_statement_assume(p):
    "statement : ASSUME boolexpression ';' "
    global isStem
    if isStem:
        p[0] = ('\t' + 'assume(' + p[2][0] + ');', '')
    else:
        assumeExpression = p[2][0]
        for var in names:
            assumeExpression = assumeExpression.replace(var, var + '_post')
        p[0] = ('\t' + 'assume(' + assumeExpression + ');', '')


def p_expression_binop(p):
    '''expression : expression '+' expression
                  | expression '-' expression
                  | expression '*' expression
                  | expression '/' expression'''

    if p[2] == '+':
        if p[1][1] is True or p[3][1] is True:
            nonLinear = True
        else:
            nonLinear = False
        p[0] = (p[1][0] + '+' + p[3][0], nonLinear)
    elif p[2] == '-':
        if p[1][1] is True or p[3][1] is True:
            nonLinear = True
        else:
            nonLinear = False
        p[0] = (p[1][0] + '-' + p[3][0], nonLinear)
    elif p[2] == '*':
        if p[1][1] is True or p[3][1] is True:
            nonLinear = True
        else:
            match1 = re.search('[a-zA-Z_][a-zA-Z0-9_]*', p[1][0])
            match2 = re.search('[a-zA-Z_][a-zA-Z0-9_]*', p[3][0])
            if match1 and match2:
                nonLinear = True
            else:
                nonLinear = False
        p[0] = (p[1][0] + '*' + p[3][0], nonLinear)
    elif p[2] == '/':
        if p[1][1] is True or p[3][1] is True:
            nonLinear = True
        else:
            match1 = re.search('[a-zA-Z_][a-zA-Z0-9_]*', p[1][0])
            match2 = re.search('[a-zA-Z_][a-zA-Z0-9_]*', p[3][0])
            if match1 and match2:
                nonLinear = True
            else:
                nonLinear = False
        p[0] = (p[1][0] + '/' + p[3][0], nonLinear)


def p_expression_uminus(p):
    "expression : '-' expression %prec UMINUS"
    p[0] = ('-' + p[2][0], p[2][1])


def p_expression_group(p):
    "expression : '(' expression ')'"
    p[0] = ('(' + p[2][0] + ')', p[2][1])


def p_expression_number(p):
    "expression : NUMBER"
    p[0] = (str(p[1]), False)


def p_expression_name(p):
    "expression : NAME"
    p[0] = (p[1], False)


def p_bool_expression_paranthesis(p):
    '''boolexpression : '(' boolexpression ')' '''
    p[0] = ('(' + p[2][0] + ')', p[2][1])


def p_bool_expression_not(p):
    '''boolexpression : NOT  boolexpression %prec UMINUS '''
    p[0] = ('not (' + p[2][0] + ')', p[2][1])


def p_bool_expression_and(p):
    '''boolexpression : boolexpression AND boolexpression'''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + ' and ' + p[3][0], nonLinear)


def p_bool_expression(p):
    '''boolexpression : constraint'''
    p[0] = p[1]


def p_constraint_1(p):
    '''constraint : expression EQ expression '''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + '==' + p[3][0], nonLinear)


def p_constraint_2(p):
    '''constraint : expression LE expression '''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + '<=' + p[3][0], nonLinear)


def p_constraint_3(p):
    '''constraint : expression GE expression '''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + '>=' + p[3][0], nonLinear)


def p_constraint_4(p):
    '''constraint : expression LT expression '''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + '<' + p[3][0], nonLinear)


def p_constraint_5(p):
    '''constraint : expression GT expression '''
    if p[1][1] is True or p[3][1] is True:
        nonLinear = True
    else:
        nonLinear = False
    p[0] = (p[1][0] + '>' + p[3][0], nonLinear)


def p_constraint_true(p):
    '''constraint : TRUE '''
    p[0] = ('true', False)


def p_error(p):
    pdb.set_trace()
    print("Error while Parsing Concrete Program")
    if p:
        print("Syntax error at '%s':line = %d" % (p.value, p.lineno))
    else:
        print("Syntax error at EOF")
    sys.exit()


def main_func(filename):
    # Build the lexer
    import ply.yacc as yacc
    lexer = lexer_main()
    yacc.yacc(debug=0, write_tables=0)

    # f = open("interprocOut.txt", "r")
    f = open(filename, "r")
    out = yacc.parse(f.read(), lexer=lexer)
    f.close()
    return out


if __name__ == "__main__":
    main_func()
