from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int, Real, ToReal
from pysmt.shortcuts import Not, And, Or, Iff
from pysmt.shortcuts import GT, GE, Equals, LE, LT
from pysmt.shortcuts import Plus, Times
from pysmt.typing import INT, REAL, BOOL


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Simple linear test"""
    # symbols
    a = Symbol('a', INT)
    b = Symbol('b', REAL)
    c = Symbol('c', BOOL)

    x_a = Symbol(symb_next('a'), INT)
    x_b = Symbol(symb_next('b'), REAL)
    x_c = Symbol(symb_next('c'), BOOL)

    symbols = [a, b, c]

    one_i = Int(1)
    zero_i = Int(0)
    zero_r = Real(0)
    m_one_r = Real(-1)

    # initial condition.
    init = TRUE()
    # transition relation.
    trans = And(GE(x_a, Plus(a, one_i)), Iff(x_c, Not(c)))
    # fairness.
    fairness = Or(And(GT(a, zero_i), c, GT(b, zero_r)), LE(b, zero_r))

    # define automata to be composed.
    automata = [None, None, None, None]
    automata[0] = AGAutomaton(symbols, [a], "Aut_A", 1)
    automata[0].set_invar(0, GT(a, zero_i))
    automata[0].set_assume(0, GE(b, zero_r))
    automata[0].set_transitions(0, [(0, [Equals(x_a, Plus(a, one_i))])])

    automata[1] = AGAutomaton(symbols, [b], "Aut_B", 1)
    automata[1].set_invar(0, GT(b, zero_r))
    automata[1].set_assume(0, GE(a, zero_i))
    automata[1].set_transitions(0, [(0, [GE(x_b, Plus(b, ToReal(a)))])])

    automata[2] = AGAutomaton(symbols, [c], "Aut_C", 2)
    automata[2].set_invar(0, c)
    automata[2].set_invar(1, Not(c))
    automata[2].set_assume(0, TRUE())
    automata[2].set_assume(1, TRUE())
    automata[2].set_transitions(0, [(1, [Not(x_c)])])
    automata[2].set_transitions(1, [(0, [x_c])])

    automata[3] = AGAutomaton(symbols, [a], "Aut_A1", 2)
    automata[3].set_invar(0, GT(a, zero_i))
    automata[3].set_assume(0, GT(b, zero_r))
    automata[3].set_invar(1, LT(a, zero_i))
    automata[3].set_assume(1, GT(b, zero_r))
    automata[3].set_transitions(0, [(1, [LE(ToReal(x_a),
                                            Times(m_one_r, b, ToReal(a)))])])
    automata[3].set_transitions(1, [(0, [GE(ToReal(x_a),
                                            Times(m_one_r, b, ToReal(a)))])])

    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file,
                                    trace_file, cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
