from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Or, Implies
from pysmt.shortcuts import GT, GE, Equals, LE, LT
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/power.smv"""
    # symbols.
    e = Symbol("e", INT)
    i = Symbol("i", INT)
    n = Symbol("n", INT)
    pc = Symbol("pc", INT)
    x_e = Symbol(symb_next("e"), INT)
    x_i = Symbol(symb_next("i"), INT)
    x_n = Symbol(symb_next("n"), INT)
    x_pc = Symbol(symb_next("pc"), INT)
    symbols = [e, i, n, pc]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(n >= 2) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(n, Int(2)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & n >= 2 : 1,
        Implies(And(Equals(pc, Int(0)), GE(n, Int(2))), Equals(x_pc, Int(1))),
        # pc = 1 : 2,
        Implies(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
        # pc = 2 : 3,
        Implies(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
        # pc = 3 & !(i <= e) : -1,
        Implies(And(Equals(pc, Int(3)), Not(LE(i, e))),
                Equals(x_pc, Int(-1))),
        # pc = 3 & i <= e : 4,
        Implies(And(Equals(pc, Int(3)), LE(i, e)),
                Equals(x_pc, Int(4))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 : 3,
        Implies(Equals(pc, Int(5)), Equals(x_pc, Int(3)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (n' = n & i' = i & e' = e),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, e))),
        # (pc = 0 & pc' = -1) -> (n' = n & i' = i & e' = e),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, e))),
        # (pc = 0 & pc' = 1)  -> (n' = n & i' = i & e' = e),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, e))),
        # (pc = 1 & pc' = 2)  -> (n' = n & i' = 1 & e' = e),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_n, n), Equals(x_i, Int(1)), Equals(x_e, e))),
        # (pc = 2 & pc' = 3)  -> (n' = n & i' = i & e' = 1),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, Int(1)))),
        # (pc = 3 & pc' = -1) -> (n' = n & i' = i & e' = e),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, e))),
        # (pc = 3 & pc' = 4)  -> (n' = n & i' = i & e' = e),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, e))),
        # (pc = 4 & pc' = 5)  -> (n' = n & i' = i & e' = e*n),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_e, Times(e, n)))),
        # (pc = 5 & pc' = 3)  -> (n' = n & i' = i+1 & e' = e),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_i, Plus(i, Int(1))),
                    Equals(x_e, e)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_e = AGAutomaton(symbols, [e], "aut_e", 1)
    aut_e.set_assume(0, TRUE())
    aut_e.set_invar(0, TRUE())
    aut_e.set_transitions(0, [(0, [Equals(x_e, e),
                                   Equals(x_e, Times(e, n))])])

    aut_i = AGAutomaton(symbols, [i], "aut_i", 1)
    aut_i.set_assume(0, TRUE())
    aut_i.set_invar(0, TRUE())
    aut_i.set_transitions(0, [(0, [Equals(x_i, i),
                                   Equals(x_i, Plus(i, Int(1)))])])

    aut_ei = AGAutomaton(symbols, [e, i], "aut_ei", 2)
    aut_ei.set_assume(0, GE(n, Int(2)))
    aut_ei.set_invar(0, And(GE(e, i), GT(e, Int(1))))
    aut_ei.set_transitions(0, [(0, [And(Equals(x_e, e), Equals(x_i, i))]),
                               (1, [And(Equals(x_e, Times(e, n)),
                                        Equals(x_i, i))])])
    aut_ei.set_assume(1, GE(n, Int(2)))
    aut_ei.set_invar(1, And(GT(e, i), GT(e, Int(1))))
    aut_ei.set_transitions(1, [(1, [And(Equals(x_e, e), Equals(x_i, i))]),
                               (0, [And(Equals(x_e, e),
                                        Equals(x_i, Plus(i, Int(1))))])])

    aut_n = AGAutomaton(symbols, [n], "aut_n", 1)
    aut_n.set_assume(0, TRUE())
    aut_n.set_invar(0, TRUE())
    aut_n.set_transitions(0, [(0, [Equals(x_n, n)])])

    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 3)
        n_pc = Int(n_loc + 3)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    automata = [aut_e, aut_i, aut_n, aut_pc, aut_ei]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
