from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GT, GE, Equals
from pysmt.shortcuts import Plus, Div
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/integer_log.smv"""
    # symbols.
    d = Symbol("d", INT)
    log = Symbol("log", INT)
    n = Symbol("n", INT)
    pc = Symbol("pc", INT)
    x_d = Symbol(symb_next("d"), INT)
    x_log = Symbol(symb_next("log"), INT)
    x_n = Symbol(symb_next("n"), INT)
    x_pc = Symbol(symb_next("pc"), INT)
    symbols = [d, log, n, pc]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(n >= 1) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(n, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & n >= 1 : 1,
        Implies(And(Equals(pc, Int(0)), GE(n, Int(1))), Equals(x_pc, Int(1))),
        # pc = 1 & !(d >= 2) : -1,
        Implies(And(Equals(pc, Int(1)), Not(GE(d, Int(2)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & d >= 2 : 2,
        Implies(And(Equals(pc, Int(1)), GE(d, Int(2))), Equals(x_pc, Int(2))),
        # pc = 2 : 3,
        Implies(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
        # pc = 3 & !(n >= 0) : -1,
        Implies(And(Equals(pc, Int(3)), Not(GE(n, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 3 & n >= 0 : 4,
        Implies(And(Equals(pc, Int(3)), GE(n, Int(0))), Equals(x_pc, Int(4))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 : 3,
        Implies(Equals(pc, Int(5)), Equals(x_pc, Int(3)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 0 & pc' = -1) -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 0 & pc' = 1)  -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 1 & pc' = -1) -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 1 & pc' = 2)  -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 2 & pc' = 3)  -> (n' = n & d' = d & log' = 0),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, Int(0)))),
        # (pc = 3 & pc' = -1) -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 3 & pc' = 4)  -> (n' = n & d' = d & log' = log),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_n, n), Equals(x_d, d), Equals(x_log, log))),
        # (pc = 4 & pc' = 5)  -> (n' = n/d & d' = d & log' = log),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_n, Div(n, d)), Equals(x_d, d),
                    Equals(x_log, log))),
        # (pc = 5 & pc' = 3)  -> (n' = n & d' = d & log' = log+1),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_d, d),
                    Equals(x_log, Plus(log, Int(1)))))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_d = AGAutomaton(symbols, [d], "aut_d", 1)
    aut_d.set_assume(0, TRUE())
    aut_d.set_invar(0, TRUE())
    aut_d.set_transitions(0, [(0, [Equals(x_d, d)])])

    aut_log = AGAutomaton(symbols, [log], "aut_log", 1)
    aut_log.set_assume(0, TRUE())
    aut_log.set_invar(0, TRUE())
    aut_log.set_transitions(0, [(0, [Equals(x_log, log),
                                     Equals(x_log, Plus(log, Int(1)))])])

    aut_n = AGAutomaton(symbols, [n], "aut_n", 1)
    aut_n.set_assume(0, GT(d, Int(0)))
    aut_n.set_invar(0, GE(n, Int(0)))
    aut_n.set_transitions(0, [(0, [Equals(x_n, n),
                                   Equals(x_n, Div(n, d))])])

    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 3)
        n_pc = Int(n_loc + 3)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    automata = [aut_d, aut_log, aut_n, aut_pc]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
