from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GT, GE, Equals, LE
from pysmt.shortcuts import Plus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/factorial.smv"""
    # symbols.
    fact = Symbol("fact", INT)
    i = Symbol("i", INT)
    n = Symbol("n", INT)
    pc = Symbol("pc", INT)
    x_fact = Symbol(symb_next("fact"), INT)
    x_i = Symbol(symb_next("i"), INT)
    x_n = Symbol(symb_next("n"), INT)
    x_pc = Symbol(symb_next("pc"), INT)
    symbols = [fact, i, n, pc]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(n >= 1) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(n, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & n >= 1 : 1,
        Implies(And(Equals(pc, Int(0)), GE(n, Int(1))), Equals(x_pc, Int(1))),
        # pc = 1 : 2,
        Implies(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
        # pc = 2 : 3,
        Implies(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
        # pc = 3 & !(i <= fact) : -1,
        Implies(And(Equals(pc, Int(3)), Not(LE(i, fact))),
                Equals(x_pc, Int(-1))),
        # pc = 3 & i <= fact : 4,
        Implies(And(Equals(pc, Int(3)), LE(i, fact)), Equals(x_pc, Int(4))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 : 3,
        Implies(Equals(pc, Int(5)), Equals(x_pc, Int(3)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (n' = n & i' = i & fact' = fact),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, fact))),
        # (pc = 0 & pc' = -1) -> (n' = n & i' = i & fact' = fact),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, fact))),
        # (pc = 0 & pc' = 1)  -> (n' = n & i' = i & fact' = fact),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, fact))),
        # (pc = 1 & pc' = 2)  -> (n' = n & i' = i & fact' = 2),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, Int(2)))),
        # (pc = 2 & pc' = 3)  -> (n' = n & i' = 1 & fact' = fact),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_i, Int(1)),
                    Equals(x_fact, fact))),
        # (pc = 3 & pc' = -1) -> (n' = n & i' = i & fact' = fact),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(-1))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, fact))),
        # (pc = 3 & pc' = 4)  -> (n' = n & i' = i & fact' = fact),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_n, n), Equals(x_i, i), Equals(x_fact, fact))),
        # (pc = 4 & pc' = 5)  -> (n' = n & i' = i & fact' = fact*i),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_n, n), Equals(x_i, i),
                    Equals(x_fact, Times(fact, i)))),
        # (pc = 5 & pc' = 3)  -> (n' = n & i' = i+1 & fact' = fact),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(3))),
                And(Equals(x_n, n), Equals(x_i, Plus(i, Int(1))),
                    Equals(x_fact, fact)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_fact = AGAutomaton(symbols, [fact], "aut_fact", 1)
    aut_fact.set_assume(0, GE(i, Int(1)))
    aut_fact.set_invar(0, GT(fact, Int(1)))
    aut_fact.set_transitions(0, [(0, [Equals(x_fact, fact),
                                      Equals(x_fact, Times(fact, i))])])

    aut_i = AGAutomaton(symbols, [i], "aut_i", 1)
    aut_i.set_assume(0, TRUE())
    aut_i.set_invar(0, GE(i, Int(0)))
    aut_i.set_transitions(0, [(0, [Equals(x_i, i),
                                   Equals(x_i, Plus(i, Int(1)))])])

    aut_n = AGAutomaton(symbols, [n], "aut_n", 1)
    aut_n.set_assume(0, TRUE())
    aut_n.set_invar(0, TRUE())
    aut_n.set_transitions(0, [(0, [Equals(x_n, n)])])

    aut_facti = AGAutomaton(symbols, [fact, i], "aut_facti", 2)
    aut_facti.set_assume(0, TRUE())
    aut_facti.set_invar(0, And(GT(fact, Int(1)), GT(i, Int(1)), GE(fact, i)))
    aut_facti.set_transitions(0, [(0, [And(Equals(x_fact, fact),
                                           Equals(x_i, i))]),
                                  (1, [And(Equals(x_fact, Times(fact, i)),
                                           Equals(x_i, i))])])
    aut_facti.set_assume(1, TRUE())
    aut_facti.set_invar(1, And(GT(fact, Int(1)), GT(i, Int(1)), GT(fact, i)))
    aut_facti.set_transitions(1, [(1, [And(Equals(x_fact, fact),
                                           Equals(x_i, i)),
                                       And(Equals(x_fact, Times(fact, i)),
                                           Equals(x_i, i))]),
                                  (0, [And(Equals(x_fact, fact),
                                           Equals(x_i, Plus(i, Int(1))))])])

    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 3)
        n_pc = Int(n_loc + 3)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    automata = [aut_fact, aut_i, aut_n, aut_pc, aut_facti]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
