from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GE, Equals
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/8.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(z >= 3) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(z, Int(3)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & z >= 3 : 1,
        Implies(And(Equals(pc, Int(0)), GE(z, Int(3))), Equals(x_pc, Int(1))),
        # pc = 1 & !(z >= 3) : -1,
        Implies(And(Equals(pc, Int(1)), Not(GE(z, Int(3)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & z >= 3 : 2,
        Implies(And(Equals(pc, Int(1)), GE(z, Int(3))), Equals(x_pc, Int(2))),
        # pc = 2 : 1,
        Implies(Equals(pc, Int(2)), Equals(x_pc, Int(1)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_z, z))),
        # (pc = 0 & pc' = 1) -> (z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_z, z))),
        # (pc = 1 & pc' = -1) -> (z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_z, z))),
        # (pc = 1 & pc' = 2) -> (z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 2)
    aut_pc.set_assume(0, TRUE())
    aut_pc.set_invar(0, Equals(pc, Int(1)))
    aut_pc.set_transitions(0, [(1, [Equals(x_pc, Int(2))])])
    aut_pc.set_assume(1, TRUE())
    aut_pc.set_invar(1, Equals(pc, Int(2)))
    aut_pc.set_transitions(1, [(0, [Equals(x_pc, Int(1))])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, GE(z, Int(3)))
    aut_z.set_transitions(0, [(0, [Equals(x_z, z)])])

    automata = [aut_pc, aut_z]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
