from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GT, GE, Equals, LE
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/3.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(z >= 4) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(z, Int(4)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & z >= 4 : 1,
        Implies(And(Equals(pc, Int(0)), GE(z, Int(4))), Equals(x_pc, Int(1))),
        # pc = 1 : 2,
        Implies(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
        # pc = 2 & x >= 0 : 3,
        Implies(And(Equals(pc, Int(2)), GE(x, Int(0))), Equals(x_pc, Int(3))),
        # pc = 2 & !(x >= 0) : 4,
        Implies(And(Equals(pc, Int(2)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(4))),
        # pc = 3 : 5,
        Implies(Equals(pc, Int(3)), Equals(x_pc, Int(5))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 & !(y >= 1) : -1,
        Implies(And(Equals(pc, Int(5)), Not(GE(y, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 5 & y >= 1 : 6,
        Implies(And(Equals(pc, Int(5)), GE(y, Int(1))), Equals(x_pc, Int(6))),
        # pc = 6 & !(y <= 5) : -1,
        Implies(And(Equals(pc, Int(6)), Not(LE(y, Int(5)))),
                Equals(x_pc, Int(-1))),
        # pc = 6 & y <= 5 : 7,
        Implies(And(Equals(pc, Int(6)), LE(y, Int(5))), Equals(x_pc, Int(7))),
        # pc = 7 & !(x >= 0) : 10,
        Implies(And(Equals(pc, Int(7)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(10))),
        # pc = 7 & x >= 0 : 8,
        Implies(And(Equals(pc, Int(7)), GE(x, Int(0))), Equals(x_pc, Int(8))),
        # pc = 8 : 9,
        Implies(Equals(pc, Int(8)), Equals(x_pc, Int(9))),
        # pc = 9 : 7,
        Implies(Equals(pc, Int(9)), Equals(x_pc, Int(7))),
        # pc = 10 : -1,
        Implies(Equals(pc, Int(10)), Equals(x_pc, Int(-1)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Plus(z, Int(1))))),
        # (pc = 2 & pc' = 3)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 4)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(4))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 5)  -> (x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Plus(z, Int(1))))),
        # (pc = 4 & pc' = 5)  -> (x' = x & y' = y & z' = z-1),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Minus(z, Int(1))))),
        # (pc = 5 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = 6)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 6 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 6 & pc' = 7)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 7 & pc' = 10) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(7)), Equals(x_pc, Int(10))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 7 & pc' = 8)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(7)), Equals(x_pc, Int(8))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 8 & pc' = 9)  -> (x' = z*z - y*z & y' = y & z' = z),
        Implies(And(Equals(pc, Int(8)), Equals(x_pc, Int(9))),
                And(Equals(x_x, Minus(Times(z, z), Times(y, z))),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 9 & pc' = 7)  -> (x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(9)), Equals(x_pc, Int(7))),
                And(Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Plus(z, Int(1))))),
        # (pc = 10 & pc' = -1) -> (x' = x & y' = y & z' = z-1),
        Implies(And(Equals(pc, Int(10)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Minus(z, Int(1)))))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(3):
        n_loc = (loc + 1) % 3
        _pc = loc + 7
        n_pc = n_loc + 7
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, Int(_pc)))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, Int(n_pc))])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, And(GT(z, y), GE(z, Int(0))))
    aut_x.set_invar(0, GE(x, Int(0)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(Times(z, z),
                                                     Times(y, z)))])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, TRUE())
    aut_y.set_transitions(0, [(0, [Equals(x_y, y)])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, GE(z, Int(0)))
    aut_z.set_transitions(0, [(0, [Equals(x_z, z),
                                   Equals(x_z, Plus(z, Int(1)))])])

    automata = [aut_pc, aut_x, aut_y, aut_z]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
