from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Or, Implies
from pysmt.shortcuts import GE, Equals, LE, LT
from pysmt.shortcuts import Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/15.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(x >= -1) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(x, Int(-1)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & x >= -1 : 1,
        Implies(And(Equals(pc, Int(0)), GE(x, Int(-1))), Equals(x_pc, Int(1))),
        # pc = 1 & !(y <= -10) : -1,
        Implies(And(Equals(pc, Int(1)), Not(LE(y, Int(-10)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & y <= -10 : 2,
        Implies(And(Equals(pc, Int(1)), LE(y, Int(-10))),
                Equals(x_pc, Int(2))),
        # pc = 2 & !(x >= 1 & y <= -20) : -1,
        Implies(And(Equals(pc, Int(2)),
                    Not(And(GE(x, Int(1)), LE(y, Int(-20))))),
                Equals(x_pc, Int(-1))),
        # pc = 2 & x >= 1 & y <= -20 : 3,
        Implies(And(Equals(pc, Int(2)), And(GE(x, Int(1)), LE(y, Int(-20)))),
                Equals(x_pc, Int(3))),
        # pc = 3 : {4, 5},
        Implies(Equals(pc, Int(3)),
                Or(Equals(x_pc, Int(4)), Equals(x_pc, Int(5)))),
        # pc = 4 & !(x < 0) : -1,
        Implies(And(Equals(pc, Int(4)), Not(LT(x, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 4 & x < 0 : 5,
        Implies(And(Equals(pc, Int(4)), LT(x, Int(0))), Equals(x_pc, Int(5))),
        # pc = 5 : 6,
        Implies(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
        # pc = 6 : 7,
        Implies(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
        # pc = 7 : 2,
        Implies(Equals(pc, Int(7)), Equals(x_pc, Int(2)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 3)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 4)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 5)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 4 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 4 & pc' = 5)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = 6)  -> (x' = x & y' = y & z' = x*y),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, Times(x, y)))),
        # (pc = 6 & pc' = 7)  -> (x' = x - 2*y & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
                And(Equals(x_x, Minus(x, Times(Int(2), y))),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 7 & pc' = 2)  -> (x' = x & y' = y-1 & z' = z),
        Implies(And(Equals(pc, Int(7)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, Minus(y, Int(1))),
                    Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    loc2pc = [Int(2), Int(3), Int(5), Int(6), Int(7)]
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", len(loc2pc))
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = loc2pc[loc]
        n_pc = loc2pc[n_loc]
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, LE(y, Int(0)))
    aut_x.set_invar(0, GE(x, Int(1)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(x, Times(Int(2), y)))])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, LE(y, Int(-20)))
    aut_y.set_transitions(0, [(0, [Equals(x_y, y),
                                   Equals(x_y, Minus(y, Int(1)))])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, And(GE(x, Int(0)), LE(y, Int(0))))
    aut_z.set_invar(0, LE(z, Int(0)))
    aut_z.set_transitions(0, [(0, [Equals(x_z, z),
                                   Equals(x_z, Times(x, y))])])

    automata = [aut_pc, aut_x, aut_y, aut_z]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
