from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Or, Implies
from pysmt.shortcuts import GT, GE, Equals, LE, LT
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/12.smv"""
    # symbols.
    a = Symbol("a", INT)
    b = Symbol("b", INT)
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_a = Symbol(symb_next("a"), INT)
    x_b = Symbol(symb_next("b"), INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [a, b, pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(z >= 4) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(z, Int(4)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & z >= 4 : 1,
        Implies(And(Equals(pc, Int(0)), GE(z, Int(4))), Equals(x_pc, Int(1))),
        # pc = 1 : 2,
        Implies(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
        # pc = 2 & x >= 0 : 3,
        Implies(And(Equals(pc, Int(2)), GE(x, Int(0))), Equals(x_pc, Int(3))),
        # pc = 2 & !(x >= 0) : 4,
        Implies(And(Equals(pc, Int(2)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(4))),
        # pc = 3 : 5,
        Implies(Equals(pc, Int(3)), Equals(x_pc, Int(5))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 & !(y >= 2) : -1,
        Implies(And(Equals(pc, Int(5)), Not(GE(y, Int(2)))),
                Equals(x_pc, Int(-1))),
        # pc = 5 & y >= 2 : 6,
        Implies(And(Equals(pc, Int(5)), GE(y, Int(2))), Equals(x_pc, Int(6))),
        # pc = 6 & !(x >= 0) : -1,
        Implies(And(Equals(pc, Int(6)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 6 & x >= 0 : 7,
        Implies(And(Equals(pc, Int(6)), GE(x, Int(0))), Equals(x_pc, Int(7))),
        # pc = 7 : 8,
        Implies(Equals(pc, Int(7)), Equals(x_pc, Int(8))),
        # pc = 8 : 9,
        Implies(Equals(pc, Int(8)), Equals(x_pc, Int(9))),
        # pc = 9 : 6,
        Implies(Equals(pc, Int(9)), Equals(x_pc, Int(6)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (a' = a & b' = b & x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, Plus(z, Int(1))))),
        # (pc = 2 & pc' = 3)  -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 4)  -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(4))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 5)  -> (a' = a & b' = b & x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(5))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, Plus(z, Int(1))))),
        # (pc = 4 & pc' = 5)  -> (a' = a & b' = b & x' = x & y' = y & z' = z-1),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, Minus(z, Int(1))))),
        # (pc = 5 & pc' = -1) -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(-1))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = 6)  -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 6 & pc' = -1) -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(-1))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 6 & pc' = 7)  -> (a' = a & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 7 & pc' = 8)  -> (a' = z*z & b' = b & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(7)), Equals(x_pc, Int(8))),
                And(Equals(x_a, Times(z, z)), Equals(x_b, b), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 8 & pc' = 9)  -> (a' = a & b' = y*z & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(8)), Equals(x_pc, Int(9))),
                And(Equals(x_a, a), Equals(x_b, Times(y, z)), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 9 & pc' = 6)  -> (a' = a & b' = b & x' = a-b & y' = y & z' = z),
        Implies(And(Equals(pc, Int(9)), Equals(x_pc, Int(6))),
                And(Equals(x_a, a), Equals(x_b, b), Equals(x_x, Minus(a, b)),
                    Equals(x_y, y), Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_a = AGAutomaton(symbols, [a], "aut_a", 1)
    aut_a.set_assume(0, TRUE())
    aut_a.set_invar(0, GE(a, Int(0)))
    aut_a.set_transitions(0, [(0, [Equals(x_a, a),
                                   Equals(x_a, Times(z, z))])])

    aut_b = AGAutomaton(symbols, [b], "aut_b", 1)
    aut_b.set_assume(0, TRUE())
    aut_b.set_invar(0, TRUE())
    aut_b.set_transitions(0, [(0, [Equals(x_b, b),
                                   Equals(x_b, Times(y, z))])])

    aut_ab = AGAutomaton(symbols, [a, b], "aut_ab", 1)
    aut_ab.set_assume(0, And(GE(z, Int(4)), LT(y, Int(4))))
    aut_ab.set_invar(0, And(GE(a, b), GE(a, Int(0)), LE(b, Times(Int(4), z))))
    aut_ab.set_transitions(0, [(0, [And(Equals(x_a, a), Equals(x_b, b)),
                                    And(Equals(x_a, Times(z, z)),
                                        Equals(x_b, b)),
                                    And(Equals(x_a, a),
                                        Equals(x_b, Times(y, z))),
                                    And(Equals(x_a, Times(z, z)),
                                        Equals(x_b, Times(y, z)))])])

    aut_abyz = AGAutomaton(symbols, [a, b, y, z], "aut_abyz", 1)
    aut_abyz.set_assume(0, TRUE())
    aut_abyz.set_invar(0, And(Equals(a, Times(z, z)), Equals(b, Times(y, z)),
                              GE(z, y), GE(z, Int(0))))
    aut_abyz.set_transitions(0, [(0, [And(Equals(x_a, a), Equals(x_b, b),
                                          Equals(x_y, y), Equals(x_z, z)),
                                      And(Equals(x_a, Times(z, z)),
                                          Equals(x_b, b),
                                          Equals(x_y, y), Equals(x_z, z)),
                                      And(Equals(x_a, a),
                                          Equals(x_b, Times(y, z)),
                                          Equals(x_y, y), Equals(x_z, z))])])

    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 4)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 6)
        n_pc = Int(n_loc + 6)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, And(GE(a, b), GE(a, Int(0))))
    aut_x.set_invar(0, GE(x, Int(0)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(a, b))])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, TRUE())
    aut_y.set_transitions(0, [(0, [Equals(x_y, y)])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, TRUE())
    aut_z.set_transitions(0, [(0, [Equals(x_z, z)])])

    automata = [aut_ab, aut_pc, aut_x, aut_y, aut_z, aut_abyz]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
