from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Or, Implies
from pysmt.shortcuts import GT, GE, Equals, LE, LT
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/11.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    w = Symbol("w", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_w = Symbol(symb_next("w"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, w, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(z >= 5) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(z, Int(5)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & z >= 5 : 1,
        Implies(And(Equals(pc, Int(0)), GE(z, Int(5))), Equals(x_pc, Int(1))),
        # pc = 1 & !(y <= 2) : -1,
        Implies(And(Equals(pc, Int(1)), Not(LE(y, Int(2)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & y <= 2 : 2,
        Implies(And(Equals(pc, Int(1)), LE(y, Int(2))), Equals(x_pc, Int(2))),
        # pc = 2 & !(w <= -5) : -1,
        Implies(And(Equals(pc, Int(2)), Not(LE(w, Int(-5)))),
                Equals(x_pc, Int(-1))),
        # pc = 2 & w <= -5 : 3,
        Implies(And(Equals(pc, Int(2)), LE(w, Int(-5))), Equals(x_pc, Int(3))),
        # pc = 3 & !(x >= y) : -1,
        Implies(And(Equals(pc, Int(3)), Not(GE(x, y))),
                Equals(x_pc, Int(-1))),
        # pc = 3 & (x >= y) : 4,
        Implies(And(Equals(pc, Int(3)), GE(x, y)),
                Equals(x_pc, Int(4))),
        # pc = 4 : 5,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
        # pc = 5 : 6,
        Implies(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
        # pc = 6 : 3,
        Implies(Equals(pc, Int(6)), Equals(x_pc, Int(3)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 1 & pc' = -1) -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 2 & pc' = -1) -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(-1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 2 & pc' = 3)  -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 3 & pc' = -1) -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(-1))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 3 & pc' = 4)  -> (w' = w & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 4 & pc' = 5)  -> (w' = w & x' = -z*w & y' = y & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_w, w), Equals(x_x, Times(Int(-1), z, w)),
                    Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = 6)  -> (w' = w & x' = x & y' = y & z' = z+1),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
                And(Equals(x_w, w), Equals(x_x, x), Equals(x_y, y),
                    Equals(x_z, Plus(z, Int(1))))),
        # (pc = 6 & pc' = 3)  -> (w' = w-1 & x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(3))),
                And(Equals(x_w, Minus(w, Int(1))), Equals(x_x, x),
                    Equals(x_y, y), Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 4)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 3)
        n_pc = Int(n_loc + 3)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_w = AGAutomaton(symbols, [w], "aut_w", 1)
    aut_w.set_assume(0, TRUE())
    aut_w.set_invar(0, LE(w, Int(0)))
    aut_w.set_transitions(0, [(0, [Equals(x_w, w),
                                   Equals(x_w, Minus(w, Int(1)))])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, And(LT(y, Int(0)), LE(w, Int(0)), GE(z, Int(0))))
    aut_x.set_invar(0, GE(x, Int(0)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Times(Int(-1), z, w))])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, LT(y, Int(0)))
    aut_y.set_transitions(0, [(0, [Equals(x_y, y)])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, GE(z, Int(0)))
    aut_z.set_transitions(0, [(0, [Equals(x_z, z),
                                   Equals(x_z, Plus(z, Int(1)))])])

    automata = [aut_pc, aut_w, aut_x, aut_y, aut_z]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
