from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GE, Equals, LE
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/1.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 : 1,
        Implies(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
        # pc = 1 & !(x >= y) : -1,
        Implies(And(Equals(pc, Int(1)), Not(GE(x, y))), Equals(x_pc, Int(-1))),
        # pc = 1 & (x >= y) : 2,
        Implies(And(Equals(pc, Int(1)), GE(x, y)), Equals(x_pc, Int(2))),
        # pc = 2 : 3,
        Implies(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
        # pc = 3 : 4,
        Implies(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
        # pc = 4 & !(z >= 0) : -1,
        Implies(And(Equals(pc, Int(4)), Not(GE(z, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 4 & z >= 0 : 5,
        Implies(And(Equals(pc, Int(4)), GE(z, Int(0))), Equals(x_pc, Int(5))),
        # pc = 5 & !(x <= y - 1) : -1,
        Implies(And(Equals(pc, Int(5)), Not(LE(x, Minus(y, Int(1))))),
                Equals(x_pc, Int(-1))),
        # pc = 5 & x <= y - 1 : 6,
        Implies(And(Equals(pc, Int(5)), LE(x, Minus(y, Int(1)))),
                Equals(x_pc, Int(6))),
        # pc = 6 : 7,
        Implies(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
        # pc = 7 : 8,
        Implies(Equals(pc, Int(7)), Equals(x_pc, Int(8))),
        # pc = 8 : 5,
        Implies(Equals(pc, Int(8)), Equals(x_pc, Int(5)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (x' = x & y' = y & z' = 2),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, Int(2)))),
        # (pc = 1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 3)  -> (x' = 5 & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_x, Int(5)), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 4)  -> (x' = x & y' = 6 & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_x, x), Equals(x_y, Int(6)), Equals(x_z, z))),
        # (pc = 4 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 4 & pc' = 5)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 5 & pc' = 6)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(5)), Equals(x_pc, Int(6))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 6 & pc' = 7)  -> (x' = x & y' = z*z & z' = z),
        Implies(And(Equals(pc, Int(6)), Equals(x_pc, Int(7))),
                And(Equals(x_x, x), Equals(x_y, Times(z, z)), Equals(x_z, z))),
        # (pc = 7 & pc' = 8)  -> (x' = y & y' = y & z' = z),
        Implies(And(Equals(pc, Int(7)), Equals(x_pc, Int(8))),
                And(Equals(x_x, y), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 8 & pc' = 5)  -> (x' = x & y' = y+1 & z' = z),
        Implies(And(Equals(pc, Int(8)), Equals(x_pc, Int(5))),
                And(Equals(x_x, x), Equals(x_y, Plus(y, Int(1))),
                    Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 4)
    for l in range(4):
        n_l = (l + 1) % 4
        _pc = l + 5
        n_pc = n_l + 5
        aut_pc.set_assume(l, TRUE())
        aut_pc.set_invar(l, Equals(pc, Int(_pc)))
        aut_pc.set_transitions(l, [(n_l, [Equals(x_pc, Int(n_pc))])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 2)
    aut_x.set_invar(0, Equals(x, Int(4)))
    aut_x.set_assume(0, Equals(y, Int(5)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x)]), (1, [Equals(x_x, x)])])
    aut_x.set_invar(1, Equals(x, Int(4)))
    aut_x.set_assume(1, Equals(y, Int(4)))
    aut_x.set_transitions(1, [(0, [Equals(x_x, y)]), (1, [Equals(x_x, y)])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 2)
    aut_y.set_invar(0, Equals(y, Int(4)))
    aut_y.set_assume(0, Equals(z, Int(2)))
    aut_y.set_transitions(0, [(0, [Equals(x_y, y), Equals(x_y, Times(z, z))]),
                              (1, [Equals(x_y, Plus(y, Int(1)))])])
    aut_y.set_invar(1, Equals(y, Int(5)))
    aut_y.set_assume(1, Equals(z, Int(2)))
    aut_y.set_transitions(1, [(1, [Equals(x_y, y)]),
                              (0, [Equals(x_y, Times(z, z))])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_invar(0, Equals(z, Int(2)))
    aut_z.set_assume(0, TRUE())
    aut_z.set_transitions(0, [(0, [Equals(x_z, z), Equals(x_z, Int(2))])])

    automata = [aut_pc, aut_x, aut_y, aut_z]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
