from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GT, GE, Equals, LT, LE
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/19.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(y >= 1) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(y, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & y >= 1 : 1,
        Implies(And(Equals(pc, Int(0)), GE(y, Int(1))), Equals(x_pc, Int(1))),
        # pc = 1 & !(z >= 1) : -1,
        Implies(And(Equals(pc, Int(1)), Not(GE(z, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & z >= 1 : 2,
        Implies(And(Equals(pc, Int(1)), GE(z, Int(1))), Equals(x_pc, Int(2))),
        # pc = 2 & !(x >= 0) : -1,
        Implies(And(Equals(pc, Int(2)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 2 & x >= 0 : 3,
        Implies(And(Equals(pc, Int(2)), GE(x, Int(0))), Equals(x_pc, Int(3))),
        # pc = 3 : 4,
        Implies(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
        # pc = 4 : 2,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(2)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 3)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 4)  -> (x' = y*z - 1 & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_x, Minus(Times(y, z), Int(1))), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 4 & pc' = 2)  -> (x' = x & y' = y+1 & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, Plus(y, Int(1))),
                    Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc1 = AGAutomaton(symbols, [pc], "aut_pc1", 3)
    for loc in range(aut_pc1.num_locations):
        n_loc = (loc + 1) % aut_pc1.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc1.set_assume(loc, LT(x, Int(loc)))
        aut_pc1.set_invar(loc, Equals(pc, c_pc))
        aut_pc1.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc2 = AGAutomaton(symbols, [pc], "aut_pc2", 1)
    aut_pc2.set_transitions(0, [(0, [TRUE()])])

    aut_pc3 = AGAutomaton(symbols, [pc], "aut_pc3", 3)
    for loc in range(aut_pc3.num_locations):
        n_loc = (loc + 1) % aut_pc3.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc3.set_assume(loc, LE(y, Int(0)))
        aut_pc3.set_invar(loc, Equals(pc, c_pc))
        aut_pc3.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc4 = AGAutomaton(symbols, [pc], "aut_pc4", 3)
    for loc in range(aut_pc4.num_locations):
        n_loc = (loc + 2) % aut_pc4.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc4.set_assume(loc, TRUE())
        aut_pc4.set_invar(loc, Equals(pc, c_pc))
        aut_pc4.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc5 = AGAutomaton(symbols, [pc], "aut_pc5", 4)
    for loc in range(aut_pc5.num_locations):
        n_loc = (loc + 1) % aut_pc5.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc5.set_assume(loc, TRUE())
        aut_pc5.set_invar(loc, Equals(pc, c_pc))
        aut_pc5.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc6 = AGAutomaton(symbols, [pc], "aut_pc6", 4)
    for loc in range(aut_pc6.num_locations):
        n_loc = (loc + 1) % aut_pc6.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc6.set_assume(loc, Equals(y, c_pc))
        aut_pc6.set_invar(loc, Equals(pc, c_pc))
        aut_pc6.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc7 = AGAutomaton(symbols, [pc], "aut_pc7", 3)
    for loc in range(aut_pc7.num_locations):
        n_loc = (loc + 1) % aut_pc7.num_locations
        c_pc = Int(loc + 2)
        aut_pc7.set_assume(loc, GE(y, Int(0)))
        aut_pc7.set_invar(loc, GE(pc, Int(0)))
        aut_pc7.set_transitions(loc, [(n_loc, [Equals(x_pc, y)])])

    aut_pc8 = AGAutomaton(symbols, [pc], "aut_pc8", 3)
    for loc in range(aut_pc8.num_locations):
        n_loc = (loc + 1) % aut_pc8.num_locations
        c_pc = Int(loc + 2)
        aut_pc8.set_assume(loc, And(GE(Times(x, y), Int(2)),
                                    LE(Times(x, y), Int(5))))
        aut_pc8.set_invar(loc, And(GE(pc, Int(2)), LE(pc, Int(5))))
        aut_pc8.set_transitions(loc, [(n_loc, [Equals(x_pc, Times(x, y))])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, And(GT(y, Int(0)), GT(z, Int(0))))
    aut_x.set_invar(0, GE(x, Int(0)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x1 = AGAutomaton(symbols, [x], "aut_x1", 1)
    aut_x1.set_invar(0, GE(x, Int(0)))
    aut_x1.set_transitions(0, [(0, [Equals(x_x, Plus(x, Int(1)))])])

    aut_x2 = AGAutomaton(symbols, [x], "aut_x2", 1)
    aut_x2.set_invar(0, GE(x, Int(0)))
    aut_x2.set_assume(0, GE(y, Int(0)))
    aut_x2.set_transitions(0, [(0, [Equals(x_x, Plus(x, y))])])

    aut_x3 = AGAutomaton(symbols, [x], "aut_x3", 1)
    aut_x3.set_assume(0, And(GT(y, Int(0)), LT(z, Int(0))))
    aut_x3.set_invar(0, LE(x, Int(0)))
    aut_x3.set_transitions(0, [(0, [Equals(x_x, x),
                                    Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x4 = AGAutomaton(symbols, [x], "aut_x4", 1)
    aut_x4.set_assume(0, And(GT(y, Int(0)), Equals(z, Int(1))))
    aut_x4.set_invar(0, GE(x, Int(0)))
    aut_x4.set_transitions(0, [(0, [Equals(x_x, Plus(Times(x, z), y)),
                                    Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x5 = AGAutomaton(symbols, [x], "aut_x5", 2)
    aut_x5.set_invar(0, GE(x, Int(0)))
    aut_x5.set_transitions(0, [(1, [Equals(x_x, x)])])
    aut_x5.set_assume(1, And(GT(y, Int(0)), GT(z, Int(0))))
    aut_x5.set_invar(1, GE(x, Int(0)))
    aut_x5.set_transitions(1, [(0, [Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x6 = AGAutomaton(symbols, [x], "aut_x6", 2)
    aut_x6.set_invar(0, GE(x, Int(0)))
    aut_x6.set_transitions(0, [(1, [Equals(x_x, x)])])
    aut_x6.set_assume(1, And(LT(y, Int(0)), LT(z, Int(0))))
    aut_x6.set_invar(1, GE(x, Int(0)))
    aut_x6.set_transitions(1, [(0, [Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x7 = AGAutomaton(symbols, [x], "aut_x7", 2)
    aut_x7.set_assume(0, And(GT(y, Int(0)), GT(z, Int(0))))
    aut_x7.set_invar(0, GE(x, Int(0)))
    aut_x7.set_transitions(0, [(0, [Equals(x_x, Minus(Times(y, z), Int(1)))]),
                               (1, [Equals(x_x, x)])])
    aut_x7.set_assume(1, And(LT(y, Int(0)), LT(z, Int(0))))
    aut_x7.set_invar(1, GE(x, Int(0)))
    aut_x7.set_transitions(1, [(0, [Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_x8 = AGAutomaton(symbols, [x], "aut_x8", 1)
    aut_x8.set_assume(0, And(GT(y, Int(0)), GT(z, Int(0)),
                             LT(pc, Int(3)), GE(pc, Int(2))))
    aut_x8.set_invar(0, GE(x, Int(0)))
    aut_x8.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(Times(y, z), Int(1)))])])


    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, GE(y, Int(0)))
    aut_y.set_transitions(0, [(0, [Equals(x_y, y),
                                   Equals(x_y, Plus(y, Int(1)))])])

    aut_y1 = AGAutomaton(symbols, [y], "aut_y1", 1)
    aut_y1.set_invar(0, GE(y, Int(0)))
    aut_y1.set_transitions(0, [(0, [Equals(x_y, Plus(y, Int(1)))])])

    aut_y2 = AGAutomaton(symbols, [y], "aut_y2", 1)
    aut_y2.set_assume(0, GE(x, Int(0)))
    aut_y2.set_invar(0, GE(y, Int(0)))
    aut_y2.set_transitions(0, [(0, [Equals(x_y, Plus(y, Int(1))),
                                    Equals(x_y, Plus(y, x))])])

    aut_y3 = AGAutomaton(symbols, [y], "aut_y3", 1)
    aut_y3.set_assume(0, GE(x, Int(0)))
    aut_y3.set_invar(0, GE(y, Int(0)))
    aut_y3.set_transitions(0, [(0, [Equals(x_y, y),
                                    Equals(x_y, Plus(y, x))])])

    aut_y4 = AGAutomaton(symbols, [y], "aut_y4", 2)
    aut_y4.set_assume(0, TRUE())
    aut_y4.set_invar(0, GE(y, Int(0)))
    aut_y4.set_transitions(0, [(0, [Equals(x_y, y)]),
                               (1, [Equals(x_y, y)])])
    aut_y4.set_assume(1, LE(Times(x, y), Int(0)))
    aut_y4.set_invar(1, GE(y, Int(0)))
    aut_y4.set_transitions(1, [(0, [Equals(x_y, Plus(y, Int(1)))]),
                               (1, [Equals(x_y, Plus(y, Int(1)))])])

    aut_y5 = AGAutomaton(symbols, [y], "aut_y5", 2)
    aut_y5.set_assume(0, TRUE())
    aut_y5.set_invar(0, GE(y, Int(0)))
    aut_y5.set_transitions(0, [(0, [Equals(x_y, y)]),
                               (1, [Equals(x_y, y)])])
    aut_y5.set_assume(1, LE(Times(x, y), Int(0)))
    aut_y5.set_invar(1, GE(y, Int(0)))
    aut_y5.set_transitions(1, [(0, [Equals(x_y, Plus(y, Int(2)))]),
                               (1, [Equals(x_y, Plus(y, Int(1)))])])

    aut_y6 = AGAutomaton(symbols, [y], "aut_y6", 2)
    aut_y6.set_assume(0, TRUE())
    aut_y6.set_invar(0, GE(y, Int(0)))
    aut_y6.set_transitions(0, [(0, [Equals(x_y, y)]),
                               (1, [Equals(x_y, y)])])
    aut_y6.set_assume(1, LE(Times(x, y), Int(10)))
    aut_y6.set_invar(1, GE(y, Int(0)))
    aut_y6.set_transitions(1, [(0, [Equals(x_y, Plus(y, Int(1)))]),
                               (1, [Equals(x_y, Plus(y, Int(1)))])])

    aut_y7 = AGAutomaton(symbols, [y], "aut_y7", 2)
    aut_y7.set_assume(0, TRUE())
    aut_y7.set_invar(0, GE(y, Int(0)))
    aut_y7.set_transitions(0, [(0, [Equals(x_y, y)]),
                               (1, [Equals(x_y, y)])])
    aut_y7.set_assume(1, GE(Times(x, y), Int(10)))
    aut_y7.set_invar(1, GE(y, Int(0)))
    aut_y7.set_transitions(1, [(0, [Equals(x_y, Plus(y, Int(1)))]),
                               (1, [Equals(x_y, Plus(y, Int(1)))])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, TRUE())
    aut_z.set_transitions(0, [(0, [Equals(x_z, z)])])

    aut_z1 = AGAutomaton(symbols, [z], "aut_z1", 1)
    aut_z1.set_assume(0, TRUE())
    aut_z1.set_invar(0, GE(z, Int(0)))
    aut_z1.set_transitions(0, [(0, [Equals(x_z, Plus(z, Int(1)))])])

    aut_z2 = AGAutomaton(symbols, [z], "aut_z2", 1)
    aut_z2.set_assume(0, LE(x, Int(0)))
    aut_z2.set_transitions(0, [(0, [Equals(x_z, z)])])

    aut_z3 = AGAutomaton(symbols, [z], "aut_z3", 1)
    aut_z3.set_assume(0, Equals(Times(x, y), Int(0)))
    aut_z3.set_transitions(0, [(0, [Equals(x_z, Plus(z, Times(x, y)))])])

    aut_z4 = AGAutomaton(symbols, [z], "aut_z4", 2)
    aut_z4.set_assume(0, TRUE())
    aut_z4.set_invar(0, TRUE())
    aut_z4.set_transitions(0, [(1, [Equals(x_z, z)])])
    aut_z4.set_assume(1, Equals(x, Int(1)))
    aut_z4.set_invar(1, TRUE())
    aut_z4.set_transitions(1, [(0, [Equals(x_z, Times(x, z))])])

    aut_z5 = AGAutomaton(symbols, [z], "aut_z5", 1)
    aut_z5.set_assume(0, LT(y, Int(5)))
    aut_z5.set_invar(0, TRUE())
    aut_z5.set_transitions(0, [(0, [Equals(x_z, z)])])

    aut_z6 = AGAutomaton(symbols, [z], "aut_z6", 1)
    aut_z6.set_assume(0, And(GT(x, Int(0)), LT(x, Int(10))))
    aut_z6.set_invar(0, GT(z, Int(0)))
    aut_z6.set_transitions(0, [(0, [Equals(x_z, z)])])

    aut_z7 = AGAutomaton(symbols, [z], "aut_z7", 2)
    aut_z7.set_assume(0, TRUE())
    aut_z7.set_invar(0, TRUE())
    aut_z7.set_transitions(0, [(1, [Equals(x_z, z)])])
    aut_z7.set_assume(1, GE(Times(x, y), Int(1)))
    aut_z7.set_invar(1, TRUE())
    aut_z7.set_transitions(1, [(0, [Equals(x_z, z)])])

    automata = [aut_pc, aut_x, aut_y, aut_z,
                aut_pc1, aut_x1, aut_y1, aut_z1,
                aut_pc2, aut_x2, aut_y2, aut_z2,
                aut_pc3, aut_x3, aut_y3, aut_z3,
                aut_pc4, aut_x4, aut_y4, aut_z4,
                aut_pc5, aut_x5, aut_y5, aut_z5,
                aut_pc6, aut_x6, aut_y6, aut_z6,
                aut_pc7, aut_x7, aut_y7, aut_z7,
                aut_pc8, aut_x8]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
