from find_composition import find_composition
from automata_composition import AGAutomaton
from utils import symb_next

from pysmt.shortcuts import Symbol, TRUE, Int
from pysmt.shortcuts import Not, And, Implies
from pysmt.shortcuts import GT, GE, Equals, LT
from pysmt.shortcuts import Plus, Minus, Times
from pysmt.typing import INT


def test(nuxmv_path: str, model_file: str, trace_file: str, cmd_file: str,
         output_file: str) -> bool:
    """Test corresponding to benchmarks/19.smv"""
    # symbols.
    pc = Symbol("pc", INT)
    x = Symbol("x", INT)
    y = Symbol("y", INT)
    z = Symbol("z", INT)
    x_pc = Symbol(symb_next("pc"), INT)
    x_x = Symbol(symb_next("x"), INT)
    x_y = Symbol(symb_next("y"), INT)
    x_z = Symbol(symb_next("z"), INT)
    symbols = [pc, x, y, z]

    # initial location.
    init = Equals(pc, Int(0))

    # control flow graph.
    cfg = And(
        # pc = -1 : -1,
        Implies(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
        # pc = 0 & !(y >= 1) : -1,
        Implies(And(Equals(pc, Int(0)), Not(GE(y, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 0 & y >= 1 : 1,
        Implies(And(Equals(pc, Int(0)), GE(y, Int(1))), Equals(x_pc, Int(1))),
        # pc = 1 & !(z >= 1) : -1,
        Implies(And(Equals(pc, Int(1)), Not(GE(z, Int(1)))),
                Equals(x_pc, Int(-1))),
        # pc = 1 & z >= 1 : 2,
        Implies(And(Equals(pc, Int(1)), GE(z, Int(1))), Equals(x_pc, Int(2))),
        # pc = 2 & !(x >= 0) : -1,
        Implies(And(Equals(pc, Int(2)), Not(GE(x, Int(0)))),
                Equals(x_pc, Int(-1))),
        # pc = 2 & x >= 0 : 3,
        Implies(And(Equals(pc, Int(2)), GE(x, Int(0))), Equals(x_pc, Int(3))),
        # pc = 3 : 4,
        Implies(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
        # pc = 4 : 2,
        Implies(Equals(pc, Int(4)), Equals(x_pc, Int(2)))
    )

    # transition labels.
    labels = And(
        # (pc = -1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(-1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 0 & pc' = 1)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(0)), Equals(x_pc, Int(1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 1 & pc' = 2)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(1)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = -1) -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(-1))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 2 & pc' = 3)  -> (x' = x & y' = y & z' = z),
        Implies(And(Equals(pc, Int(2)), Equals(x_pc, Int(3))),
                And(Equals(x_x, x), Equals(x_y, y), Equals(x_z, z))),
        # (pc = 3 & pc' = 4)  -> (x' = y*z - 1 & y' = y & z' = z),
        Implies(And(Equals(pc, Int(3)), Equals(x_pc, Int(4))),
                And(Equals(x_x, Minus(Times(y, z), Int(1))), Equals(x_y, y),
                    Equals(x_z, z))),
        # (pc = 4 & pc' = 2)  -> (x' = x & y' = y+1 & z' = z),
        Implies(And(Equals(pc, Int(4)), Equals(x_pc, Int(2))),
                And(Equals(x_x, x), Equals(x_y, Plus(y, Int(1))),
                    Equals(x_z, z)))
    )

    # transition relation.
    trans = And(cfg, labels)

    # fairness.
    fairness = Not(Equals(pc, Int(-1)))

    # define automata to be composed.
    aut_pc = AGAutomaton(symbols, [pc], "aut_pc", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc.set_assume(loc, TRUE())
        aut_pc.set_invar(loc, Equals(pc, c_pc))
        aut_pc.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_pc1 = AGAutomaton(symbols, [pc], "aut_pc1", 3)
    for loc in range(aut_pc.num_locations):
        n_loc = (loc + 1) % aut_pc.num_locations
        c_pc = Int(loc + 2)
        n_pc = Int(n_loc + 2)
        aut_pc1.set_assume(loc, LT(x, Int(loc)))
        aut_pc1.set_invar(loc, Equals(pc, c_pc))
        aut_pc1.set_transitions(loc, [(n_loc, [Equals(x_pc, n_pc)])])

    aut_x = AGAutomaton(symbols, [x], "aut_x", 1)
    aut_x.set_assume(0, And(GT(y, Int(0)), GT(z, Int(0))))
    aut_x.set_invar(0, GE(x, Int(0)))
    aut_x.set_transitions(0, [(0, [Equals(x_x, x),
                                   Equals(x_x, Minus(Times(y, z), Int(1)))])])

    aut_y = AGAutomaton(symbols, [y], "aut_y", 1)
    aut_y.set_assume(0, TRUE())
    aut_y.set_invar(0, GE(y, Int(0)))
    aut_y.set_transitions(0, [(0, [Equals(x_y, y),
                                   Equals(x_y, Plus(y, Int(1)))])])

    aut_z = AGAutomaton(symbols, [z], "aut_z", 1)
    aut_z.set_assume(0, TRUE())
    aut_z.set_invar(0, TRUE())
    aut_z.set_transitions(0, [(0, [Equals(x_z, z)])])

    automata = [aut_pc, aut_x, aut_y, aut_z,
                aut_pc1]

    # search composition.
    comp, undefs = find_composition(automata, init, trans, fairness,
                                    nuxmv_path, model_file, trace_file,
                                    cmd_file)
    if comp is not None:
        with open(output_file, 'w') as out:
            out.write(str(comp))

    if comp and not undefs:
        res = True
    elif not comp:
        res = False
    else:
        assert not comp and undefs
        res = None
    return res
