#! /usr/bin/env python3

import argparse
import os
import re


def parse_fairfind_output(input_file: str) -> dict:

    def normalise_name(name: str):
        name = name.strip()
        if name.startswith("bench_"):
            name = name[len("bench_"):]
        if name == "nCr_combinations":
            name = name[:-1]
        return name

    assert os.path.isfile(input_file), "Not a file: {}".format(input_file)
    # regexp.
    test_name_re = re.compile(r"run \d+/\d+: `(?P<name>\S+)`")
    model_gen_re = re.compile(r"Reachability problem `\S+` generated in "
                              r"(?P<time>\d+\.\d*)s")
    nuxmv_re = re.compile(r"nuXmv returned in (?P<time>\d+\.\d*)s")
    parse_trace_re = re.compile(r"Trace `\S+` parsed in (?P<time>\d+\.\d*)s")
    comp_from_trace_re = re.compile(r"Candidate composition checked in "
                                    r"(?P<time>\d+\.\d*)s")
    check_comp_re = re.compile(r"Composition "
                               r"(?P<result>(VERIFIED)|(NOT VERIFIED)|(WRONG)), "
                               r"(?P<time>\d+\.\d*)s")
    test_end_re = re.compile(r"end of `\S+`: "
                             r"(?P<result>(SUCCESS)|(FAILURE)|(UNKNOWN))")

    results = {}
    with open(input_file, 'r') as in_stream:
        test_name = None
        model_gen_time = None
        nuxmv_time = None
        parse_trace_time = None
        comp_from_trace_time = None
        check_comp_time = None
        check_comp_res = None
        final_res = None
        for line in in_stream:
            line = line.strip()
            m = test_name_re.match(line)
            if m:
                assert test_name is None
                test_name = normalise_name(m.group("name"))
                continue
            m = model_gen_re.match(line)
            if m:
                val = float(m.group("time"))
                model_gen_time = model_gen_time + val \
                    if model_gen_time else val
                continue
            m = nuxmv_re.match(line)
            if m:
                val = float(m.group("time"))
                nuxmv_time = nuxmv_time + val if nuxmv_time else val
                continue
            m = parse_trace_re.match(line)
            if m:
                val = float(m.group("time"))
                parse_trace_time = parse_trace_time + val \
                    if parse_trace_time else val
                continue
            m = comp_from_trace_re.match(line)
            if m:
                val = float(m.group("time"))
                comp_from_trace_time = comp_from_trace_time + val \
                    if comp_from_trace_time else val
            m = check_comp_re.match(line)
            if m:
                val = float(m.group("time"))
                check_comp_time = check_comp_time + val \
                    if check_comp_time else val
                if m.group("result") == "VERIFIED":
                    check_comp_res = True
                elif m.group("result") == "NOT VERIFIED":
                    check_comp_res = None
                else:
                    assert m.group("result") == "WRONG"
                    check_comp_res = False
            m = test_end_re.match(line)
            if m:
                res = m.group("result")
                if res == "SUCCESS":
                    final_res = True
                elif res == "FAILURE":
                    final_res = False
                else:
                    assert res == "UNKNOWN"
                    final_res = None

                assert test_name not in results, \
                    "Name: {}, file: {}".format(test_name, input_file)
                assert model_gen_time
                assert nuxmv_time
                assert parse_trace_time, "{}".format(test_name)
                assert comp_from_trace_time
                assert check_comp_time
                results[test_name] = (model_gen_time, nuxmv_time,
                                      parse_trace_time, comp_from_trace_time,
                                      check_comp_time, check_comp_res,
                                      final_res)
                test_name = None
                model_gen_time = None
                nuxmv_time = None
                parse_trace_time = None
                comp_from_trace_time = None
                check_comp_time = None
                check_comp_res = None
                final_res = None
                continue

    return results


def parse_anant_output(input_dir: str) -> list:

    def normalise_name(name: str):
        name = name.strip()
        return name

    assert os.path.isdir(input_dir), "Not a directory: {}".format(input_dir)

    results = {}
    res_re = re.compile(r"Result\s*:\s*(?P<res>(Non-Terminating)|(Unknown))")
    runtime_re = re.compile(r"\[runlim\] real:\s+(?P<t>\d+\.\d*) seconds")
    for in_file in [el for el in os.scandir(input_dir)
                    if el.is_file() and el.name.endswith(".out")]:
        test_name = normalise_name(in_file.name[:-len(".out")])
        res = None
        runtime = None
        with open(in_file.path, 'r') as in_stream:
            for line in in_stream:
                line = line.strip()
                m = res_re.match(line)
                if m:
                    assert res is None
                    res = True if m.group("res") == "Non-Terminating" \
                        else None
                m = runtime_re.match(line)
                if m:
                    assert runtime is None
                    runtime = float(m.group("t"))
                    if runtime >= 600:
                        res = "TO"

        assert runtime is not None, "file: {}".format(in_file.name)
        results[test_name] = (res, runtime)
    return results


def parse_aprove_output(input_dir: str) -> list:

    def normalise_name(name: str):
        name = name.strip()
        p_re = re.compile(r"p\w+a?")
        if p_re.fullmatch(name):
            name = name[1:]
        return name

    assert os.path.isdir(input_dir), "Not a directory: {}".format(input_dir)

    results = {}
    res_re = re.compile(r"(?P<res>(NO)|(MAYBE)|(KILLED))")
    runtime_re = re.compile(r"\[runlim\] real:\s+(?P<t>\d+\.\d*) seconds")
    for in_file in [el for el in os.scandir(input_dir)
                    if el.is_file() and el.name.endswith(".out")]:
        test_name = normalise_name(in_file.name[:-len(".out")])
        res = None
        runtime = None
        with open(in_file.path, 'r') as in_stream:
            for line in in_stream:
                line = line.strip()
                m = res_re.match(line)
                if m:
                    assert res is None
                    res = True if m.group("res") == "NO" \
                        else None
                m = runtime_re.match(line)
                if m:
                    assert runtime is None
                    runtime = float(m.group("t"))
                    if runtime >= 600:
                        res = "TO"

        assert runtime is not None, "file: {}".format(in_file.name)
        results[test_name] = (res, runtime)
    return results


def parse_nuxmv_output(input_dir: str) -> list:

    def normalise_name(name: str):
        name = name.strip()
        return name

    assert os.path.isdir(input_dir), "Not a directory: {}".format(input_dir)

    results = {}
    res_re = re.compile(r"-- LTL specification .* is\s+(?P<res>(false)|(true))")
    runtime_re = re.compile(r"\[runlim\] real:\s+(?P<t>\d+\.\d*) seconds")
    for in_file in [el for el in os.scandir(input_dir)
                    if el.is_file() and el.name.endswith(".out")]:
        test_name = normalise_name(in_file.name[:-len(".out")])
        res = None
        runtime = None
        with open(in_file.path, 'r') as in_stream:
            for line in in_stream:
                line = line.strip()
                m = res_re.match(line)
                if m:
                    assert res is None
                    res = True if m.group("res") == "false" \
                        else False
                m = runtime_re.match(line)
                if m:
                    assert runtime is None
                    runtime = float(m.group("t"))
                    if runtime >= 600:
                        res = "TO"

        assert runtime is not None, "file: {}".format(in_file.name)
        results[test_name] = (res, runtime)
    return results


def getopts():
    p = argparse.ArgumentParser()
    p.add_argument("--aprove_out_dir", type=str, required=True)
    p.add_argument("--anant_out_dir", type=str, required=True)
    p.add_argument("--nuxmv_out_dir", type=str, required=True)
    p.add_argument("--comp_out_file", type=str, required=True)
    return p.parse_args()


def main(opts):
    assert os.path.isdir(opts.aprove_out_dir), \
        "Not a directory: {}".format(opts.aprove_out_dir)
    assert os.path.isdir(opts.anant_out_dir), \
        "Not a directory: {}".format(opts.anant_out_dir)
    assert os.path.isdir(opts.nuxmv_out_dir), \
        "Not a directory: {}".format(opts.nuxmv_out_dir)
    assert os.path.isfile(opts.comp_out_file), \
        "Not a file: {}".format(opts.comp_out_file)

    # print("ANANT")
    # parsed_anant = parse_anant_output(opts.anant_out_dir)
    # for name, (res, runtime) in sorted(parsed_anant.items(),
    #                                    key=lambda x: x[0]):
    #     print("name: {}".format(name))
    #     print("\tres: {}".format(res))
    #     print("\ttime: {}s".format(runtime))


    # print("\n\nAPROVE")
    # parsed_aprove = parse_aprove_output(opts.aprove_out_dir)
    # for name, (res, runtime) in sorted(parsed_aprove.items(),
    #                                    key=lambda x: x[0]):
    #     print("name: {}".format(name))
    #     print("\tres: {}".format(res))
    #     print("\ttime: {}s".format(runtime))

    # print("\n\nNUXMV")
    # parsed_nuxmv = parse_nuxmv_output(opts.nuxmv_out_dir)
    # for name, (res, runtime) in sorted(parsed_nuxmv.items(),
    #                                    key=lambda x: x[0]):
    #     print("name: {}".format(name))
    #     print("\tres: {}".format(res))
    #     print("\ttime: {}s".format(runtime))

    print("\n\nCOMPOSITIONAL")
    parsed_comp = parse_fairfind_output(opts.comp_out_file)

    for name, (gen, nuxmv, trace, comp, check,
               check_res, res) in sorted(parsed_comp.items(), key=lambda x: x[0]):
        total = gen + nuxmv + trace + comp
        print("\n{}: {}s".format(name, total))
        print("\tgen_comp_problem: {}s".format(gen))
        print("\tsolve_comp_problem: {}s".format(nuxmv))
        print("\tparse-trace: {}s".format(trace))
        print("\tcomp-from-trace: {}s".format(comp))
        print("\tcheck correctness: {}s".format(check))
        print("\tcheck correctness res: {}".format(check_res))
        print("\tres: {}".format(res))


if __name__ == "__main__":
    main(getopts())
