/**
Copyright (c) 2012-2014 Microsoft Corporation
   
Module Name:

    Sort.java

Abstract:

Author:

    @author Christoph Wintersteiger (cwinter) 2012-03-15

Notes:
    
**/ 

package com.microsoft.z3;

import com.microsoft.z3.enumerations.Z3_ast_kind;
import com.microsoft.z3.enumerations.Z3_sort_kind;

/**
 * The Sort class implements type information for ASTs.
 **/
public class Sort extends AST
{
    /* Overloaded operators are not translated. */

    /**
     * Equality operator for objects of type Sort. 
     * @param o
     * @return
     **/
    public boolean equals(Object o)
    {
        Sort casted = null;

    try {
        casted = Sort.class.cast(o);
    } catch (ClassCastException e) {
        return false;
    }

    return this.getNativeObject() == casted.getNativeObject();
    }

    /**
     * Hash code generation for Sorts
     * 
     * @return A hash code
     **/
    public int hashCode()
    {
        return super.hashCode();
    }

    /**
     * Returns a unique identifier for the sort.
     **/
    public int getId() throws Z3Exception
    {
        return Native.getSortId(getContext().nCtx(), getNativeObject());
    }

    /**
     * The kind of the sort.
     **/
    public Z3_sort_kind getSortKind() throws Z3Exception
    {
        return Z3_sort_kind.fromInt(Native.getSortKind(getContext().nCtx(),
                getNativeObject()));
    }

    /**
     * The name of the sort
     **/
    public Symbol getName() throws Z3Exception
    {
        return Symbol.create(getContext(),
                Native.getSortName(getContext().nCtx(), getNativeObject()));
    }

    /**
     * A string representation of the sort.
     **/
    public String toString()
    {
        try
        {
            return Native.sortToString(getContext().nCtx(), getNativeObject());
        } catch (Z3Exception e)
        {
            return "Z3Exception: " + e.getMessage();
        }
    }

    /**
     * Sort constructor
     **/
    Sort(Context ctx, long obj) throws Z3Exception
    {
        super(ctx, obj);
    }

    void checkNativeObject(long obj) throws Z3Exception
    {
        if (Native.getAstKind(getContext().nCtx(), obj) != Z3_ast_kind.Z3_SORT_AST
                .toInt())
            throw new Z3Exception("Underlying object is not a sort");
        super.checkNativeObject(obj);
    }

    static Sort create(Context ctx, long obj) throws Z3Exception
    {
        Z3_sort_kind sk = Z3_sort_kind.fromInt(Native.getSortKind(ctx.nCtx(), obj));
        switch (sk)
        {
        case Z3_ARRAY_SORT:
            return new ArraySort(ctx, obj);
        case Z3_BOOL_SORT:
            return new BoolSort(ctx, obj);
        case Z3_BV_SORT:
            return new BitVecSort(ctx, obj);
        case Z3_DATATYPE_SORT:
            return new DatatypeSort(ctx, obj);
        case Z3_INT_SORT:
            return new IntSort(ctx, obj);
        case Z3_REAL_SORT:
            return new RealSort(ctx, obj);
        case Z3_UNINTERPRETED_SORT:
            return new UninterpretedSort(ctx, obj);
        case Z3_FINITE_DOMAIN_SORT:
            return new FiniteDomainSort(ctx, obj);
        case Z3_RELATION_SORT:
            return new RelationSort(ctx, obj);
        case Z3_FLOATING_POINT_SORT:
            return new FPSort(ctx, obj);
        case Z3_ROUNDING_MODE_SORT:
            return new FPRMSort(ctx, obj);
        default:
            throw new Z3Exception("Unknown sort kind");
        }
    }
}
