#!/usr/bin/perl

use strict;
use warnings;
use Getopt::Long;

my $outputname = undef;
my $nameA = "nameA";
my $nameB = "nameB";
my $cutoff = 0;
my $timeout = 60;
my $minTimeDiff = 0.5;
my $treatFailAsTimeout = 0;

###############################################################################
sub usage();
sub readCSV($);
sub createScatterPlot($$$$);
sub createHTMLOverview($$$$$);
###############################################################################

my $res =
	GetOptions(
		"outputName=s" => \$outputname,
		"nameA=s" => \$nameA,
		"nameB=s" => \$nameB,
		"cutoff=f" => \$cutoff,
		"timeout=i" => \$timeout,
		"min-intr-time-diff=f" => \$minTimeDiff,
		"fail-as-timeout" => \$treatFailAsTimeout,
	);

my ($dirA, $dirB) = @ARGV;

if (!defined($dirA) || (! -r ($dirA . "/RESULT.csv"))) {
	print STDERR $dirA . " not a well-formed benchmark result directory.\n";
	usage();
} elsif (!defined($dirB) || (! -r ($dirB . "/RESULT.csv"))) {
	print STDERR $dirB . " not a well-formed benchmark result directory.\n";
	usage();
} elsif (!defined($outputname)) {
	print STDERR "No name for output given.\n";
	usage();
}

#Without this, we get // in a path, which IE cannot handle. Seriously. 80s, or what?
$dirA =~ s!/$!!;
$dirB =~ s!/$!!;

my $dataA = readCSV($dirA . "/RESULT.csv");
my $dataB = readCSV($dirB . "/RESULT.csv");

createScatterPlot($outputname . "-YES-scatter.tex", "YES", $dataA, $dataB);
createScatterPlot($outputname . "-NO-scatter.tex", "NO", $dataA, $dataB);
createHTMLOverview($outputname . "-Overview.html", $dirA, $dataA, $dirB, $dataB);

exit;

sub usage() {
	print <<EOS;
$0 -- benchmark pretty printer
Usage: $0 --outputname BASEFILENAME DIRA DIRB
Options:
 --nameA STRING	name used for the results from DIRA
 --nameB STRING	name used for the results from DIRB
 --timeout INT	the runtime assumed for examples without proper result
HTML Overview options:
 --intr-time-diff FLOAT	min. time diff for an ex. to be interesting/yellow
Scatterplot options:
 --cutoff FLOAT	examples where both results are faster will not be shown
 --fail-as-timeout	makes non-definite results look like timeout

DIRA and DIRB are expected to be generated by the benchmarking thing
implemented in the Makefile (i.e., you should have called it with
make TAG=DIRA/TAG=DIRB before).
EOS
	exit 1;
}

sub readCSV($) {
	my ($file) = @_;
	open(DATA, '<', $file) or die "Couldn't open file $file: $!";
	my %data;
	while (my $line = <DATA>) {
		$line =~ s/[\r\n]//g; #this is not OS-dependent, while chomp is
		my ($exname,$result,$time) = split /,/, $line;
		#skip results not related to termination:
		next if ($result eq 'UNSAFE');
		$time = $timeout if ($time !~ /\d+\.\d+/ || $time == 0);
		$data{$exname} = [$result, $time];
	}
	close(DATA);
	return \%data;
}

sub createHTMLOverview($$$$$) {
	my ($outname, $dirA, $dataA, $dirB, $dataB) = @_;

	open (HTMLOUT, '>', $outname) or die "Couldn't open file $outname: $!";
	print HTMLOUT <<EOHHEADER;
<html>
 <head>
  <title>$nameA vs. $nameB</title>
 <head>
 <body>
  <h1>$nameA vs. $nameB</h1>
  <h2>Example overview</h2>
  <table border="1">
   <tr>
    <th rowspan=2>Ex. name</th>
    <th colspan=2>$nameA</th>
    <th colspan=2>$nameB</th>
   </tr>
   <tr>
    <th>Result</th>
    <th>Time</th>
    <th>Result</th>
    <th>Time</th>
   </tr>

EOHHEADER
	my (%t);
	for my $ex (keys %$dataA) { $t{$ex} = 1; }
	for my $ex (keys %$dataB) { $t{$ex} = 1; }
	my @allexamples = keys %t;


	my ($aTERM, $aNONTERM, $sumATime, $sumATERMTime, $sumANONTERMTime) = (0,0,0,0,0);
	my ($bTERM, $bNONTERM, $sumBTime, $sumBTERMTime, $sumBNONTERMTime) = (0,0,0,0,0);
	my $count = 0;
	for my $ex (sort @allexamples) {
		my ($resA, $timeA) = @{$dataA->{$ex} || ["UNDEF", 0]};
		my ($resB, $timeB) = @{$dataB->{$ex} || ["UNDEF", 0]};

		#Normalize for given timeout:
		if ($timeA > $timeout) {
			$timeA = $timeout;
			$resA = "TIMEOUT";
		}

		if ($timeB > $timeout) {
			$timeB = $timeout;
			$resB = "TIMEOUT";
		}

		if ($resA eq "YES") {
			$aTERM++ if ($resA eq "YES");
			$sumATERMTime += $timeA;
		}
		if ($resB eq "YES") {
			$bTERM++ if ($resB eq "YES");
			$sumBTERMTime += $timeB;
		}
		if ($resA eq "NO") {
			$aNONTERM++ if ($resA eq "NO");
			$sumANONTERMTime += $timeA;
		}
		if ($resB eq "NO") {
			$bNONTERM++ if ($resB eq "NO");
			$sumBNONTERMTime += $timeB;
		}
		$sumATime += $timeA;
		$sumBTime += $timeB;
		$count++;

		my $colorstring = "";
		if (($resA eq "YES" && $resB eq "NO")
		    || ($resB eq "YES" && $resA eq "NO")) {
			$colorstring = 'style="background-color:red"';
		} elsif ($resA ne $resB || ($resA eq $resB && $resA !~ /UNKNOWN|No result/ && abs($timeA - $timeB) > $minTimeDiff)) {
			$colorstring = 'style="background-color:yellow"';
		}

		$timeA = sprintf "%.2f", $timeA;
		$timeB = sprintf "%.2f", $timeB;
		print HTMLOUT <<EOROW;
   <tr $colorstring>
    <td>$ex</td>
    <td><a href="$dirA/$ex.out">$resA</a></td>
    <td>$timeA</td>
    <td><a href="$dirB/$ex.out">$resB</a></td>
    <td>$timeB</td>
   </tr>
EOROW
	}
	my $avgATime = sprintf "%.2f", ($count>0)?$sumATime/$count:0;
	my $avgATERMTime = sprintf "%.2f", ($aTERM>0)?$sumATERMTime/$aTERM:0;
	my $avgANONTERMTime = sprintf "%.2f", ($aNONTERM>0)?$sumANONTERMTime/$aNONTERM:0;
	my $avgBTime = sprintf "%.2f", ($count>0)?$sumBTime/$count:0;
	my $avgBTERMTime = sprintf "%.2f", ($bTERM>0)?$sumBTERMTime/$bTERM:0;
	my $avgBNONTERMTime = sprintf "%.2f", ($bNONTERM>0)?$sumBNONTERMTime/$bNONTERM:0;
	print HTMLOUT <<EOHFOOTER;
  </table>
  <h2>Aggregate numbers</h2>
  <table border="1">
   <tr>
    <th></th>
    <th>$nameA</th>
    <th>$nameB</th>
   </tr>
   <tr>
    <td>Number YES</td>
    <td>$aTERM</td>
    <td>$bTERM</td>
   </tr>
   <tr>
    <td>Avg. time YES</td>
    <td>$avgATERMTime</td>
    <td>$avgBTERMTime</td>
   </tr>
   <tr>
    <td>Number NO</td>
    <td>$aNONTERM</td>
    <td>$bNONTERM</td>
   </tr>
   <tr>
    <td>Avg. time NO</td>
    <td>$avgANONTERMTime</td>
    <td>$avgBNONTERMTime</td>
   </tr>
   <tr>
    <td>Avg. time total</td>
    <td>$avgATime</td>
    <td>$avgBTime</td>
   </tr>
  </table>
 </body>
</html>
EOHFOOTER
	close(HTMLOUT);
}

sub createScatterPlot($$$$) {
	my ($outname, $filterres, $dataA, $dataB) = @_;
	
	#Prepare data, removing uninteresting stuff
	my @aexamples = keys %$dataA;
	my @tuples;
	my $maxtime = $timeout;
	foreach my $ex (@aexamples) {
		next unless (defined($dataB->{$ex}));
	
		my ($resA, $timeA) = @{$dataA->{$ex}};
		my ($resB, $timeB) = @{$dataB->{$ex}};

		#Only scatter if one has the right result:
		next unless ($resA eq $filterres || $resB eq $filterres);
		#Ignore if both below cutoff:
		next if ($timeA < $cutoff && $timeB < $cutoff);


		#Normalize for given timeout:
		if ($timeA > $timeout || ($treatFailAsTimeout && $resA ne $filterres)) {
			$timeA = $timeout;
			$resA = "TIMEOUT";
		}

		if ($timeB > $timeout || ($treatFailAsTimeout && $resB ne $filterres)) {
			$timeB = $timeout;
			$resB = "TIMEOUT";
		}

		$maxtime = $timeA if $timeA > $maxtime;
		$maxtime = $timeB if $timeB > $maxtime;

		push @tuples, [$timeA, $timeB];
	}
	#Compute 7 equi-distant points on time scale:
	my $stepwidth = $maxtime/6;
	my @timescale;
	for my $i (0..6) { $timescale[$i] = sprintf "%.0f", $i*$stepwidth; }
	my $timescaleStr = join ",", @timescale;

	open (SCATTEROUT, '>', $outname) or die "Couldn't open file $outname: $!";
	print SCATTEROUT <<EOSHEADER;
\\documentclass{article}

\\usepackage{tikz}
\\usepackage{pgfplots}

\\begin{document}

\\begin{tikzpicture}
    \\begin{axis}[
        width=12cm,
        height={},
        xlabel=$nameA (seconds),
        ylabel=$nameB (seconds),
        ymin=0,ymax=$maxtime, xmin=0,xmax=$maxtime,
        xtick={$timescaleStr},
        ytick={$timescaleStr},
        xticklabels={$timescaleStr},
        yticklabels={$timescaleStr},
        ]

    \\addplot [dashed,domain=0:$maxtime,black] {x};
    \\addplot [black,very thick] plot coordinates {(0,$maxtime) ($maxtime,$maxtime)}; %"timeout" line
    \\addplot [black,very thick] plot coordinates {($maxtime,0) ($maxtime,$maxtime)}; %"timeout" line

    \\addplot+[only marks,mark=x,mark size=3,blue] plot coordinates {
EOSHEADER
	foreach my $tuple (@tuples) {
		my ($tA,$tB) = @$tuple;
		print SCATTEROUT "($tA,$tB)\n";
	}
	print SCATTEROUT <<EOSFOOTER;
    };
    \\end{axis}
    \\end{tikzpicture}


\\end{document}
EOSFOOTER
	close (SCATTEROUT);
}
