;--------------------------------------------------------------------------------------
;
;  October 2020
;  Script associated to publication:
;  "Sensitivity of precipitation and temperature over Mount Kenya area to
;   physics parameterization options in a high-resolution model simulation
;   performed with WRFV3.8.1"
;  Authors: M. Messmer, S.j. González-Rojí, C.C. Raible and T.F. Stocker
;
;
;  Creative Commons Licence: Attribution-NonCommercial 4.0 International (CC BY-NC 4.0)
;--------------------------------------------------------------------------------------
begin

  dom     = getenv("DOM")    ;get domain number you would like to plot
  country = getenv("COU")    ;give country name you would like to plot, so that filenames are properly named
  setting = getenv("SET")    ;provide parent grid and number of nests, used for naming the input files
  nr_doms = getenv("NDOM")   ;provide number of domains in total, used for naming the input files
  era5_r  = getenv("E5_R")   ;filename of ERA5 precipitation
  alt_r   = getenv("ALT_R")  ;alternative data for precipitation, i.e., TRMM
  alt_r2  = getenv("ALT_R2") ;alternative data for precipitation, i.e., IMERG
  alt_r3  = getenv("ALT_R3") ;alternative data for precipitation, i.e., CHIRPS
  wrf_f   = getenv("WRF")
  alt_setting = getenv("ALT");add naming convention, just used to finde the proper file
  nr_stations = 63
  station = new((/nr_stations,12/),float)
  col_p_s = new((/nr_stations,12/),float)
  lat_s   = new(nr_stations, float)
  lon_s   = new(nr_stations, float)
  level_p = (/5,10,25,50,75,100,150,200,300,500,650/)
  color_p = (/2, 3, 4, 5, 6,  7,  8,  9, 10, 11, 12,13/)+9

  data = asciiread("Station_2008_mon_precipitation.txt",(/nr_stations,15/),"float")
  do ii = 0,dimsizes(lat_s)-1
    do jj = 0,11
      station(ii,jj) = data(ii,jj)
      if (station(ii,jj).lt.level_p(0)) then
        col_p_s(ii,jj) = color_p(0)
      end if
      do kk = 0,dimsizes(level_p)-2
        if (station(ii,jj).ge.level_p(kk) .and. station(ii,jj).lt.level_p(kk+1)) then
          col_p_s(ii,jj) = color_p(kk+1)
        end if
      end do
      if (station(ii,jj).ge.level_p(kk)) then
         col_p_s(ii,jj) = color_p(kk+1)
      end if
    end do
    lat_s(ii) = data(ii,12)
    lon_s(ii) = data(ii,13)
  end do

  files_RR = systemfunc("ls RR-D0"+dom+"-ERA5-"+country+"-"+setting+"-*-"+nr_doms+"-2008-monthlysum.nc")
  dd = dimsizes(files_RR)

  in = addfile(files_RR(0),"r")
  vv = in->prc
  t_d = dimsizes(vv)
  t_dim = dimsizes(t_d)
  delete(vv)
  dSizes = getfiledimsizes(in)

  if (t_dim .eq. 4) then
    ntime = t_d(0)
    nlat  = t_d(2)
    nlon  = t_d(3)
  else if (t_dim .eq.3) then
    ntime = t_d(0)
    nlat  = t_d(1)
    nlon  = t_d(2)
  end if
  end if
 
  experiment = (/"ERA5","AltData","AltData2","Alt","Cumulus3-1Way","Cumulus3","S.America","Europe","NoCumulus"/)
  RR = new((/9,ntime,nlat,nlon/),float)
  if (dd .eq.5) then
    mi = 5
  else
    mi = 6
  end if
  do i = 0,dimsizes(experiment)-mi 
    if (t_dim .eq. 4) then
      in = addfile(files_RR(i),"r")
      lat = in->lat
      lon = in->lon
      RR(i+4,:,:,:) = tofloat(in->prc)
    else if (t_dim .eq. 3) then
      in = addfile(files_RR(i),"r")
      lat = in->lat
      lon = in->lon
      RR(i+4,:,:,:) = tofloat(in->prc)
    end if
    end if 
  end do

  in = addfile(era5_r,"r")
  RR(0,:,:,:) = tofloat(in->var1(0:11,:,:))
  in = addfile(alt_r,"r")
  RR(1,:,:,:) = tofloat(in->PR(0:11,:,:))
  in = addfile(alt_r2,"r")
  RR(2,:,:,:) = tofloat(in->PR(0:11,:,:))
  in = addfile(alt_r3,"r")
  RR(3,:,:,:) = tofloat(in->precip(0:11,:,:))
  ; print to check if variable have the same units
  print(""+min(RR(0,:,:,:))+"  "+avg(RR(0,:,:,:))+"  "+max(RR(0,:,:,:)))
  print(""+min(RR(1,:,:,:))+"  "+avg(RR(1,:,:,:))+"  "+max(RR(1,:,:,:)))
  print(""+min(RR(2,:,:,:))+"  "+avg(RR(2,:,:,:))+"  "+max(RR(2,:,:,:)))
  print(""+min(RR(8,:,:,:))+"  "+avg(RR(8,:,:,:))+"  "+max(RR(8,:,:,:)))

  in = addfile(wrf_f,"r")

  wks = gsn_open_wks("pdf","RR_"+country+"_2008_D0"+dom+"_"+alt_setting)

  gsn_merge_colormaps(wks,"parameterization","precip_11lev")

  cnres                       = True
  cnres@gsnFrame              = False
  cnres@gsnDraw               = False
  cnres@cnLinesOn             = False
  cnres@cnLineLabelsOn        = False
  cnres@cnInfoLabelOn         = False

  cnres@cnFillOn              = True
  cnres@cnFillMode            = "AreaFill"    ; This is the default.
  cnres@lbLabelFontHeightF    = 0.010
  cnres@lbLabelStride         = 1 
  cnres@lbLabelBarOn          = False

  ; Set contour levels.
  cnres@cnLevelSelectionMode  = "ExplicitLevels"
  cnres@gsnMaximize           = True
  cnres@gsnLeftString         = ""
  cnres@gsnCenterString       = ""
  cnres@gsnRightString        = "" 
  res = cnres   ; Copy contour resources for 1st plot.
  res = wrf_map_resources(in,res)
  dx = in@DX
  dx = dx / 1000
  res@mpFillOn                = True
  res@mpPerimOn               = True
  res@mpFillColors            = (/ -1, -1 ,-1, -1/)
; res@mpInlandWaterFillColor  = 3
  res@mpFillDrawOrder         = "PostDraw"
  res@mpOutlineBoundarySets   = "AllBoundaries"
  res@mpNationalLineThicknessF= 1.0
  res@mpGeophysicalLineThicknessF= 1.5
  res@mpGeophysicalLineColor  = "black"
  res@mpNationalLineColor     = "black"
  res@mpDataBaseVersion       = "MediumRes"
  res@mpDataSetName           = "Earth..4"
; res@pmTickMarkDisplayMode   = "Never"
  res@mpGridAndLimbDrawOrder  = "PostDraw"
  res@mpGridAndLimbOn         = True
  res@mpGridLatSpacingF       = 30
  res@mpGridLonSpacingF       = 30
  res@tfDoNDCOverlay          = True
  res@tmXBLabelFontHeightF    = 0.02
  res@tmYLLabelFontHeightF    = 0.02

  gsres               = True
  gsres@gsMarkerIndex = 16
  gsres@gsMarkerSizeF = 0.009
  gsres@gsMarkerThicknessF = 1.5

  gsres2 = gsres
  gsres2@gsMarkerIndex = 4
  gsres2@gsMarkerThicknessF = 1.0
  gsres2@gsMarkerColor = "black"

  kres = True
  kres@gsMarkerSizeF = gsres@gsMarkerSizeF
  kres@gsMarkerIndex = 12
  kres@gsMarkerThicknessF = 2.0
  kres@gsMarkerColor = (/0.95,0.95,0.95,1/)
  tres = True
  tres@txFontHeightF = 0.02

  t2 = new((/dimsizes(experiment),12/),graphic)
  rr = new((/dimsizes(experiment),12/),graphic)
  stat_rr = new((/dimsizes(experiment),12,dimsizes(lat_s)/),graphic)
  stat_r2 = new((/dimsizes(experiment),12,dimsizes(lat_s)/),graphic)
  mt_kenya = new((/dimsizes(experiment),12/),graphic)
  rr_text = new((/dimsizes(experiment),12,dimsizes(lat_s)/),graphic)

  panres = True
  panres@gsnMaximize = True
  panres@gsnPaperOrientation = "landscape"
  panres@gsnPanelLabelBar    = True
  panres@lbLabelFontHeightF  = 0.012
  panres@lbTitlePosition     = "bottom"
  panres@lbTitleFontHeightF  = 0.012
  panres@lbTitleString       = "monthly precipitation sums [mm]"
  panres@lbTitleFont         = "helvetica"

  res1                     = True
  res2                     = True
  res1@gsnMaximize         = True
  res2@gsnAttachPlotsYAxis = True

  mon = new(12,string)
  mon(0) = "Jan"
  mon(1) = "Feb"
  mon(2) = "Mar"
  mon(3) = "Apr"
  mon(4) = "May"
  mon(5) = "Jun"
  mon(6) = "Jul"
  mon(7) = "Aug"
  mon(8) = "Sep"
  mon(9) = "Oct"
  mon(10)= "Nov"
  mon(11)= "Dec"

  exp_name = (/"ERA5","TRMM","IMERG","CHIRPS","Cumulus3-1Way","Cumulus3","South America","Europe","No Cumulus"/)

  do ii = 0,dimsizes(experiment)-1
    do jj = 0,11

      res@cnLevels    = level_p 
      res@cnFillColors  = color_p 
      res@gsnStringFont = "helvetica-bold"
      res@gsnStringFontHeightF = 0.02
      res@gsnStringFontColor = "black" ; "white"
      res@gsnLeftString = "  "
      res@gsnRightString = ""+mon(jj)+"  "
      
      txres                       = True
      txres@txPerimOn             = True
      txres@txFontHeightF         = 0.03
      txres@txBackgroundFillColor = ii+2
      txres@txPerimColor          = "white"
      txres@txFont                = "helvetica-bold"
      txres@txFontColor           = "white"
      text_label = "    "+exp_name(ii)+"    "
      txid = gsn_create_text(wks, text_label, txres)

      amres = True
      amres@amParallelPosF   = -0.5   ; This is the center of the plot.
      amres@amOrthogonalPosF = -0.507   ; This is the top edge of the plot.
      amres@amJust           = "BottomLeft"
      rr(ii,jj) = gsn_csm_contour_map(wks, RR(ii,jj,5:nlat-6,5:nlon-6), res)
      annoid = gsn_add_annotation(rr(ii,jj), txid, amres)
      ll = 1
      do kk = 0,dimsizes(lat_s)-1
        gsres@gsMarkerColor = col_p_s(kk,jj)
        if (kk .eq. 13 .and. jj.eq.3) then
          print("col: "+col_p_s(kk,jj))
        end if 
        stat_rr(ii,jj,kk) = gsn_add_polymarker(wks,rr(ii,jj),lon_s(kk),lat_s(kk),gsres)
        stat_r2(ii,jj,kk) = gsn_add_polymarker(wks,rr(ii,jj),lon_s(kk),lat_s(kk),gsres2)
        if (ii .eq. 0) then
          if (dx.eq.1 .and. country.eq."Kenya") then
            if (lon_s(kk).gt.min(lon) .and. lon_s(kk).lt.max(lon) .and. lat_s(kk).gt.min(lat) .and. lat_s(kk).lt.max(lat)) then
              rr_text(ii,jj,kk) = gsn_add_text(wks,rr(ii,jj),tostring(ll),lon_s(kk),lat_s(kk)-0.055,tres)
              ll = ll + 1
            end if
          end if
        end if
        ; add marker over Mount Kenya
        mt_kenya(ii,jj) = gsn_add_polymarker(wks,rr(ii,jj),37.3085,-0.1518,kres)
      end do
      delete(res@cnLevels)
      delete(res@cnFillColors)
      
    end do
  end do
  
  do jj = 0,11
    ann1 = gsn_attach_plots(rr(2,jj),(/rr(4,jj),rr(6,jj),rr(8,jj)/),res1,res2)
    ann2 = gsn_attach_plots(rr(3,jj),(/rr(5,jj),rr(7,jj),rr(0,jj)/),res1,res2)
    gsn_panel(wks,(/rr(2,jj),rr(3,jj)/),(/2,1/),panres)
  end do

end

