#####################################################################
# WhatIsRiskMarket Analysis Script
#####################################################################

rm(list=ls()) # Clears memory
graphics.off() # Clears graphs
if (!require("pacman")) install.packages("pacman") #Installs package for package installation
pacman::p_load("reshape","PerformanceAnalytics","xlsx","MASS","texreg","plm")

##### Parameters

#~~~ Specify full path and file name of source files ~~~#
ReturnData<-T
SourceFiles<-list.files("FilesForAnalysisReturns/.",pattern="[0-9]{6}_[0-9]{4}.xls",full.names=T,recursive=F)
QSourceFiles<-list.files("FilesForAnalysisReturns/.",pattern="[0-9]{6}_[0-9]{4}.sbj",full.names=T,recursive=F)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#
Tables<-c("globals","returns","subjects","transactions")
RemovePracticePeriodTables<-Tables[!Tables=="session"] # Subset of tables which have life < session
ShowPlots<-T # Should plots be shown on screen or only written to disk?
RemovePracticePeriods<-T
RemoveOutliers<-F    #Should session 7 period 1 of risk data be removed?

NumPeriods<-8
NumSubjects<-8

source("GIMS_DataPreparation_Return.r")    #Creates the file if it does not

Data$subjects$RiskPerception<-Data$subjects$RiskPerception+1    #Recodes RiskPerception to run from 1 to 7 instead of from 0 to 6
if (RemoveOutliers){    #Removes outliers
    Data$globals<-Data$globals[!(Data$globals$R.Session==7&Data$globals$Period==1),]
    Data$subjects<-Data$subjects[!(Data$subjects$R.Session==7&Data$subjects$Period==1),]
    Data$transactions<-Data$transactions[!(Data$transactions$R.Session==7&Data$transactions$Period==1),]
}

############## WhatIsRiskMarket ##########################

R.WRM.Lookup<-merge(Lookup["R.PeriodID"],Data$globals[,c("R.PeriodID","Distribution","StartTime","StartTimeCDA")])[,-1] # Generates matrix containing R.PeriodID and several variables from the globals table
R.WRM.transactions<-merge(Data$transactions,R.WRM.Lookup) # Generates matrix containing transaction data with Distribution information
R.WRM.transactions<-cbind(R.WRM.transactions, R.TradeTime=R.WRM.transactions$Time-(R.WRM.transactions$StartTimeCDA-R.WRM.transactions$StartTime)) # Adds precise timing for trades
NumDistributions<-max(R.WRM.transactions[,"Distribution"])
NumSessions<-max(as.numeric(R.WRM.transactions[,"R.Session"]))
DistributionText<-c("Normal","Negative Skewness","Positive Skewness","Big Loss","Wider","Many Small Losses","No Losses","Fat Tails")
R.WRM.subjects<-merge(Data$subjects,R.WRM.Lookup) # Generates subjects table with Distribution information

SessionPeriods<-list()
for (Session in 1:NumSessions){    #Generates variable to list the periods per session that are in the data
    SessionPeriods[[Session]]<-Data$globals$Period[Data$globals$R.Session==Session]
}

##### Creates matrix holding average risk perception of distributions
AvgDistributionPerception<-matrix(c(1:NumDistributions,rep(NA,3*NumDistributions)),ncol=4)
colnames(AvgDistributionPerception)<-c("Distribution","Avg. RiskPerception (0=not risky, 6=very risky)","Avg. Price","Avg. Price Last Minute") # Prepares table for risk perception averages by distribution
for (i in 1:NumDistributions){
    AvgDistributionPerception[i,2]<-mean(R.WRM.subjects$RiskPerception[R.WRM.subjects$Distribution==i]) # Writes risk perception
}



### Calculates characteristic measures of distributions
Returns<-matrix(NA,nrow=1000,ncol=NumDistributions) # Creates empty matrix for returns
for (Distribution in 1:NumDistributions) {
    Returns[,Distribution]<-RawData$returns$Return[RawData$returns$Distribution==Distribution&RawData$returns$Date==RawData$returns$Date[1]] # Fills it
}

SummaryStats<-c("Mean","StdDev","Semivar", "PLoss", "ELoss", "Skewness","Kurtosis","MinReturn","MaxReturn")
# Defines matrix columns
DistributionCharacteristics<-matrix(rep(NA,length(c("Distribution",SummaryStats))*NumDistributions),nrow=NumDistributions) # Creates matrix
dimnames(DistributionCharacteristics)<-list(1:NumDistributions,c("Distribution",SummaryStats)) # Names columns
# Fills matrix
DistributionCharacteristics[,1]<-1:NumDistributions
DistributionCharacteristics[,2]<-apply(Returns,2,mean)
DistributionCharacteristics[,3]<-apply(Returns,2,sd)
DistributionCharacteristics[,4]<-apply(Returns,2,SemiVariance)
DistributionCharacteristics[,5]<-apply(Returns,2,function(x) sum(x<0)/nrow(Returns))
DistributionCharacteristics[,6]<-apply(Returns,2,function(x) sum(x[x<0]/nrow(Returns)))
DistributionCharacteristics[,7]<-apply(Returns,2,skewness)
DistributionCharacteristics[,8]<-apply(Returns,2,kurtosis)
DistributionCharacteristics[,9]<-apply(Returns,2,min)
DistributionCharacteristics[,10]<-apply(Returns,2,max)

### Prepares regression datasets

Data$subjects<-merge(Data$subjects,Data$globals[,c("R.Session","Period","Distribution")],by=c("R.Session","Period"))
Data$transactions<-merge(Data$transactions,Data$globals[,c("R.Session","Period","Distribution")],by=c("R.Session","Period"))


# Prepares matrix of risk perceptions merged with regressor data
RiskPerceptions<-merge(Data$subjects,DistributionCharacteristics,by="Distribution")


# Prepares matrix of market prices merged with regressor data
MeanPrices<-matrix(c(Data$globals$R.Session,Data$globals$Distribution,rep(NA,8*length(Data$globals$R.Session))),nrow=length(Data$globals$Distribution)) 
dimnames(MeanPrices)<-list(1:nrow(MeanPrices),c("R.Session","Distribution","AvgPrice","AvgPriceLast60","AvgRiskPerception","Period","Volume","VolumeLast60","MedianPrice","MedianPriceLast60"))
MeanPrices<-MeanPrices[order(MeanPrices[,"R.Session"],MeanPrices[,"Distribution"]),]

CurrentRow<-0
for (Session in 1:NumSessions){
    for (Distribution in 1:NumDistributions){
        if (length(MeanPrices[MeanPrices[,"R.Session"]==Session&MeanPrices[,"Distribution"]==Distribution,])>0){
            CurrentRow<-CurrentRow+1
            MeanPrices[CurrentRow,3]<-mean(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,4]<-mean(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
            MeanPrices[CurrentRow,5]<-mean(Data$subjects$RiskPerception[Data$subjects$R.Session==Session&Data$subjects$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,6]<-mean(Data$subjects$Period[Data$subjects$R.Session==Session&Data$subjects$Distribution==Distribution],na.rm=T)  
            MeanPrices[CurrentRow,7]<-sum(Data$transactions$Volume[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,8]<-sum(Data$transactions$Volume[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
            MeanPrices[CurrentRow,9]<-median(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,10]<-median(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
        }
    }
}
MeanPrices<-merge(MeanPrices,DistributionCharacteristics,by="Distribution") # Merges in regressor data
if (RemoveOutliers){MeanPrices<-MeanPrices[!(MeanPrices$R.Session==7&is.nan(MeanPrices$Period)),]}   #Removes outliers


# Adds average of average prices to AvgDistributionPerception matrix
for (Distribution in 1:NumDistributions){
    AvgDistributionPerception[Distribution,2]<-mean(MeanPrices$AvgRiskPerception[MeanPrices$Distribution==Distribution]) # Writes average price
    AvgDistributionPerception[Distribution,3]<-mean(MeanPrices$AvgPrice[MeanPrices$Distribution==Distribution]) # Writes average price
    AvgDistributionPerception[Distribution,4]<-mean(MeanPrices$AvgPriceLast60[MeanPrices$Distribution==Distribution]) # Writes average price in the last minute
}


#R08
RiskPerceptions<-transform(RiskPerceptions, UniqueSessionSubjectID = as.factor(interaction(R.Session,Subject, drop=TRUE)))   #Generates unique 


# Regression on distributions

# Creates dummy variables for the distributions
{
    Temp1<-ncol(MeanPrices)
    for (Distribution in 1:NumDistributions){
        MeanPrices[,Temp1+Distribution]<-as.integer(MeanPrices$Distribution==Distribution)
    }
    
    colnames(MeanPrices)<-c(colnames(MeanPrices[,1:Temp1]),"Normal","NegSkew","PosSkew","BigLoss","Wider","LargeLossProb","NoLosses","FatTails5")
}

# ### Mean prices
DistributionMeans<-aggregate(MeanPrices[, 3:ncol(MeanPrices)], list(MeanPrices$Distribution), FUN=mean, na.rm=TRUE)
colnames(DistributionMeans)<-c("Distribution",colnames(DistributionMeans[,2:ncol(DistributionMeans)]))


### Distribution-level graphs
#R05
AvgMeanPrices<-aggregate(MeanPrices, by=list(MeanPrices$Distribution), mean)   #Generate AvgMeanPrices to contain distribution averages of data in MeanPrices

# Plot AvgPrice vs. AvgRiskPerception
# Figure B.8 --------------------------------------------------------------

MPlot.AvgPrice_AvgRiskPerception<-lm(AvgMeanPrices$AvgPrice~AvgMeanPrices$AvgRiskPerception)
dev.new("DistAvgPriceVsRiskPerception")
{
    plot(x=AvgMeanPrices$AvgRiskPerception,y=AvgMeanPrices$AvgPrice,xlab="Average attractiveness perception", ylab="Average price")
    abline(MPlot.AvgPrice_AvgRiskPerception)
    dev.copy(png,"WhatIsRiskMarket_Scatter_DistAvgPriceVsRiskPerception.png", bg="white", width=1700, height=1700, res=300)
}
dev.off() # Turns off graphics device



#R07
# Figure B.7 --------------------------------------------------------------

dev.new("AvgRiskPerception")
{
    par(mfrow=c(4,2), oma=c(1,3.5,1,1), mar=c(5, 0, 1, 1), bg="white")
    MPlot <- lm(AvgRiskPerception~StdDev, data=DistributionMeans)
    plot(DistributionMeans$StdDev,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Standard deviation, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Semivar, data=DistributionMeans)
    plot(DistributionMeans$Semivar,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Semivariance, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))), ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~PLoss, data=DistributionMeans)
    plot(DistributionMeans$PLoss,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Probability of loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="",  main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~ELoss, data=DistributionMeans)
    plot(DistributionMeans$ELoss,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Expected loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Skewness, data=DistributionMeans)
    plot(DistributionMeans$Skewness,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Skewness, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Kurtosis, data=DistributionMeans)
    plot(DistributionMeans$Kurtosis,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Kurtosis, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~MinReturn, data=DistributionMeans)
    plot(DistributionMeans$MinReturn,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Minimum return, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    title(xlab = "", ylab = "Average attractiveness perception", outer = TRUE, line = 2)
    dev.copy(png,"WhatIsRiskMarket_Scatter_DistRiskPerception.png", bg="white", width=1700, height=2300, res=300)
}
dev.off() # Turns off graphics device if even Distribution number