# WhatIsRiskMarket Analysis Script

rm(list=ls()) # Clears memory
graphics.off() # Clears graphs
if (!require("pacman")) install.packages("pacman") #Installs package for package installation
pacman::p_load("reshape","PerformanceAnalytics","xlsx","MASS","texreg","plm","zTree","reshape2","Hmisc","car","devtools")
library("devtools") #Loads necessary package
install_github("stpalan/SPTools") #Installs necessary package
library("SPTools")

##### Parameters

#~~~ Specify full path and file name of source files ~~~#
SourceFiles<-list.files("FilesForAnalysis/.",pattern="[0-9]{6}_[0-9]{4}.xls",full.names=T,recursive=F)
QSourceFiles<-list.files("FilesForAnalysis/.",pattern="[0-9]{6}_[0-9]{4}.sbj",full.names=T,recursive=F)
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#
Tables<-c("globals","returns","subjects","transactions","offers","timelog")
RemovePracticePeriodTables<-Tables[!Tables=="session"] # Subset of tables which have life < session
ShowPlots<-T # Should plots be shown on screen or only written to disk?
RemovePracticePeriods<-T
RemoveOutliers<-T    #Should session 7 period 1 be removed?

NumPeriods<-8
NumSubjects<-8

source("GIMS_DataPreparation.R")    #Loads data

Data$subjects$RiskPerception<-Data$subjects$RiskPerception+1    #Recodes RiskPerception to run from 1 to 7 instead of from 0 to 6 as in raw data
if (RemoveOutliers==T){    #Removes outliers
    Data$transactions<-Data$transactions[!(Data$transactions$R.Session==7&Data$transactions$Period==1),]
}



# Start analysis

R.WRM.Lookup<-merge(Lookup["R.PeriodID"],Data$globals[,c("R.PeriodID","Distribution","StartTime","StartTimeCDA")])[,-1] # Generates matrix containing R.PeriodID and several variables from the globals table
R.WRM.transactions<-merge(Data$transactions,R.WRM.Lookup) # Generates matrix containing transaction data with Distribution information
R.WRM.transactions<-cbind(R.WRM.transactions, R.TradeTime=R.WRM.transactions$Time-(R.WRM.transactions$StartTimeCDA-R.WRM.transactions$StartTime)) # Adds precise timing for trades
NumDistributions<-max(R.WRM.transactions[,"Distribution"])
NumSessions<-max(as.numeric(R.WRM.transactions[,"R.Session"]))
DistributionText<-c("NORMAL","NegSKEWNESS","PosSKEWNESS","BigLOSS","WIDER","FrequentLOSS","NoLOSS","KURTOSIS")
R.WRM.subjects<-merge(Data$subjects,R.WRM.Lookup) # Generates subjects table with Distribution information

SessionPeriods<-list()
for (Session in 1:NumSessions){    #Generates variable to list the periods per session that are in the data
    SessionPeriods[[Session]]<-Data$globals$Period[Data$globals$R.Session==Session]
}

##### Creates matrix holding average risk perception of distributions
AvgDistributionPerception<-matrix(c(1:NumDistributions,rep(NA,3*NumDistributions)),ncol=4)
colnames(AvgDistributionPerception)<-c("Distribution","Avg. RiskPerception (0=not risky, 6=very risky)","Avg. Price","Avg. Price Last Minute") # Prepares table for risk perception averages by distribution
for (i in 1:NumDistributions){
    AvgDistributionPerception[i,2]<-mean(R.WRM.subjects$RiskPerception[R.WRM.subjects$Distribution==i]) # Writes risk perception
}


### Calculates characteristic measures of distributions
Returns<-matrix(NA,nrow=100,ncol=NumDistributions) # Creates empty matrix for returns
for (Distribution in 1:NumDistributions) {
    Returns[,Distribution]<-RawData$returns$Return[RawData$returns$Treatment==2&RawData$returns$Distribution==Distribution&RawData$returns$Date==RawData$returns$Date[1]] # Fills it
}

SummaryStats<-c("Mean","StdDev","Semivar", "PLoss", "ELoss", "Skewness","Kurtosis","MinReturn","MaxReturn")
# Defines matrix columns
DistributionCharacteristics<-matrix(rep(NA,length(c("Distribution",SummaryStats))*NumDistributions),nrow=NumDistributions) # Creates matrix
dimnames(DistributionCharacteristics)<-list(1:NumDistributions,c("Distribution",SummaryStats)) # Names columns
# Fills matrix
DistributionCharacteristics[,1]<-1:NumDistributions
DistributionCharacteristics[,2]<-apply(Returns,2,mean)
DistributionCharacteristics[,3]<-apply(Returns,2,sd)*(nrow(Returns)-1)/nrow(Returns)
DistributionCharacteristics[,4]<-apply(Returns,2,SemiVariance)
DistributionCharacteristics[,5]<-apply(Returns,2,function(x) sum(x<0)/nrow(Returns))
DistributionCharacteristics[,6]<-apply(Returns,2,function(x) sum(x[x<0]/nrow(Returns)))
DistributionCharacteristics[,7]<-apply(Returns,2,skewness)
DistributionCharacteristics[,8]<-apply(Returns,2,kurtosis)
DistributionCharacteristics[,9]<-apply(Returns,2,min)
DistributionCharacteristics[,10]<-apply(Returns,2,max)


#R132
# Table 1 -----------------------------------------------------------------
DistributionCharacteristics




### Prepares regression datasets

Data$subjects<-merge(Data$subjects,Data$globals[,c("R.Session","Period","Distribution")],by=c("R.Session","Period"))
Data$transactions<-merge(Data$transactions,Data$globals[,c("R.Session","Period","Distribution")],by=c("R.Session","Period"))
Data$subjects$RiskFinancial<-suppressWarnings(as.numeric(levels(Data$subjects$RiskFinancial))[Data$subjects$RiskFinancial])
Data$subjects$RiskGeneral<-suppressWarnings(as.numeric(levels(Data$subjects$RiskGeneral))[Data$subjects$RiskGeneral])


# Prepares matrix of risk perceptions merged with regressor data
RiskPerceptions<-merge(Data$subjects,DistributionCharacteristics,by="Distribution")

# Prepares matrix of market prices merged with regressor data
MeanPrices<-matrix(as.numeric(levels(Data$globals$R.Session))[Data$globals$R.Session])    #Recodes R.Session to numeric (from factor))
MeanPrices<-cbind(MeanPrices,Data$globals$Distribution,matrix(rep(NA,8*nrow(MeanPrices)),nrow=nrow(MeanPrices)))
dimnames(MeanPrices)<-list(1:nrow(MeanPrices),c("R.Session","Distribution","AvgPrice","AvgPriceLast60","AvgRiskPerception","Period","Volume","VolumeLast60","MedianPrice","MedianPriceLast60"))
MeanPrices<-MeanPrices[order(MeanPrices[,"R.Session"],MeanPrices[,"Distribution"]),]

CurrentRow<-0
for (Session in 1:NumSessions){
    for (Distribution in 1:NumDistributions){
        if (length(MeanPrices[MeanPrices[,"R.Session"]==Session&MeanPrices[,"Distribution"]==Distribution,])>0){
            CurrentRow<-CurrentRow+1
            MeanPrices[CurrentRow,3]<-mean(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,4]<-mean(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
            MeanPrices[CurrentRow,5]<-mean(Data$subjects$RiskPerception[Data$subjects$R.Session==Session&Data$subjects$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,6]<-mean(Data$subjects$Period[Data$subjects$R.Session==Session&Data$subjects$Distribution==Distribution],na.rm=T)  
            MeanPrices[CurrentRow,7]<-sum(Data$transactions$Volume[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,8]<-sum(Data$transactions$Volume[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
            MeanPrices[CurrentRow,9]<-median(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution],na.rm=T)
            MeanPrices[CurrentRow,10]<-median(Data$transactions$Price[Data$transactions$R.Session==Session&Data$transactions$Distribution==Distribution&Data$transactions$Time>120],na.rm=T)
        }
    }
}
MeanPrices<-merge(MeanPrices,DistributionCharacteristics,by="Distribution") # Merges in regressor data
if(RemoveOutliers){MeanPrices<-MeanPrices[!is.na(MeanPrices$AvgPrice),]} #Removes outlier

# Adds average of average prices to AvgDistributionPerception matrix
for (Distribution in 1:NumDistributions){
    AvgDistributionPerception[Distribution,2]<-mean(MeanPrices$AvgRiskPerception[MeanPrices$Distribution==Distribution]) # Writes average price
    AvgDistributionPerception[Distribution,3]<-mean(MeanPrices$AvgPrice[MeanPrices$Distribution==Distribution]) # Writes average price
    AvgDistributionPerception[Distribution,4]<-mean(MeanPrices$AvgPriceLast60[MeanPrices$Distribution==Distribution]) # Writes average price in the last minute
}


### Regression

# RiskPerceptions<-merge(x=RiskPerceptions, y=DistributionRanks,by=c("R.Session", "Subject", "Distribution"))    #Adds Rank to RiskPerceptions
RiskPerceptions<-merge(x=RiskPerceptions,y=Data$subjects[Data$subjects$Period==NumPeriods,c("R.Session","Subject","FinLitScore","AnswerFinLit[2]","AnswerFinLit[3]","AnswerFinLit[4]","AnswerFinLit[7]","AnswerFinLit[10]","AnswerFinLit[12]","AnswerFinLit[16]")],all.x=T)    #Adds FinLitScore and FinLit answers to RiskPerceptions


#R08
RiskPerceptions<-transform(RiskPerceptions, UniqueSessionSubjectID = as.factor(interaction(R.Session,Subject, drop=TRUE)))   #Generates unique SessionXSubject ID
MRiskPerception.1<-lm(RiskPerception~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=RiskPerceptions)
# #summary(MRiskPerception.1)
MRiskPerception.1FE<-plm(RiskPerception~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=RiskPerceptions[,c("RiskPerception","StdDev","Semivar","PLoss","ELoss","Skewness","Kurtosis","MinReturn","UniqueSessionSubjectID")], effect="individual", Model="within",index="UniqueSessionSubjectID")    #Estimates linear subject fixed effects panel regression of RiskPerception on distribution characteristics

MRiskPerception.3<-lm(RiskPerception~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=RiskPerceptions)
MRiskPerception.3FE<-plm(RiskPerception~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=RiskPerceptions[,c("RiskPerception","StdDev","Semivar","PLoss","ELoss","Skewness","Kurtosis","MinReturn","UniqueSessionSubjectID")], effect="individual", Model="within",index="UniqueSessionSubjectID")    #Estimates linear subject fixed effects panel regression of RiskPerception on distribution characteristics


#R23
# Table 2 -----------------------------------------------------------------

RiskPerceptions$R.UniqueSubject<-interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject) #Generates unique subject ID variable
RegressionOutput3b.LaTeX <- texreg(
    list(MRiskPerception.1,MRiskPerception.1FE),
    file="WhatIsRiskMarket_Regression_Reg3b.tex",
    stars = c(0.01, 0.05, 0.1),
    custom.model.names=c("OLS","Subject fixed effects"),
    custom.note="%stars. Standard errors, clustered at the subject level, in parentheses.",
    override.se=list( #Calculates standard errors clustered at the subject level
        SPClusterSE(RiskPerceptions,MRiskPerception.1,RiskPerceptions$R.UniqueSubject)[9:16],
        coeftest(MRiskPerception.1FE, vcov=vcovHC(MRiskPerception.1FE,type="HC0",cluster="group"))[8:14]),
    override.pvalues=list( #Calculates p-values for standard errors clustered at the subject level
        SPClusterSE(RiskPerceptions,MRiskPerception.1,RiskPerceptions$R.UniqueSubject)[25:32],
        coeftest(MRiskPerception.1FE, vcov=vcovHC(MRiskPerception.1FE,type="HC0",cluster="group"))[22:28]),
    caption="Regressions of RiskPerception (per trader) on distribution characteristics. PLoss stands for probability of a loss; ELoss for expected loss in case of a loss, and MinReturn for the minimum return possible.",
    label="tab:Regression1",
    digits=3, booktabs=TRUE, dcolumn=TRUE, sideways=FALSE, longtable=FALSE, float.pos="ht!", use.packages=F
    ) #Creates LaTeX table from regression model



MMarketAvgPrice.1<-lm(AvgPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
summary(MMarketAvgPrice.1)

MMarketAvgPriceLast60.1<-lm(AvgPriceLast60~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
summary(MMarketAvgPriceLast60.1)

MMarketMedianPrice.1<-lm(MedianPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
summary(MMarketMedianPrice.1)

MMarketMedianPriceLast60.1<-lm(MedianPriceLast60~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
summary(MMarketMedianPriceLast60.1)

MVolume.1<-lm(Volume~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
summary(MVolume.1)


# Regression on distributions

# Creates dummy variables for the distributions
{
    Temp1<-ncol(MeanPrices)
    for (Distribution in 1:NumDistributions){
        MeanPrices[,Temp1+Distribution]<-as.integer(MeanPrices$Distribution==Distribution)
    }
    
    colnames(MeanPrices)<-c(colnames(MeanPrices[,1:Temp1]),DistributionText)
}



#R22
MMarketAvgPrice.3<-lm(AvgPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
MMarketAvgPrice.3FE<-plm(AvgPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices), effect="individual", Model="within",index="R.Session")    #Estimates linear session fixed effects panel regression

MMarketMedianPrice.3<-lm(MedianPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
MMarketMedianPrice.3FE<-plm(MedianPrice~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices), effect="individual", Model="within",index="R.Session")    #Estimates linear session fixed effects panel regression

MVolume.3<-lm(Volume~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices))
MVolume.3FE<-plm(Volume~StdDev+Semivar+PLoss+ELoss+Skewness+Kurtosis+MinReturn, data=as.data.frame(MeanPrices), effect="individual", Model="within",index="R.Session")    #Estimates linear session fixed effects panel regression

t.Coef<-length(MMarketAvgPrice.3$coefficients) #Temporary variable for number of coefficients in OLS model



### Mean prices
DistributionMeans<-aggregate(MeanPrices[, 3:ncol(MeanPrices)], list(MeanPrices$Distribution), FUN=mean, na.rm=TRUE)
colnames(DistributionMeans)<-c("Distribution",colnames(DistributionMeans[,2:ncol(DistributionMeans)]))
write.xlsx(DistributionMeans,"WhatIsRiskMarket_DistributionMeans.xlsx", sheetName="DistributionMeans", row.names=F)



# Figure A.4 --------------------------------------------------------------

dev.new("RiskPerception")
{
    par(mfrow=c(4,2), oma=c(1,3.5,1,1), mar=c(5, 0, 1, 1), bg="white")
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$StdDev)
    plot(MeanPrices$StdDev,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Standard deviation, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$Semivar)
    plot(MeanPrices$Semivar,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Semivariance, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))), ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$PLoss)
    plot(MeanPrices$PLoss,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Probability of loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="",  main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$ELoss)
    plot(MeanPrices$ELoss,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Expected loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$Skewness)
    plot(MeanPrices$Skewness,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Skewness, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$Kurtosis)
    plot(MeanPrices$Kurtosis,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Kurtosis, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgRiskPerception~MeanPrices$MinReturn)
    plot(MeanPrices$MinReturn,MeanPrices$AvgRiskPerception, xlab=substitute(paste('Minimum return, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    title(xlab = "", ylab = "Risk perception", outer = TRUE, line = 2)
    dev.copy(png,"WhatIsRiskMarket_Scatter_RiskPerception.png", bg="white", width=1700, height=2300, res=300)
}
dev.off() # Turns off graphics device if even Distribution number



# Figure A.6 --------------------------------------------------------------

dev.new("AvgPrice")
{
    par(mfrow=c(4,2), oma=c(1,3.5,1,1), mar=c(5, 0, 1, 1), bg="white")
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$StdDev)
    plot(MeanPrices$StdDev,MeanPrices$AvgPrice, xlab=substitute(paste('Standard deviation, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$Semivar)
    plot(MeanPrices$Semivar,MeanPrices$AvgPrice, xlab=substitute(paste('Semivariance, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))), ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$PLoss)
    plot(MeanPrices$PLoss,MeanPrices$AvgPrice, xlab=substitute(paste('Probability of loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="",  main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$ELoss)
    plot(MeanPrices$ELoss,MeanPrices$AvgPrice, xlab=substitute(paste('Expected loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$Skewness)
    plot(MeanPrices$Skewness,MeanPrices$AvgPrice, xlab=substitute(paste('Skewness, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$Kurtosis)
    plot(MeanPrices$Kurtosis,MeanPrices$AvgPrice, xlab=substitute(paste('Kurtosis, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPrices$AvgPrice~MeanPrices$MinReturn)
    plot(MeanPrices$MinReturn,MeanPrices$AvgPrice, xlab=substitute(paste('Minimum return, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    title(xlab = "", ylab = "Average price", outer = TRUE, line = 2)
    dev.copy(png,"WhatIsRiskMarket_Scatter_AvgPrice.png", bg="white", width=1700, height=2300, res=300)
}
dev.off() # Turns off graphics device if even Distribution number



# Figure A.5 --------------------------------------------------------------

MeanPricesAfterP1<-MeanPrices[MeanPrices$Period>1,]
dev.new("RiskPerception")
{
    par(mfrow=c(4,2), oma=c(1,3.5,1,1), mar=c(5, 0, 1, 1), bg="white")
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$StdDev)
    plot(MeanPricesAfterP1$StdDev,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Standard deviation, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$Semivar)
    plot(MeanPricesAfterP1$Semivar,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Semivariance, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))), ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$PLoss)
    plot(MeanPricesAfterP1$PLoss,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Probability of loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="",  main="")
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$ELoss)
    plot(MeanPricesAfterP1$ELoss,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Expected loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$Skewness)
    plot(MeanPricesAfterP1$Skewness,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Skewness, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$Kurtosis)
    plot(MeanPricesAfterP1$Kurtosis,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Kurtosis, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(MeanPricesAfterP1$AvgRiskPerception~MeanPricesAfterP1$MinReturn)
    plot(MeanPricesAfterP1$MinReturn,MeanPricesAfterP1$AvgRiskPerception, xlab=substitute(paste('Minimum return, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    title(xlab = "", ylab = "Risk perception", outer = TRUE, line = 2)
    dev.copy(png,"WhatIsRiskMarket_Scatter_RiskPerception_P2-8.png", bg="white", width=1700, height=2300, res=300)
}
dev.off() # Turns off graphics device

par(mfrow=c(1,1))



##### Regressions

### OLS: Asset holdings ~ RiskPerception
#R02

Data$subjects$"R.Assets"<-Data$subjects[,"Assets[1]"]    # Copies Assets[1] into a better variable name
SubjectUnique<-transform(Data$subjects, UniqueSessionSubjectID = as.factor(interaction(R.Session,Subject, drop=TRUE)))   #Generates unique SessionXSubject ID
if(RemoveOutliers){SubjectUnique<-SubjectUnique[!(SubjectUnique$R.Session==7&SubjectUnique$Period==1),]}
for (fR in 1:nrow(SubjectUnique)){
    for (fV in c("FinLitScore")){ #Copies subject characteristics from final period to all periods, since FinLitScore is only elicited in last period, but in ztt, not ztq
        SubjectUnique[fR,fV]<-SubjectUnique[SubjectUnique$R.Session==SubjectUnique$R.Session[fR]&SubjectUnique$Subject==SubjectUnique$Subject[fR]&SubjectUnique$Period==NumPeriods,fV] 
    }
}

MAssetHoldings.Risk<-lm(R.Assets~RiskPerception,SubjectUnique)    # Regresses
MAssetHoldings.Risk.Controls<-lm(R.Assets~RiskPerception+FinLitScore+RiskFinancial+as.numeric(LossAversion.1),SubjectUnique)
MAssetHoldings.Risk.ControlsFemale<-lm(R.Assets~RiskPerception+FinLitScore+RiskFinancial+as.numeric(LossAversion.1)+Female,SubjectUnique)

SubjectUnique$LA<-SPNum(SubjectUnique$LossAversion.1)
SubjectUnique$Fem<-ifelse(SubjectUnique$Female=="Weiblich",1,0)
MAssetHoldings.Risk.ControlsFemale.Dist<-list()

#R24
t.Coef1<-length(MAssetHoldings.Risk$coefficients) #Temporary variable for number of coefficients in OLS model
t.Coef2<-length(MAssetHoldings.Risk.Controls$coefficients) #Temporary variable for number of coefficients in OLS model
t.Coef3<-length(MAssetHoldings.Risk.ControlsFemale$coefficients) #Temporary variable for number of coefficients in OLS model
SubjectUnique$R.UniqueSubject<-interaction(SubjectUnique$R.Session,SubjectUnique$Subject) #Defines unique subject index

#R27
# Table 4 -----------------------------------------------------------------
# Set RemoveOutliers<-F at the beginning of the script for this to work properly.

RegressionOutputAssetHoldingsControls.2 <- texreg(
    list(MAssetHoldings.Risk, MAssetHoldings.Risk.Controls, MAssetHoldings.Risk.ControlsFemale),
    file="WhatIsRiskMarket_Regression_AssetHoldingsControls2.tex",
    stars = c(0.01, 0.05, 0.1),
    custom.model.names=c("Final asset holdings", "Final asset holdings","Final asset holdings"),
    custom.coef.names=c("Intercept","RiskPerception","FinLitScore","FinancialRiskTolerance","FinancialLossTolerance","Female"),
    override.se=list( #Calculates standard errors clustered at the subject level
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[(t.Coef1+1):(2*t.Coef1)],
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk.Controls,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[(t.Coef2+1):(2*t.Coef2)],
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk.ControlsFemale,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[(t.Coef3+1):(2*t.Coef3)]
    ),
    override.pvalues=list( #Calculates p-values for standard errors clustered at the subject level
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[(t.Coef1*3+1):(4*t.Coef1)],
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk.Controls,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[((t.Coef2)*3+1):((t.Coef2)*4)],
        SPClusterSE(Data=NA,Model=MAssetHoldings.Risk.ControlsFemale,ClusterVar1=SubjectUnique$R.UniqueSubject,ClusterVar2=RiskPerceptions$R.Session)[(t.Coef3*3+1):(4*t.Coef3)]
    ),
    custom.note="%stars. Standard errors, clustered at the subject and session levels, in parentheses.",
    caption="OLS regressions of final asset holdings on RiskPerception and control variables",
    label="tab:RegressionAssetHoldings",
    digits=3, booktabs=TRUE, dcolumn=TRUE, sideways=FALSE, longtable=FALSE, float.pos="ht!", use.packages=F
) #Creates LaTeX table from regression model



### Distribution-level graphs

#R05
# Figure 3 ----------------------------------------------------------------

AvgMeanPrices<-aggregate(MeanPrices, by=list(MeanPrices$Distribution), mean)   #Generate AvgMeanPrices to contain distribution averages of data in MeanPrices
    
# Plot AvgPrice vs. AvgRiskPerception
MPlot.AvgPrice_AvgRiskPerception<-lm(AvgMeanPrices$AvgPrice~AvgMeanPrices$AvgRiskPerception)
summary(MPlot.AvgPrice_AvgRiskPerception)
dev.new("DistAvgPriceVsRiskPerception")
{
    par(mfrow=c(1,1),  bg="white")
    plot(x=AvgMeanPrices$AvgRiskPerception,y=AvgMeanPrices$AvgPrice,xlab="Average risk perception", ylab="Average price", pch=1)
    abline(MPlot.AvgPrice_AvgRiskPerception)
    text(x=AvgMeanPrices$AvgRiskPerception[1]+0.4,y=AvgMeanPrices$AvgPrice[1],labels=DistributionText[1],cex=0.8) 
    text(x=AvgMeanPrices$AvgRiskPerception[2]-0.64,y=AvgMeanPrices$AvgPrice[2],labels=DistributionText[2],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[3]-0.62,y=AvgMeanPrices$AvgPrice[3],labels=DistributionText[3],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[4]+0.4,y=AvgMeanPrices$AvgPrice[4],labels=DistributionText[4],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[5]+0.35,y=AvgMeanPrices$AvgPrice[5],labels=DistributionText[5],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[6]-0.56,y=AvgMeanPrices$AvgPrice[6],labels=DistributionText[6],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[7]+0.39,y=AvgMeanPrices$AvgPrice[7],labels=DistributionText[7],cex=0.8)
    text(x=AvgMeanPrices$AvgRiskPerception[8]-0.45,y=AvgMeanPrices$AvgPrice[8],labels=DistributionText[8],cex=0.8)
    dev.copy(png,"WhatIsRiskMarket_Scatter_DistAvgPriceVsRiskPerception.png", bg="white", width=1600, height=1600, res=300)
}
dev.off() # Turns off graphics device



#R07
# Figure 2 ----------------------------------------------------------------

dev.new("AvgRiskPerception")
{
    par(mfrow=c(4,2), oma=c(1,3.5,1,1), mar=c(5, 0, 1, 1), bg="white")
    MPlot <- lm(AvgRiskPerception~StdDev, data=DistributionMeans)
    plot(DistributionMeans$StdDev,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Standard deviation, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Semivar, data=DistributionMeans)
    plot(DistributionMeans$Semivar,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Semivariance, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))), ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~PLoss, data=DistributionMeans)
    plot(DistributionMeans$PLoss,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Probability of loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="",  main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~ELoss, data=DistributionMeans)
    plot(DistributionMeans$ELoss,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Expected loss, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Skewness, data=DistributionMeans)
    plot(DistributionMeans$Skewness,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Skewness, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~Kurtosis, data=DistributionMeans)
    plot(DistributionMeans$Kurtosis,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Kurtosis, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", yaxt="n", main="")
    axis(side=2,labels=F) 
    abline(MPlot)
    MPlot <- lm(AvgRiskPerception~MinReturn, data=DistributionMeans)
    plot(DistributionMeans$MinReturn,DistributionMeans$AvgRiskPerception, xlab=substitute(paste('Minimum return, R'^2,"=",r2),list(r2=round(summary(MPlot)$r.squared,3))),ylab="", main="")
    abline(MPlot)
    title(xlab = "", ylab = "Average RiskPerception", outer = TRUE, line = 2)
    dev.copy(png,"WhatIsRiskMarket_Scatter_DistRiskPerception.png", bg="white", width=1700, height=2300, res=300)
}
dev.off() # Turns off graphics device if even Distribution number



# #R12a
RiskPerceptions[,"PeriodEndWealth"]<-RiskPerceptions$Cash+RiskPerceptions$Assets.1.*(100)*(1+RiskPerceptions$Mean)    #Adds PeriodEndWealth (at asset expected value) to RiskPerceptions
MSubjectProfitNoGeneralRiskLoss<-lm(PeriodEndWealth~FinLitScore+as.numeric(RiskFinancial)+as.numeric(LossAversion.1), data=RiskPerceptions)
MSubjectProfitNoGeneralRiskLossIncRiskPerception<-lm(PeriodEndWealth~RiskPerception+FinLitScore+as.numeric(RiskFinancial)+as.numeric(LossAversion.1), data=RiskPerceptions)
MSubjectProfitNoGeneralRiskLossIncFemale<-lm(PeriodEndWealth~RiskPerception+Female+FinLitScore+as.numeric(RiskFinancial)+as.numeric(LossAversion.1), data=RiskPerceptions)

#R25
# Table 5 -----------------------------------------------------------------

t.Coef1<-length(MSubjectProfitNoGeneralRiskLoss$coefficients) #Temporary variable for number of coefficients in OLS model
t.Coef2<-length(MSubjectProfitNoGeneralRiskLossIncRiskPerception$coefficients) #Temporary variable for number of coefficients in OLS model
t.Coef3<-length(MSubjectProfitNoGeneralRiskLossIncFemale$coefficients) #Temporary variable for number of coefficients in OLS model
texreg(
    list(MSubjectProfitNoGeneralRiskLoss,MSubjectProfitNoGeneralRiskLossIncRiskPerception,MSubjectProfitNoGeneralRiskLossIncFemale),
    file="WhatIsRiskMarket_Regression_Subjects_ProfitNoRiskLoss1.tex", stars = c(0.01, 0.05, 0.1),
    custom.model.names=c("PeriodEndWealth","PeriodEndWealth","PeriodEndWealth"),
    custom.coef.names =c("Intercept","FinLitScore","FinancialRiskTolerance","FinancialLossTolerance","RiskPerception","Female"),
    override.se=list( #Calculates standard errors clustered at the subject level
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLoss,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef1+1):(2*t.Coef1)],
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLossIncRiskPerception,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef2+1):(2*t.Coef2)],
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLossIncFemale,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef3+1):(2*t.Coef3)]
    ),
    override.pvalues=list( #Calculates p-values for standard errors clustered at the subject level
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLoss,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef1*3+1):(4*t.Coef1)],
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLossIncRiskPerception,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef2*3+1):(4*t.Coef2)],
        SPClusterSE(RiskPerceptions,MSubjectProfitNoGeneralRiskLossIncFemale,interaction(RiskPerceptions$R.Session,RiskPerceptions$Subject))[(t.Coef3*3+1):(4*t.Coef3)]
    ),
    custom.note="%stars. Standard errors, clustered at the subject level, in parentheses.",
    caption="OLS regression of subjects' period end wealth on subject characteristics risk and loss preferences (0=very risk/loss averse, 10=not at all risk/loss averse)",
    label="tab:RegressionSubjectsProfit",
    digits=3,booktabs=TRUE, dcolumn=TRUE, sideways=FALSE, longtable=FALSE, float.pos="ht!", use.packages=F
    ) #Creates LaTeX table from regression model



#R14
# Section "Market Data" ---------------------------------------------------

# Subject average of absolute net change in asset holdings per period
mean(abs(Data$subjects$"Assets[1]"-Data$subjects$"InitialAssets[1]"))
mean(abs(Data$subjects$"Assets[1]"[Data$subjects$Female=="Weiblich"]-Data$subjects$"InitialAssets[1]"[Data$subjects$Female=="Weiblich"])) #Only females
mean(abs(Data$subjects$"Assets[1]"[Data$subjects$Female!="Weiblich"]-Data$subjects$"InitialAssets[1]"[Data$subjects$Female!="Weiblich"])) #Only males
t.test(abs(Data$subjects$"Assets[1]"[Data$subjects$Female!="Weiblich"]-Data$subjects$"InitialAssets[1]"[Data$subjects$Female!="Weiblich"]),abs(Data$subjects$"Assets[1]"[Data$subjects$Female=="Weiblich"]-Data$subjects$"InitialAssets[1]"[Data$subjects$Female=="Weiblich"])) #Test for difference
# Subject average of number of transactions per period
sum(Data$transactions$Volume*2)/(11*8*8+1*7*8)    #Sessions*Periods*Subjects - one outlier removed

#Plot change in asset holdings over price
PriceBuySell<-data.frame(Session=rep(1:NumSessions,each=NumSubjects*NumDistributions),Subject=rep(rep(1:NumSubjects,each=NumDistributions),NumSessions),Distribution=rep(1:NumDistributions,NumSessions*NumSubjects)) #Defines PriceBuySell and fills it with Session and Subject IDs)
PriceBuySell<-merge(x=PriceBuySell,y=MeanPrices[,c("R.Session","Distribution","AvgPrice")],by.x=c("Session","Distribution"),by.y=c("R.Session","Distribution")) #Adds average prices
Temp.subjects<-Data$subjects[,c("R.Session","Subject","Distribution","Assets[1]","InitialAssets[1]","Female")] #Copies subset of Data$subjects
Temp.subjects$Change<-Temp.subjects$"Assets[1]"-Temp.subjects$"InitialAssets[1]" #Creates variable of change in asset holdings
PriceBuySell<-merge(x=PriceBuySell,y=Temp.subjects[,c("R.Session","Subject","Distribution","Female","Change")],by.x=c("Session","Distribution","Subject"),by.y=c("R.Session","Distribution","Subject")) #Adds change in asst holdings to PriceBuySell

# 
# ###R17 Calculation of trading volume by gender
Data$transactions<-merge(Data$transactions,Data$offers[,c("R.PeriodID","ID","Offerer","Type")],by.x=c("R.PeriodID","OfferID"),by.y=c("R.PeriodID","ID")) #Adds Offerer ID to Data$transactions
Data$transactions<-merge(Data$transactions,Data$subjects[,c("R.PeriodID","Subject","Female")],by.x=c("R.PeriodID","Offerer"),by.y=c("R.PeriodID","Subject")) #Adds offerer gender to Data$transactions
colnames(Data$transactions)[ncol(Data$transactions)] <- "OffererGender" #Renames
Data$transactions<-merge(Data$transactions,Data$subjects[,c("R.PeriodID","Subject","Female")],by.x=c("R.PeriodID","AccepterID"),by.y=c("R.PeriodID","Subject")) #Adds accepter gender to Data$transactions
colnames(Data$transactions)[ncol(Data$transactions)] <- "AccepterGender" #Renames

#R19
# Section "Individual Trading Behavior and Profits" -----------------------

#Determine prices at which women and men buy and sell
print("Women buy")
mean(Data$transactions[(Data$transactions$OffererGender=="Weiblich"&Data$transactions$Type==1)|(Data$transactions$AccepterGender=="Weiblich"&Data$transactions$Type==-1),]$Price)
print("Women sell")
mean(Data$transactions[(Data$transactions$OffererGender=="Weiblich"&Data$transactions$Type==-1)|(Data$transactions$AccepterGender=="Weiblich"&Data$transactions$Type==1),]$Price)
print("Men buy")
mean(Data$transactions[(Data$transactions$OffererGender=="Männlich"&Data$transactions$Type==1)|(Data$transactions$AccepterGender=="Männlich"&Data$transactions$Type==-1),]$Price)
print("Men sell")
mean(Data$transactions[(Data$transactions$OffererGender=="Männlich"&Data$transactions$Type==-1)|(Data$transactions$AccepterGender=="Männlich"&Data$transactions$Type==1),]$Price)



#R21 Calculates summary statistics by gender
# Section "Individual Trading Behavior and Profits" -----------------------

Temp.subjects<-Data$subjects #Creates temporary copy of Data$subjects
Temp.subjects$Female<-Recode(Temp.subjects$Female,recode="'Weiblich'=1;'Männlich'=0") #Recodes Female to numeric
GenderAverages<-data.frame(Female=c(0,1,"p")) #Prepares dataframe to hold gender averages and p-values of t-test
GenderAverages[1:2,"N"]<-c(sum((1-as.numeric(Temp.subjects$Female)[Temp.subjects$Female==0])),sum(as.numeric(Temp.subjects$Female[Temp.subjects$Female==1]))) #Fills in gender N
GenderAverages[1:2,"N"]<-c(length(Data$subjects$Female[Data$subjects$Period==8&Data$subjects$Female!="Weiblich"]),length(Data$subjects$Female[Data$subjects$Period==8&Data$subjects$Female=="Weiblich"]))
 #Fills in gender N
for (Var in c("FinLitScore","RiskPerception","RiskGeneral","RiskFinancial","LossAversion","LossAversion.1")){
    GenderAverages[1:2,Var]<-by(as.numeric(Temp.subjects[Temp.subjects$Period==8,Var]),INDICES=Temp.subjects$Female[Temp.subjects$Period==8],FUN=mean) #Fills in gender averages
    GenderAverages[3,Var]<-t.test(as.numeric(Temp.subjects[Temp.subjects$Period==8&Temp.subjects$Female==1,Var]),as.numeric(Temp.subjects[Temp.subjects$Period==8&Temp.subjects$Female==0,Var]))$p.value #Adds p-value
    print(t.test(as.numeric(Temp.subjects[Temp.subjects$Period==8&Temp.subjects$Female==1,Var]),as.numeric(Temp.subjects[Temp.subjects$Period==8&Temp.subjects$Female==0,Var]))) #Prints t-test
}
GenderAverages[,"ChangeAssetHoldings"]<-c( #Adds net changes in asset holdings
    mean(PriceBuySell$Change[PriceBuySell$Female=="Männlich"]),
    mean(PriceBuySell$Change[PriceBuySell$Female=="Weiblich"]),
    t.test(PriceBuySell$Change[PriceBuySell$Female=="Weiblich"],PriceBuySell$Change[PriceBuySell$Female=="Männlich"])$p.value
    )
GenderAverages


#R29
for (Session in 1:NumSessions){
    MeanPrices$PropFemale[MeanPrices$R.Session==Session]<-mean(Data$subjects$Female[Data$subjects$R.Session==Session&Data$subjects$Period==8]=="Weiblich") #Adds proportion of female subjects to MeanPrices
    for (Characteristic in c("FinLitScore","Age","Patience","RiskGeneral","RiskFinancial","LossAversion")){ #Adds other characteristics to MeanPrices
        MeanPrices[MeanPrices$R.Session==Session,paste("Avg",Characteristic,sep="")]<-mean(as.numeric(as.character(Data$subjects[Data$subjects$R.Session==Session&Data$subjects$Period==8,Characteristic])))
    }
}


#R31
# Table 3 -----------------------------------------------------------------

Temp.1<-aggregate(Data$subjects$RiskPerception,list(SPNum(Data$subjects$R.Session),SPNum(Data$subjects$Period)),sd) #Generates variable holding standard deviation and range of RiskPerception per R.Session and Period
Temp.2<-aggregate(Data$subjects$RiskPerception,list(SPNum(Data$subjects$R.Session),SPNum(Data$subjects$Period)),range)
Temp.1$RiskPerception.Range<-Temp.2[,3][,2]-Temp.2[,3][,1] #Calculates difference between highest and lowest RiskPerception value per R.Session and Period
colnames(Temp.1)<-c("R.Session","Period","RiskPerception.SD","RiskPerception.Range")
MeanPrices<-merge(MeanPrices,Temp.1,by=c("R.Session","Period"),all.x=T,all.y=F) #Merges newly calculates variables into MeanPrices
MeanPrices$Price.SD<-aggregate(Data$transactions$Price,list(SPNum(Data$transactions$R.Session),SPNum(Data$transactions$Period)),sd)[,3] #Adds price standard deviation to MeanPrices

MMarketAvgPrice.RPMult<-lm(AvgPrice~AvgRiskPerception+RiskPerception.SD, data=as.data.frame(MeanPrices))
MMarketAvgPrice.RP1Mult<-lm(AvgPrice~AvgRiskPerception+RiskPerception.SD+PropFemale+AvgFinLitScore+AvgRiskFinancial+AvgLossAversion, data=as.data.frame(MeanPrices))

MMarketMedianPrice.RPMult<-lm(MedianPrice~AvgRiskPerception+RiskPerception.SD, data=as.data.frame(MeanPrices))
MMarketMedianPrice.RP1Mult<-lm(MedianPrice~AvgRiskPerception+RiskPerception.SD+PropFemale+AvgFinLitScore+AvgRiskFinancial+AvgLossAversion, data=as.data.frame(MeanPrices))

MVolume.RPMult<-lm(Volume~AvgRiskPerception+RiskPerception.SD, data=as.data.frame(MeanPrices))
MVolume.RP1Mult<-lm(Volume~AvgRiskPerception+RiskPerception.SD+PropFemale+AvgFinLitScore+AvgRiskFinancial+AvgLossAversion, data=as.data.frame(MeanPrices))

MMarketPriceSD.RPMult<-lm(Price.SD~AvgRiskPerception+RiskPerception.SD, data=as.data.frame(MeanPrices))
MMarketPriceSD.RP1Mult<-lm(Price.SD~AvgRiskPerception+RiskPerception.SD+PropFemale+AvgFinLitScore+AvgRiskFinancial+AvgLossAversion, data=as.data.frame(MeanPrices))

t.Coef<-length(MMarketAvgPrice.RPMult$coefficients) #Temporary variable for number of coefficients in OLS model
t.Coef1<-length(MMarketAvgPrice.RP1Mult$coefficients) #Temporary variable for number of coefficients in OLS model

RegressionOutputRPMult.LaTeX <- texreg( #AvgPrice, MedianPrice, Volume ~ AvgRiskPerception, clustered SEs
    list(MMarketAvgPrice.RPMult,MMarketAvgPrice.RP1Mult,MMarketMedianPrice.RPMult,MMarketMedianPrice.RP1Mult,MVolume.RPMult,MVolume.RP1Mult,MMarketPriceSD.RPMult,MMarketPriceSD.RP1Mult),
    file="WhatIsRiskMarket_Regression_RegRPMult.tex", stars = c(0.01, 0.05, 0.1),
    custom.model.names=c("AvgPrice","AvgPrice","MedianPrice","MedianPrice","Volume","Volume","PriceSD","PriceSD"),
    custom.note="%stars. Standard errors, clustered at the session level, in parentheses.",
    caption="OLS regressions of average and median prices as well as trading volume and price volatility on average risk perception, standard deviation of risk perception, and on market-level averages of subject characteristics.",
    override.se=list( #Calculates standard errors clustered at the session level
        SPClusterSE(as.data.frame(MeanPrices),MMarketAvgPrice.RPMult,MeanPrices$R.Session)[(t.Coef+1):(2*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketAvgPrice.RP1Mult,MeanPrices$R.Session)[(t.Coef1+1):(2*t.Coef1)],
        
        SPClusterSE(as.data.frame(MeanPrices),MMarketMedianPrice.RPMult,MeanPrices$R.Session)[(t.Coef+1):(2*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketMedianPrice.RP1Mult,MeanPrices$R.Session)[(t.Coef1+1):(2*t.Coef1)],
        
        SPClusterSE(as.data.frame(MeanPrices),MVolume.RPMult,MeanPrices$R.Session)[(t.Coef+1):(2*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MVolume.RP1Mult,MeanPrices$R.Session)[(t.Coef1+1):(2*t.Coef1)],
        
        SPClusterSE(as.data.frame(MeanPrices),MMarketPriceSD.RPMult,MeanPrices$R.Session)[(t.Coef+1):(2*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketPriceSD.RP1Mult,MeanPrices$R.Session)[(t.Coef1+1):(2*t.Coef1)]
    ),
    override.pvalues=list( #Calculates p-values for standard errors clustered at the session level
        SPClusterSE(as.data.frame(MeanPrices),MMarketAvgPrice.RPMult,MeanPrices$R.Session)[(t.Coef*3+1):(4*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketAvgPrice.RP1Mult,MeanPrices$R.Session)[(t.Coef1*3+1):(4*t.Coef1)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketMedianPrice.RPMult,MeanPrices$R.Session)[(t.Coef*3+1):(4*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketMedianPrice.RP1Mult,MeanPrices$R.Session)[(t.Coef1*3+1):(4*t.Coef1)],
        SPClusterSE(as.data.frame(MeanPrices),MVolume.RPMult,MeanPrices$R.Session)[(t.Coef*3+1):(4*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MVolume.RP1Mult,MeanPrices$R.Session)[(t.Coef1*3+1):(4*t.Coef1)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketPriceSD.RPMult,MeanPrices$R.Session)[(t.Coef*3+1):(4*t.Coef)],
        SPClusterSE(as.data.frame(MeanPrices),MMarketPriceSD.RP1Mult,MeanPrices$R.Session)[(t.Coef1*3+1):(4*t.Coef1)]
    ),
    label="tab:RegressionRPMulta",
    digits=3, booktabs=TRUE, dcolumn=TRUE, sideways=T, longtable=FALSE, float.pos="ht!", use.packages=F
) #Creates LaTeX table from regression model



#R30
# Section "Individual Trading Behavior and Profits" -----------------------

mean(MeanPrices$AvgPrice) #Average price
length(Data$subjects$`Assets[1]`[Data$subjects$`Assets[1]`==0]) #Number of subject x period combinations with final asset holdings of zero