% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tdm_dt.max.R
\name{tdm_dt.max}
\alias{tdm_dt.max}
\title{Calculate zero-flow conditions}
\usage{
tdm_dt.max(input, methods = c("pd","mw","dr","ed"),
zero.end = 8*60,
zero.start =  1*60,
interpolate = FALSE, det.pd = TRUE,
max.days = 7,
ed.window = 2*60,
vpd.input,
sr.input,
sel.max,
criteria = c(sr = 30, vpd = 0.1, cv = 0.5),
df = FALSE)
}
\arguments{
\item{input}{An \code{\link{is.trex}}-compliant object of \eqn{K} values containing
a timestamp and a value column.}

\item{methods}{Character vector of the requested \eqn{\Delta T_{max}}{\Delta Tmax} methods.
Options include \code{“pd”} (predawn), \code{“mw”} (moving-window), \code{“dr”} (double regression),
and \code{“ed”} (environmental-dependent; default= \code{c(“pd”, “mw”, “dr”)}).}

\item{zero.end}{Numeric, optionally defines the end of the predawn period.
Values should be in minutes (e.g. predawn conditions until 08:00 = 8 * 60).
When not provided, the algorithm will automatically analyse the cyclic behaviour
of the data and define the day length.}

\item{zero.start}{Numeric, optionally defines the beginning of the predawn period.
Values should be in minutes (e.g., 01:00 = 1*60).}

\item{interpolate}{Logical: if \code{TRUE}, detected \eqn{\Delta T_{max}}{\Delta Tmax} values are linearly
interpolated. If \code{FALSE}, constant \eqn{\Delta T_{max}}{\Delta Tmax} values will be selected daily
(default = \code{FALSE}).}

\item{det.pd}{Logical; if \code{TRUE} and no \code{zero.end} and \code{zero.start} values are provided,
predawn \eqn{\Delta T_{max}}{\Delta Tmax} will be determined based on cyclic behaviour of the entire
time-series (default = \code{TRUE}).}

\item{max.days}{Numeric, defines the number of days which the \code{mw} and \code{dr}
methods will consider for their moving windows.}

\item{ed.window}{Numeric, defines the length of the period considered for assessing the
environmental conditions and stable \eqn{\Delta T_{max}}{\Delta Tmax} values.}

\item{vpd.input}{An \code{\link{is.trex}}-compliant object (\code{zoo} time-series or \code{data.frame})
with a timestamp and a value column containing the vapour pressure deficit (\emph{vpd}; in kPa)
with the same temporal extent and time steps as the input data.}

\item{sr.input}{An \code{\link{is.trex}}-compliant object (\code{zoo} time-series or \code{data.frame})
with a timestamp and a value column the solar radiation data (\emph{sr}; e.g., global radiation or \emph{PAR}).}

\item{sel.max}{Optional \code{zoo} time-series or \code{data.frame} with the specified \eqn{\Delta T_{max}}{\Delta Tmax}.
This option is included to change predawn \eqn{\Delta T_{max}}{\Delta Tmax} values selected with the \code{ed} method.}

\item{criteria}{Numeric vector, thresholds for the \code{ed} method.
Thresholds should be provided for all environmental data included in the function
(e.g. \code{c(sr = 30, vpd = 0.1)}; coefficient of variation, \emph{cv} = 0.5)}

\item{df}{Logical; if \code{TRUE}, output is provided in a \code{data.frame}
format with a timestamp and a value (\eqn{\Delta T} or \eqn{\Delta V}) column.
If \code{FALSE}, output is provided as a \code{zoo} object (default = \code{FALSE}).}
}
\value{
A named \code{list} of \code{zoo} time series or \code{data.frame}
objects in the appropriate format for further processing.
List items include:
\describe{
 \item{max.pd}{\eqn{\Delta T_{max}}{\Delta Tmax} time series as determined by the \code{pd} method.}
 \item{max.mw}{\eqn{\Delta T_{max}}{\Delta Tmax} time series as determined by the \code{mw} method.}
 \item{max.dr}{\eqn{\Delta T_{max}}{\Delta Tmax} time series as determined by the \code{dr} method.}
 \item{max.ed}{\eqn{\Delta T_{max}}{\Delta Tmax} time series as determined by the \code{ed} method.}
 \item{daily_max.pd}{daily predawn \eqn{\Delta T_{max}}{\Delta Tmax} as determined by \code{pd}.}
 \item{daily_max.mw}{daily predawn \eqn{\Delta T_{max}}{\Delta Tmax} as determined by \code{mw}.}
 \item{daily_max.dr}{daily predawn \eqn{\Delta T_{max}}{\Delta Tmax} as determined by \code{dr}.}
 \item{daily_max.ed}{daily predawn \eqn{\Delta T_{max}}{\Delta Tmax} as determined by \code{ed}.}
 \item{all.pd}{exact predawn \eqn{\Delta T_{max}}{\Delta Tmax} values detected with \code{pd}.}
 \item{all.ed}{exact predawn \eqn{\Delta T_{max}}{\Delta Tmax} values detected with \code{ed}.}
 \item{input}{\eqn{\Delta T} input data.}
 \item{ed.criteria}{\code{data.frame} of the applied environmental and variability criteria used within \code{ed}.}
 \item{methods}{\code{data.frame} of applied methods to detect \eqn{\Delta T_{max}}{\Delta Tmax}.}
 \item{k.pd}{\eqn{K} values calculated by using the \code{pd} method.}
 \item{k.mw}{\eqn{K} values calculated by using the \code{mw} method.}
 \item{k.dr}{\eqn{K} values calculated by using the \code{dr} method.}
 \item{k.ed}{\eqn{K} values calculated by using the \code{ed} method.}
 }
}
\description{
Determine zero flow conditions (\eqn{\Delta T_{max}}{\Delta Tmax}; or \eqn{\Delta V_{max}}{\Delta Vmax})
according to four methods; namely,
 1) predawn (\code{pd}),
 2) moving-window (\code{mw}),
 3) double regression (\code{dr}),
 and 4) environmental-dependent (\code{ed}) as applied in Peters \emph{et al.} 2018.
 The function can provide (\eqn{\Delta T_{max}}{\Delta Tmax} values and subsequent \emph{K} values for all methods.
 All outputs are provided in a \code{list} including the input data and calculated outputs.
}
\details{
There are a variety of methods which can be applied to determine zero-flow conditions.
 Zero-flow conditions are required to calculate \eqn{K = (\Delta T_{max} - \Delta T) / \Delta T}{K = (\Delta Tmax - \Delta T) / \Delta T}.
 A detailed description on the methods is provided by Peters \emph{et al.} (2018).
 In short, the \code{pd} method entails the selection of daily maxima occurring prior to sunrise.
 This method assumes that during each night zero-flow conditions are obtained.
 The algorithm either requires specific times within which it searches for a maximum,
 or it analyses the cyclic pattern within the data and defines this time window.
 The \code{mw} method uses these predawn \eqn{\Delta T_{max}}{\Delta Tmax} values
 and calculates the maximum over a multi-day moving time-window (e.g., 7 days).
 The \code{dr} methods is applied by calculating the mean over predawn \eqn{\Delta T_{max}}{\Delta Tmax}
 with a specified multi-day window, removing all values below the mean,
 and calculating a second mean over the same multi-day window and uses these values as \eqn{\Delta T_{max}}{\Delta Tmax}.
 The \code{ed} method selects predawn \eqn{\Delta T_{max}}{\Delta Tmax} values based upon 2-hour averaged environmental
 conditions prior to the detected time for the predawn \eqn{\Delta T_{max}}{\Delta Tmax}.
 These environmental conditions include low vapour pressure deficit (in \eqn{kPa}) and low solar irradiance
 (e.g., in W m-2). In addition, the coefficient of variation (cv) of predawn \eqn{\Delta T_{max}}{\Delta Tmax} are scanned for low values to
 ensure the selection of stable zero-flow conditions.
}
\examples{
\dontrun{
#perform Delta Tmax calculations
raw <- is.trex(example.data(type = "doy"),
     tz = "GMT", time.format = "\%H:\%M", solar.time = TRUE,
     long.deg = 7.7459, ref.add = FALSE)
input <- dt.steps(input = raw, start = "2014-05-08 00:00",
         end = "2014-07-25 00:50", time.int = 15, max.gap = 60,
         decimals = 6, df = FALSE)
input[which(input<0.2)]<- NA
output.max <- tdm_dt.max(input, methods = c("pd", "mw", "dr"),
                 det.pd = TRUE, interpolate = FALSE,
                 max.days = 10, df = FALSE)

str(output.max)

plot(output.max$input, ylab = expression(Delta*italic("V")))

lines(output.max$max.pd, col = "green")
lines(output.max$max.mw, col = "blue")
lines(output.max$max.dr, col = "orange")

points(output.max$all.pd, col = "green", pch = 16)

legend("bottomright", c("raw", "max.pd", "max.mw", "max.dr"),
        lty = 1, col = c("black", "green", "blue", "orange") )

}
}
