% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tdm_damp.R
\name{tdm_damp}
\alias{tdm_damp}
\title{Signal dampening correction}
\usage{
tdm_damp(input, k.threshold = 0.05, make.plot = TRUE, df = FALSE)
}
\arguments{
\item{input}{An \code{\link{is.trex}}-compliant object (\code{zoo} vector, \code{data.frame}) of \eqn{K} values containing
a timestamp and a value column.}

\item{k.threshold}{Numeric, the threshold below which daily maximum \eqn{K} values should not be considered (default = 0.05).}

\item{make.plot}{Logical; if \code{TRUE}, a plot is generated presenting the correction curve and the \eqn{K} time series.}

\item{df}{Logical; If \code{TRUE}, output is provided in a \code{data.frame} format
 with a timestamp and a value column. If \code{FALSE}, output
 is provided as a \code{zoo} vector object (default = FALSE).


 @details The function fits a correction curve for signal dampening (e.g., due to wounding)
 according to Peters \emph{et al.} (2018). A sensor specific function is fitted to daily maximum
 \eqn{K} values (considering a minimum cut-off threshold; see \code{k.threshold}). Dependent variables
 for the function include seasonality (DOY) and days since installation (\eqn{t}).
 First, seasonal effects are removed by correcting the \eqn{K} series (residuals; \eqn{Kresid})
 to a second-order polynomial with DOY. These residuals are then used within a
 non-linear model:
 \deqn{K_{resid} = (a + b * t)/(1 + c * t + d * t^{2})}{Kresid = (a + b * t)/(1 + c * t + d * t2)}

 The fitted parameters for \eqn{t} (with \eqn{a}, \eqn{b}, \eqn{c} and \eqn{d}) are used to
 correct \eqn{K} and scale it to the maximum within the first year of installation.
 \strong{Note, that the stability of the fit has to be visually inspected before using the output data}.}
}
\value{
A \code{zoo} object or \code{data.frame} in the appropriate
 format for other functionalities.
 See \code{\link{tdm_dt.max}} output specifications.
 All \eqn{K} values for each method are provided when an \code{\link{is.trex}}-object was used as input.
 If an individual time series was provided for input with \eqn{K} values an alternative output is given:

 \describe{
   \item{k.cor}{corrected \eqn{K} values according to the correction curve.}
   \item{k}{\eqn{K} values provided as input.}
   \item{damp.mod}{\code{data.frame} with the coefficients of the correction curve.}

 }
}
\description{
When long-term \eqn{K} time series (~3 years) are provided, one can perform
 a signal dampening correction (when sensors were not re-installed;
 see Peters \emph{et al.} 2018). Applying the signal dampening
 correction requires  visually inspecting the correction curve
 (see \code{make.plot = TRUE}). The correction curve is constructed with
 the day since installation and the day of year (DOY) to account for seasonal changes in \eqn{K}
 values. The function returns corrected \eqn{K} values and the applied correction curve.
}
\examples{
\dontrun{
 #correct for dampening of the signal
raw   <-
  is.trex(
    example.data(type = "doy"),
    tz = "GMT",
    time.format = "\%H:\%M",
    solar.time = TRUE,
    long.deg = 7.7459,
    ref.add = FALSE
  )
input <-
  dt.steps(
    input = raw,
    time.int = 15,
    max.gap = 60,
    decimals = 6,
    df = FALSE
  )
input[which(input < 0.2)] <- NA
input <-
  tdm_dt.max(
    input,
    methods = c("pd", "mw", "dr"),
    det.pd = TRUE,
    interpolate = FALSE,
    max.days = 10,
    df = FALSE
  )
output.data <- tdm_damp(input,
                    k.threshold = 0.05,
                    make.plot = TRUE,
                    df = FALSE)
str(output.data)
head(output.data[["k.dr"]])
plot(output.data[["k.dr"]], ylab = expression(italic("K")))
}

}
\references{
Peters RL, Fonti P, Frank DC, Poyatos R, Pappas C, Kahmen A, Carraro V,
Prendin AL, Schneider L, Baltzer JL, Baron-Gafford GA, Dietrich L, Heinrich I,
Minor RL, Sonnentag O, Matheny AM, Wightman MG, Steppe K. 2018.
Quantification of uncertainties in conifer sap flow measured with the thermal
dissipation method. New Phytologist 219:1283-1299 \url{doi: 10.1111/nph.15241}
}
