! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
PROGRAM RADDEN
!Calculates radial densities of selected orbitals from a Molden input file or calculates radial densities for density matrices read from a file.
!Both: contracted gaussians or symmetry adapted functions can be used as basis functions
!Zdenek Masin (z.masin@open.ac.uk), James Munro
use global_utils, only: print_ukrmol_header
use saved_vars                                              !common variables used in integration of the orbitals and basis functions
use constants                                               !pi, conversion factors, ...
use precisn                                                 !defines the precision constant 'wp' for real numbers
use basis_atoms                                             !the common variables 'basis' and 'orbital'
use basis_fns, only: NORMALIZE_CGTO, NORMALIZE_SABF         !routines for normalization of basis functions
use basis_fns_num, only: CHECK_GTO_NORMS, CHECK_SABF_NORMS  !debugging routines for numerical integration of the overlap functions between basis functions
use basis_fns_num, only:CALC_DENSITIES                      !this routine performs the radial integration of densities
use molden, only: CALC_DM, READ_MOLDEN                      !calculate density matrices from the Molden orbitals; Read basis set from the Molden file 
use rmat_rd, only: READ_RMAT                                !read basis set from the R-matrix integrals file fort.2

IMPLICIT NONE

!auxiliary variables
INTEGER :: io, ifail, molunit, i

!namelist variables (explained below)
INTEGER :: norb, req_dm, dmunit, basunit, whichdm, dm_list(1:max_orb)
LOGICAL :: iprint, ismolden, dmfromf, ilist

CHARACTER(len=line_len) :: molden_file, header_base

REAL(kind=wp) :: rstart, rfinish 

NAMELIST/INPUT/whichdm, req_dm, dm_list, iprint, ilist, rstart, rfinish !common input always read
NAMELIST/MOLDENIN/molden_file, header_base, dmunit, dmfromf             !namelist read if molden file is to be processed
NAMELIST/RMATIN/basunit, dmunit                                         !namelist read if density matrices are to be read from a file

CALL print_ukrmol_header(6)
WRITE (*,'(//,10X,"RADDEN: Program for calculating radial charge densities of molecular orbitals/electronic states. (10/12/2010)")')

!defaults and intialization
r=0.0_wp                                     
theta=0.0_wp
phi=0.0_wp
molunit=1                                 !unit number for the Molden input file
ismolden=.true.                           !we are processing a Molden input file
!============defaults for the namelist INPUT:
rstart=0.0_wp                             !radial interval in a.u. for which the integration will be carried out
rfinish=20.0_wp
whichdm=1          !1=Molden file will be processed by default
                   !2=Density matrix generated by the MPOUTRD or by the R-matrix SWSCF programs will be read, along with the basis set from the fort.2 unit
                   !3=Density matrix generated by the R-matrix DENPROP prog. will be read, along with basis from fort.2 unit
                   !3: NOT IMPLEMENTED YET (see the subroutines on the bottom of the module 'rmat_rd')
req_dm=-1                                 !set it to <0 to get all available orbitals
ilist=.false.                             !do we want only some oribtals/density matrices from the input? If 'true' then orbitals from 'dm_list' will be read
dm_list=1                                 !sequence numbers of density matrices/orbitals to be read from the input file. Not used if ilist.eq.false
iprint=.false.                            !detailed printout not requested. Set it to true for debugging.
!===========defaults for the namelist MOLDENIN
dmunit=450                                !default input/output unit for the density matrices
molden_file=''                            !path to the Molden file
header_base="Density matrix for orbital " !default header of the density matrices if they are to be written to the dmunit, otherwise not used
dmfromf=.false.                           !density matrices will be calulated from the Molden file, otherwise read from the dmunit
!===========defaults for the namelist RMATIN
basunit=2                                 !fort.2: default unit for basis set information
dmunit=450                                !default input unit for the density matrices
!xxxxxxxxxxx

!read the common namelist INPUT
READ(5,nml=INPUT)

!check input values and decide which namelist should be read next
IF (whichdm .eq. 1) THEN
   READ(5,nml=MOLDENIN)
   ismolden = .true.
   IF (molden_file .eq. '') THEN
      STOP "Path to the Molden file not set. Program terminated."
   END IF
   IF (dmfromf) THEN
      WRITE (*,'(/,5X,a,/)') "Density matrices will be read from an external file."
   ELSE
      WRITE (*,'(/,5X,a,/)') "Density matrices will be calculated from orbitals on the Molden file."
   END IF
ELSE IF (whichdm .eq. 2) THEN
   READ(5,nml=RMATIN)
   ismolden = .false.
   IF (basunit .le. 0) THEN
      STOP "Illegal value of the basunit variable. Program terminated."
   END IF
ELSE IF (whichdm .eq. 3) THEN
   ismolden = .false.
   IF (basunit .le. 0) THEN
      STOP "Illegal value of the basunit variable. Program terminated."
   END IF
   STOP "Reading of density matrices from the DENPROP output not implemented yet. Program terminated."
ELSE
   STOP "Wrong input value of the variable 'whichdm' (currently 1-2 are accepted). Program terminated."
END IF

IF (iprint) THEN
   WRITE (*,'(/,5X,a,/)') "Detailed printout and checking of normalization of basis functions requested."
END IF

!check the input values common to all namelists
IF (req_dm .eq. 0) THEN
   STOP "No density matrices/orbitals required. Program terminated."
END IF

DO i=1,size(dm_list)
   IF (dm_list(i) .le. 0) STOP "Invalid value(s) in the dm_list. Program terminated."
END DO

IF (req_dm .le. 0 .and. ilist) THEN
   STOP "Inconsistent input. All density matrices required, but a list of density matrices specified as well. &
        &Decide what you want. Program terminated."
END IF

IF (dmunit .le. 0) THEN
   STOP "Illegal value (< 0) of the dmunit variable. Program terminated."
END IF

IF (rstart.lt.0.0_wp) THEN
   rstart = 0._wp
   WRITE(*,'(a,e25.15)') " R_start out of range (< 0), reseting to ",rstart
endif
  
IF (rfinish < rstart) THEN
   rfinish = 20.0_wp
   WRITE(*,'(a,e25.15)') " R_finish out of range (R_finish < R_start), reseting to ",rfinish
END IF

IF (ismolden) THEN   !are we processing the Molden file?
   iscgto=.true.                 !then the basis is contracted gaussians
   CALL OPEN_MOLDEN(molunit,molden_file,dmunit,dmfromf)
   CALL READ_MOLDEN(molunit,molden_file,dmfromf) !read the basis set information from the Molden file
ELSE
   iscgto=.false.                !then the basis is the R-matrix symmetry adapted basis set
   CALL READ_RMAT(basunit)       !read the basis set information from the fort.2 file
END IF

!normalize the contracted gaussians
CALL NORMALIZE_CGTO(iprint)

!normalize the symmetry-adapted functions if we are processing the R-matrix style density matrix
IF (.not. ismolden) THEN
   CALL NORMALIZE_SABF(iprint)
END IF

!Check that we have normalized everything correctly (if detailed printout requested)
IF (iprint) THEN
   CALL CHECK_GTO_NORMS 
   IF (.not. ismolden) CALL CHECK_SABF_NORMS
END IF

IF (ismolden .and. .not. dmfromf) THEN !are we processing the Molden file and not reading the density matrix from a file? 
      !construct density matrices from the Molden file for the required orbitals (and save them to a file)
      CALL CALC_DM(molunit,req_dm,ilist,dm_list,norb,dmunit,iprint,header_base)
   ELSE
      !(if whichdm=4) read density matrix produced by Molpro
      IF (whichdm .eq. 2) CALL READ_DM(req_dm,ilist,dm_list,norb,dmunit,iscgto,ismolden) !read density matrices from the file
      !IF (whichdm .eq. 3) CALL READ_RMAT_DM(basis) !read density matrix from the denprop fort.60 unit - NOT IMPLEMENTED YET
END IF

!Integrate charge densities of the required orbitals and output the radial values
CALL CALC_DENSITIES(req_dm,rstart,rfinish)

CLOSE(molunit)
CLOSE(basunit)
DEALLOCATE(orbital)
DEALLOCATE(basis % alpha, basis % contractions, basis % at_coord, basis % overlaps_cgto, basis % at_num, &
           basis % at_ind, basis % contr_alpha, basis % ang_exp, basis % no_contr, basis % at_sym, basis % prefact)

IF (.not. ismolden) DEALLOCATE(basis%no_sabf,basis%sa_contr,basis%contr_ind)

WRITE(*,'(/5X,a)') "RADDEN finished operation."
WRITE(*,'(5X,a)')  "==========================="

CONTAINS !mainly routines for maniuplation of the file containing the density matrices

  SUBROUTINE OPEN_MOLDEN(mldu,mld_file,dmunit,dmfromf)
  !Opens the Molden file and prints out some basic info
  IMPLICIT NONE
  !INPUT:
  INTEGER :: mldu,dmunit               !unit number for the Molden file;unit number for the density matrices on output
  LOGICAL :: dmfromf                   !do we want to have density matrices read from a file?
  CHARACTER(len=line_len) :: mld_file  !path to the Molden file
  !auxiliary:
  INTEGER :: io

  OPEN(unit=mldu,file=trim(mld_file),form='formatted',iostat=io,status='old')
  
  IF (io .ne. 0) THEN
     STOP "Problem opening the Molden file. Program terminated."
  END IF

  WRITE (*,'(5X,"============================================",/)')
  WRITE(*,'(5X,"Molden input file to be processed: ",a,/)') trim(mld_file)
  IF (.not. dmfromf) THEN
     WRITE(*,'(5X,"Density matrices will be written out to unit: ",i5,/)') dmunit
  ELSE
     WRITE(*,'(5X,"Density matrices will be read from the unit: ",i5,/)') dmunit
  END IF
  WRITE (*,'(5X,"============================================",/)')

  END SUBROUTINE OPEN_MOLDEN
  
  SUBROUTINE READ_DM(req_dm,ilist,dm_list,norb,dmunit,iscgto,ismolden)
  !reads density matrices stored on the dmunit file to the common variables 'orbital(:)%dm' containing the density matrices to be processed later
  use constants
  use precisn
  use basis_atoms
  use basis_fns, only: OVERLAP_CGTO, OVERLAP_SABF
  use molden
  IMPLICIT NONE
  !INPUT:
  INTEGER :: dmunit                   !unit number for the file with density matrices
  INTEGER :: req_dm                   !how many density matrices we want
  INTEGER :: norb                     !number of orbitals/density matrices on the input file
  INTEGER :: dm_list(1:max_orb)       !sequence numbers of the density matrices/orbitals we want to read in
  LOGICAL :: iscgto, ismolden, ilist  !is basis set contracted gaussians;are we processing a Molden file;do want to read onyl densities specified in dm_list
  !auxiliary:
  INTEGER :: i, j, k, m, n, io, d_dm 
  REAL(kind=wp) :: tmp, tot_ch
  CHARACTER(len=line_len) :: title, title_dm

  OPEN(unit=dmunit,form='formatted',iostat=io,status='old') !output for the density matrices
  IF (io .ne. 0) THEN
     STOP "Problem opening the DMUINT for density matrices. Program terminated."
  END IF

  READ(dmunit, '(a)') title                                  !type of the basis set corresponding to the saved density matrix elements
  WRITE (*,'(/,5X,"Type of basis set corresponding to the saved density matrix elements: ",a)') adjustl(trim(title))

  !iscgto is needed in integration of densities in the function ORB_DEN_phi
  IF (adjustl(trim(title)) .eq. trim(BFTYPE(2))) THEN !'Contracted GTO basis'
     iscgto = .true.  !the basis is contracted gtos
  ELSE IF (adjustl(trim(title)) .eq. trim(BFTYPE(1))) THEN !'Symmetry-adapted basis'
     iscgto = .false. !the basis is symmetry adapted functions
  ELSE
     STOP "Unknown density matrices on input. Program terminated."
  END IF

  IF (iscgto .neqv. ismolden) THEN
     STOP "Density matrices on input and the used basis set are not compatible. Program terminated."
  ELSE
     WRITE(*,'("OK")')
  END IF

  READ(dmunit, '(i5)') norb                                  !number of density matrices on this file
  READ(dmunit, '(i5)') d_dm                                  !dimension of the density matrix

  IF (iscgto .and. d_dm .ne. basis%ncgto) THEN 
     STOP "Dimensions of density matrices on input and size of the used basis set are not compatible. Program terminated."
  END IF

  IF (.not. iscgto .and. d_dm .ne. basis%nsabf) THEN 
     STOP "Dimensions of density matrices on input and size of the used basis set are not compatible. Program terminated."
  END IF

  WRITE (*,'(5X,a,i5)')'Number of density matrices on the file: ',norb

  IF (req_dm < 0) THEN
     req_dm=norb
  END IF

  IF (req_dm > norb) THEN
     WRITE(*,'(5X,a)') 'Required number of density matrices higher than available! &
                       &Reseting to the number of available density matrices.'
     req_dm=norb
  END IF

  WRITE (*,'(5X,a,i5)')'Required number of density matrices: ',req_dm

  IF (ilist) THEN
     WRITE (*,'(5X,a,100i5)') "Requested sequence numbers of density matrices: ",(dm_list(i),i=1,req_dm)
     DO i=1,size(dm_list)
        IF (dm_list(i) > norb) then
           STOP "A requested sequence number in 'dm_list' higher than available number of density matrices. Program terminated."
        end if
     END DO
  END IF

  ALLOCATE(orbital(1:req_dm))                            !allocate space for the density matrices
  DO i=1,req_dm
     ALLOCATE(orbital(i)%dm(1:d_dm,1:d_dm))
  END DO

  DO i=1,req_dm
     IF (ilist) CALL GET_ON_DM(dmunit,dm_list(i)) !if we are reading selected density matrices only

     READ(dmunit,'(a)') title_dm                         !title of the current density matrix
     orbital(i)%tot_ch=0.0_wp
     DO m=1,d_dm
        DO n=1,m
           READ(dmunit,*) orbital(i)%dm(m,n)             !lower triangle of the symmetric density matrix
           IF (iscgto) THEN              !calculate overlap corresponding to the basis set being used
              tmp=OVERLAP_CGTO(m,n)
           ELSE
              tmp=OVERLAP_SABF(m,n)
           END IF
           orbital(i)%tot_ch=orbital(i)%tot_ch+orbital(i)%dm(m,n)*tmp
        END DO
     END DO
  IF (i.eq.1) WRITE(*,'(/,5X,"Total charge in the required orbitals/density matrices:")')
  WRITE(*,'(5X,i5,"|",a,"|",f20.15)') i,trim(title_dm),orbital(i)%tot_ch
  END DO
  CLOSE(dmunit)
  WRITE (*,'(5X,a)')'Density matrices read successfully.'

  END SUBROUTINE READ_DM

  SUBROUTINE GET_ON_DM(IUNIT,no)
  !this routine gets on the position in the file 'IUNIT' of the density matrix with sequence number 'no'.
  IMPLICIT NONE
  !INPUT
  INTEGER :: IUNIT  !unit number of a file containing the density matrices
  INTEGER :: no     !sequence number of a density matrix we'll want to read in later
  !auxiliary:
  INTEGER :: d_dm, i, j, dm_l

  REWIND(IUNIT)

  READ(IUNIT,*) !type of the basis set corresponding to the saved density matrix elements 
  READ(IUNIT,*) !number of density matrices on this file
  READ(IUNIT, '(i5)') d_dm  !dimension of the density matrix

  dm_l=d_dm*(d_dm+1)/2      !lenght of the record for one density matrix

  DO i=1,no-1      !skip data for no-1 orbitals
     READ(IUNIT,*) !title of the current density matrix
     DO j=1,dm_l
        READ(IUNIT,*)
     END DO
  END DO
  
  END SUBROUTINE GET_ON_DM  

END PROGRAM RADDEN
