! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.

!> \brief   Utility module
!> \authors A Al-Refaie
!> \date    2017
!>
!> \note 16/01/2019 - Jakub Benda: Unifom coding style and expanded documentation.
!>
module Utility_module

    use precisn, only: longint, wp
    use mpi_gbl, only: mpi_mod_wtime

    implicit none

    public string_hash, get_real_time, get_cpu_time, compute_total_triangular, triangular_index_to_ij
    public compute_total_box, box_index_to_ij

contains

    !> \brief   Calculate triangular area
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    !> Calculate n*(n+1)/2.
    !>
    integer(longint) function compute_total_triangular (n)
        integer, intent(in) :: n

        compute_total_triangular = n * (n + 1_longint) / 2

    end function compute_total_triangular


    !> \brief   Calculate rectangular product
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    !> Calculate width*height.
    !>
    integer function compute_total_box (width, height)
        integer, intent(in) :: width, height

        compute_total_box = width * height

    end function compute_total_box


    !> \brief   Extract indices from rectangular multi-index
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    subroutine box_index_to_ij (idx, height, i, j)
        integer, intent(in)  :: idx, height
        integer, intent(out) :: i, j

        i = mod(idx - 1, height) + 1
        j = (idx - 1) / height + 1

    end subroutine box_index_to_ij


    !> \brief   Extract indices from triangular multi-index
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    subroutine triangular_index_to_ij (idx_f, N, row, column)
        integer, intent(in)  :: idx_f, N
        integer, intent(out) :: row, column
        integer              :: idx, ii, K, jj

        idx = idx_f - 1
        ii = N * (N + 1) / 2 - 1 - idx
        K = int((sqrt(8.0_wp*real(ii,wp) + 1.0_wp) - 1.0_wp) / 2.0_wp, longint)
        jj = ii - K * (K + 1) / 2

        row = N - 1 - K
        column = N - 1 - jj

    end subroutine triangular_index_to_ij


    !> \brief   Calculate a string hash
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    !> This hash assumes at least 29-bit integers. It is supposedly
    !> documented in Aho, Sethi, and Ullman, pp. 434-438
    !>
    integer function string_hash (str, table_size) result(h)
        character(len=*), intent(in) :: str
        integer,          intent(in) :: table_size
        integer :: i, chr, g, mask = int(Z"1FFFFFF")

        h = 0
        do i = 1, len_trim(str)
            chr = ichar(str(i:i))
            h   = ishft(h,  4) + chr
            g   = ishft(h,-24)
            h   = iand(ieor(h,g), mask)
        end do
        h = 1 + modulo(h, table_size)

    end function string_hash


    !> \brief   Get current (real) time
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    !> Uses a function from GBTOlib.
    !>
    function get_real_time () result(t)
        real(wp) :: t

        t = mpi_mod_wtime()

    end function get_real_time


    !> \brief   Get current (CPU) time
    !> \authors A Al-Refaie
    !> \date    2017
    !>
    function get_cpu_time () result(t)
        real(wp) :: t

        call cpu_time(t)

    end function get_cpu_time

end module Utility_module
