! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
! This file contains modules storing global data:
!   module consts, for named constants to be used for numbers and
!                  fundamental constants such as pi
!   module params, for named constants giving physically
!                  meaningful information and specifying various maxima -
!                  before dynamic memory allocation!
! Joanne Carr, February 2010
!

module consts
!
! Definition of various constants used throughout the codes
! (hence all entities are public)
! Joanne Carr, Feb 2010
!
  use precisn, only : wp ! for specifying the kind of reals
  implicit none
  public ! as it only contains global data

! Reals

  real(kind=wp), parameter :: XZERO   = 0.0_wp
  real(kind=wp), parameter :: EIGHTH  = 0.125_wp
  real(kind=wp), parameter :: QUART   = 0.25_wp
  real(kind=wp), parameter :: XHALF   = 0.5_wp
  real(kind=wp), parameter :: THQUART = 0.75_wp
  real(kind=wp), parameter :: XONE    = 1.0_wp
  real(kind=wp), parameter :: ONEP5   = 1.5_wp
  real(kind=wp), parameter :: XTWO    = 2.0_wp
  real(kind=wp), parameter :: XTHREE  = 3.0_wp
  real(kind=wp), parameter :: XFOUR   = 4.0_wp
  real(kind=wp), parameter :: XSIX    = 6.0_wp
  real(kind=wp), parameter :: XTEN    = 10.0_wp
  real(kind=wp), parameter :: XTWELVE = 12.0_wp

  real(kind=wp), parameter :: PI      = 3.14159265358979_wp ! defined like this in the code (gaustail.f); change in future???

! Integers (but why bother???)

  integer, parameter :: MINUS7 = -7
  integer, parameter :: MINUS6 = -6
  integer, parameter :: MINUS3 = -3
  integer, parameter :: MINUS2 = -2
  integer, parameter :: MINUS1 = -1
  integer, parameter :: IZERO  = 0
  integer, parameter :: IONE   = 1
  integer, parameter :: ITWO   = 2
  integer, parameter :: ITHREE = 3

! Physics

  real(wp), parameter :: AMU = 1822.832_wp

end module consts

module params
!
! Module storing physically meaningful parameters
! Joanne Carr, February 2010
!
  use precisn, only : wp ! for specifying the kind of reals
  implicit none
  public ! as it only contains global data

! Maximum number of symmetries (irreps) in the molecular point group
  integer, parameter :: MAXSYM=8

! D2h group multiplication table
! Note the change in dimensions from 8,8 to maxsym, maxsym
! to achieve consistency via the global parameters.
  integer, dimension(maxsym,maxsym), parameter :: IPD2H=RESHAPE( (/ &
     &        1,2,3,4,5,6,7,8, &
     &        2,1,4,3,6,5,8,7, &
     &        3,4,1,2,7,8,5,6, &
     &        4,3,2,1,8,7,6,5, &
     &        5,6,7,8,1,2,3,4, &
     &        6,5,8,7,2,1,4,3, &
     &        7,8,5,6,3,4,1,2, &
     &        8,7,6,5,4,3,2,1/) , (/ maxsym, maxsym /) )

  character(len=4), dimension(maxsym), parameter :: OS=(/ &
     & ' AG ', ' B3U', ' B2U', ' B1G', ' B1U', ' B2G', ' B3G', ' AU ' /)

! Buffer size for writing integrals to file
  integer, parameter :: LBUF=600

! Buffer size on transformed integrals file
  integer, parameter :: LRECLX=5000

!     We only permit the contracted basis functions
!     generated by MOLECULE to be real spherical harmonics which are
!     free of contaminants. However MOLECULE writes its pointer
!     information which maps each basis function into the list below
!     and thus we must work with the full list - at all times we have
!     avoided making changes to MOLECULE itself !!!
!
!     The pointer information mentioned above is actually stored in
!     the Mulliken population analysis data generated by MOLECULE.

  CHARACTER(LEN=4), DIMENSION(84,2) :: KKTYP = RESHAPE( &
     &   (/'S000', 'P100', 'P010', 'P001', 'D200', &
     &     'D110', 'D101', 'D020', 'D011', 'D002', 'F300', 'F210', &
     &     'F201', 'F120', 'F111', 'F102', 'F030', 'F021', 'F012', &
     &     'F003', 'G400', 'G310', 'G301', 'G220', 'G211', 'G202', &
     &     'G130', 'G121', 'G112', 'G130', 'G040', 'G031', 'G022', &
     &     'G031', 'G004', 'H500', 'H410', 'H401', 'H320', 'H311', &
     &     'H302', 'H230', 'H221', 'H212', 'H203', 'H140', 'H131', &
     &     'H122', 'H113', 'H104', 'H050', 'H041', 'H032', 'H023', &
     &     'H014', 'H005', 'I600', 'I510', 'I501', 'I420', 'I411', &
     &     'I402', 'I330', 'I321', 'I312', 'I303', 'I240', 'I231', &
     &     'I222', 'I213', 'I204', 'I150', 'I141', 'I132', 'I123', &
     &     'I114', 'I105', 'I060', 'I051', 'I042', 'I033', 'I024', &
     &     'I015', 'I006', &
     &     '1s  ', '2px ', '2py ', '2pz ', '3d0 ', &
     &     '3d2-', '3d1+', '3d2+', '3d1-', '3s  ', '4f1+', '4f1-', &
     &     '4f0 ', '4f3+', '4f2-', '4px ', '4f3-', '4f2+', '4py ', &
     &     '4pz ', '5g0 ', '5g2-', '5g1+', '5g4+', '5g3-', '5g2+', &
     &     '5g4-', '5g3+', '5d2-', '5d1+', '5s  ', '5g1-', '5d2+', &
     &     '5d1-', '5d0 ', '6h1+', '6h1-', '6h2+', '6h3+', '6h4-', &
     &     '6px ', '6h5-', '6h4+', '6h3-', '6h0 ', '6h5+', '6f2-', &
     &     '6f3+', '6h2-', '6f1+', '6py ', '6pz ', '6f3-', '6f2+', &
     &     '6f1-', '6f0 ', '7i6+', '7i2-', '7i5+', '7i4+', '7i5-', &
     &     '7i2+', '7i6-', '7i3+', '7i4-', '7g3+', '7i0 ', '7i3-', &
     &     '7g4+', '7i1-', '7g2+', '7g2-', '7i1+', '7g4-', '7g1+', &
     &     '7d2-', '7d1+', '7d0 ', '7g3-', '7d2+', '7g1-', '7g0 ', &
     &     '7d1-', '7s  '/) , (/ 84, 2 /) )

! Headers for integrals files, following the MOLECULE-SWEDEN conventional form
! '********', '        ', '        ', 'ABCDEFGH'
! where the text string ABCDEFGH is one of the following:

      CHARACTER(LEN=8), PARAMETER :: ccidata  ='CIDATA  ', &
                                     cfillerx ='FILLERXX', &
                                     ckinet   ='KINETINT', &
                                     conelh   ='ONEHAMIL', &
                                     coverlap ='OVERLAP ', &
                                     cpoly    ='POLYAINP', &
                                     cmulpop  ='MULPOPDA', &
                                     csymtr   ='SYMTRANS', &
                                     ctwoel   ='TWOELSUP', &
                                     crawb    ='RAWBAMPS', &
                                     ctrueb   ='TRUEAMPS', &
                                     cregint  ='REGINTGL', &
                                     cnucr    ='NREPULSN', &
                                     crmatrix ='R-MATRIX', &
                                     ctails   ='TAILS   ', &
                                     ctrans1  ='TRANS1EL', &
                                     ctrans2  ='TRANS2EL', &
                                     cord1    ='ORDERED1', &
                                     cord2    ='ORDERED2', &
                                     csuper1  ='SUPERM1E', &
                                     csuper2  ='SUPERM2E', &
                                     c8stars  ='********', &
                                     cblank   ='        ', &
                                     crawb_pt1='RAW_AMP1', &
                                     crawb_pt2='RAW_AMP2', &
                                     camp_pts ='AMP_PTS ', &
                                     ctruebpt1='TRUEAMP1', &
                                     ctruebpt2='TRUEAMP2'
! Change: Laura Moore, May 2011
!   Additional headers above (crawb_pt1,crawb_pt2,ctruebpt1,ctruebpt2)
!   for using when writing surface amplitudes at additional points
!   inside the R-matrix boundary, and a header (crawb_pts) for using
!   when writing to file the number of amplitudes to be calculated

      CHARACTER(LEN=8), DIMENSION(4), PARAMETER :: NTYPI=(/ &
           ' I I I I', ' I J I J', ' I I J J', ' I J K L' /)

      CHARACTER(LEN=8), DIMENSION(4), PARAMETER :: ctwotails=(/ &
           'TWOEIIII', 'TWOEIJIJ', 'TWOEIIJJ', 'TWOEIJKL' /)

end module params
