! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
        SUBROUTINE DVDSON(OP,N,LIM,DIAG,
     :                   ILOW,IHIGH,ISELEC,NIV,MBLOCK,
     :                   CRITE,CRITC,CRITR,ORTHO,MAXITER,
     :                   WORK,IWRSZ,
     :                   HIEND,NLOOPS,NMV,IERR,array,iarray,ndim)
*=======================================================================
*
*       Author: Andreas Stathopoulos, Charlotte F. Fischer
*
*       Computer Science Department
*       Vanderbilt University
*       Nashville, TN 37212
*       andreas@vuse.vanderbilt.edu
*       cff@vuse.vanderbilt.edu                       DECEMBER 1993
*
*       Copyright (c) by Andreas Stathopoulos and Charlotte F. Fischer
*
*      Ammendments (in lower case) by Jonathan Tennyson April 1996
*      In particular to use NAG whole matrix diagonaliser F02ABF
*      in place of DSPEVX.
*
*       DVDSON is a Fortran77 program that finds a few selected
*       eigenvalues and their eigenvectors at either end of spectrum of
*       a large, symmetric (and usually sparse) matrix, denoted as A.
*       The matrix A is only referenced indirectly through the user
*       supplied routine OP which implements a block matrix-vector
*       operation(see below). Either the range of the eigenvalues wanted
*       or an array of the indices of selected ones can be specified.
*       DVDSON is a front-end routine for setting up arrays, and initial
*       guess (calling SETUP). It also performs detailed error checking.
*       DVDRVR is the driver routine that implements a version of the
*       Davidson algorithm. The characteristics of this version are:
*        o  All arrays used by the program are stored in MEMORY.
*        o  BLOCK method (many vectors may be targeted per iteration.)
*        o  Eigenvectors are targeted in an optimum way without
*           the need to compute all unconverged residuals,
*        o  It REORTHOGONILIZES the basis in case of orthogonality loss.
*        o  Finds HIGHEST eigenpairs by using the negative of the A.
*        o  Finds SELECTED eigenpairs specified by the user.
*        o  It accepts INITIAL eigenvector ESTIMATES or it can
*           CREATE INITIAL ESTIMATES from the diagonal elements.
*        o  It uses a USER SUPPLIED block matrix-vector operation, OP.
*           Depending on the implementation, OP can operate in either
*           memory or on disc, and for either sparse or dense matrix.
*        o  The user can provide STOPPING CRITERIA for eigenvalues,
*           and residuals. The user can also CONTROL reorthogonalization
*            and block size.
*        o  On exit INFORMATION is given about the convergence status
*           of eigenpairs and the number of loops and OP operations.
*
*       The program consists of the following routines:
*       DVDSON, SETUP, DVDRVR, ADDABS, TSTSEL,
*       MULTBC, OVFLOW, NEWVEC, ORTHNRM.
C
*       It also calls some basic BLAS routines:
*       DCOPY, DSCAL, DDOT, DAXPY, IDAMAX, DGEMV, DINIT
C
*       For solving the small eigenproblem, the routine DSPEVX from
*       LAPACK is used. DSPEVX is obtainable from NETLIB, together
*       with a series of subroutines that it calls.
*
*       All the routines have IMPLICIT DOUBLE PRECISION(A-H,O-Z)
*
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION DIAG(N),WORK(IWRSZ),array(ndim),iarray(ndim,2)
        DIMENSION ISELEC(LIM)
        LOGICAL HIEND
        external op
*-----------------------------------------------------------------------
*  (Important to the following is the concept of NUME, the distance of
*   the index of the eigenpair wanted which is farthest from the
*   extremes,i.e.,
*      if  lowest  eigepairs i1<i2<...<ik are wanted, NUME=ik
*      if highest eigenpairs i1<i2<...<ik are wanted, NUME=N-i1+1
*   where i1,...,ik are the indices of the wanted eigenpairs.
*   Obviously, NUME.GE.(No. of EiGenpairs wanted). )
C
*   on entry
*   -------
*   OP          User supplied routine with calling sequence OP(N,M,B,C).
*               B and C are N x M matrices and C stores the result AxB.
*               It should be declared external in the main program.
*   N           Order of the matrix.
*   LIM         The upper limit on the dimension of the expanding basis.
*               NUME.LT.LIM.LE.N must hold. The case LIM=NUME is allowed
*               only for LIM=NUME=N. The choice of LIM depends on the
*               available workspace (see below). If the space is
*               available it is preferable to have a large LIM, but not
*               larger than NUME$+$40.
*   DIAG        Array of size N with the diagonal elements of the
*               matrix A.
*   ILOW        The index of the lowest eigepair to be computed. If
*               (ILOW.LE.0).or.(ILOW.GT.N), the selected eigenpairs
*               to be computed should be contained in array ISELEC.
*               (Modified on exit).
*   IHIGH       The index of the highest eigenpair to be computed.
*               Considered ONLY when ILOW is in the range
*               (0.LT.ILOW.LE.N). (Modified on exit).
*   ISELEC      Array of size LIM holding the user specified indices
*               for the eigenpairs to be computed. Considered only when
*               (ILOW.LE.0).or.(ILOW.GT.N). The indices are read from
*               the first position until a non positive integer is met.
*                  Example: if N=500, ILOW=0, and ISELEC(1)=495,
*                  ISELEC(2)=497, ISELEC(3)=-1, the program will find
*                  2 of the highest eigenpairs, pairs 495 and 497.
*               Any order of indices is acceptable (Modified on exit).
*   NIV         Number of Initial Vector estimates provided by the user.
*               If NIV is in the range:  (NUME).LE.(NIV).LE.(LIM),
*               the first NIV columns of size N of WORK should contain
*               the estimates (see below). In all other cases of NIV,
*               the program generates initial estimates.
*   MBLOCK      Number of vectors to be targeted in each iteration.
*               1.LE.MBLOCK.LE.(No. EiGenpairs wanted) should hold.
*               Large block size reduces the number of iterations
*               (matrix acceses) but increases the matrix-vector
*               multiplies. It should be used when the matrix accese
*               is expensive (disc, recomputed or distributed).
*   CRITE       Convergence threshold for eigenvalues.
*               If ABS(EIGVAL-VALOLD) is less than CRITE for all wanted
*               eigenvalues, convergence is signaled.
*   CRITC       Convergence threshold for the coefficients of the last
*               added basis vector(s). If all of those corresponding to
*               unconverged eigenpairs are less than CRITC convergence
*               is signaled.
*   CRITR       Convergence threshold for residual vector norms. If
*               all the residual norms ||Ax_i-l_ix_i|| of the targeted
*               x_i are less than CRITR convergence is signaled.
*               If ANY of the criteria are satisfied the algorithm stops
*   ORTHO       The threshold over which loss of orthogonality is
*               assumed. Usually ORTHO.LE.CRITR*10 but the process can
*               be skipped by setting ORTHO to a large number(eg,1.D+3).
*   MAXITER     Upper bound on the number of iterations of the
*               algorithm. When MAXITER is exceeded the algorithm stops.
*               A typical MAXITER can be MAX(200,NUME*40), but it can
*               be increased as needed.
*   WORK        Real array of size IWRSZ. Used for both input and output
*               If NIV is in ((NUME).LE.(NIV).LE.(LIM)), on input, WORK
*               must have the NIV initial estimates. These NIV N-element
*               vectors start from WORK(1) and continue one after the
*               other. They must form an orthonormal basis.
*   IWRSZ       The size of the real workspace. It must be at least as
*               large as:
*
*                       2*N*LIM + LIM*LIM + (NUME+10)*LIM + NUME
*
*   array       work space to be passed to OP
*   ndim        dimension information to be passed to OP
*
*   on exit
*   -------
*   WORK(1)     The first NUME*N locations contain the approximations to
*               the NUME extreme eigenvectors. If the lowest eigenpairs
*               are required, (HIEND=false), eigenvectors appear in
*               ascending order, otherwise (HIEND=false), they appear in
*               descending order. If only some are requested, the order
*               is the above one for all the NUME extreme eigenvectors,
*               but convergence has been reached only for the selected
*               ones. The rest are the current approximations to the
*               non-selected eigenvectors.
*   WORK(NUME*N+1)
*               The next NUME locations contain the approximations to
*               the NUME extreme eigenvalues, corresponding to the above
*               NUME eigenvectors. The same ordering and convergence
*               status applies here as well.
*   WORK(NUME*N+NUME+1)
*               The next NUME locations contain the corresponding values
*               of ABS(EIGVAL-VALOLD) of the NUME above eigenvalues, of
*               the last step of the algorithm.
*   WORK(NUME*N+NUME+NUME+1)
*               The next NUME locations contain the corresponding
*               residual norms of the NUME above eigenvectors, of the
*               last step.
*   HIEND       Logical. If .true. on exit the highest eigenpairs are
*               found in descending order. Otherwise, the lowest
*               eigenpairs are arranged in ascending order.
*   NLOOPS      The number of iterations it took to reach convergence.
*               This is also the number of matrix references.
*   NMV         The number of Matrix-vector(M-V) multiplies. Each matrix
*               reference can have up to size(block) M-V multiplies.
*   IERR        An integer denoting the completions status:
*               IERR = 0        denotes normal completion.
*               IERR = -k       denotes error in DSPEVX (k eigenpairs
*                               not converged)
*               0<IERR<=2048    denotes some inconsistency as follows:
*        If (INT( MOD(IERR,  2)/1  ) N < LIM
*        If (INT( MOD(IERR,  4)/2  ) LIM < 1
*        If (INT( MOD(IERR,  8)/4  ) ISELEC(1)<1, and no range specified
*        If (INT( MOD(IERR, 16)/8  ) IHIGH > N (in range or ISELEC)
*        If (INT( MOD(IERR, 32)/16 ) IHIGH < ILOW (Invalid range)
*        If (INT( MOD(IERR, 64)/32 ) NEIG >= LIM (Too many wanted)
*        If (INT( MOD(IERR,128)/64 ) Probable duplication in ISELEC
*        If (INT( MOD(IERR,256)/128) NUME >= LIM (max eigen very far)
*        If (INT( MOD(IERR,512)/256) MBLOCK is out of bounds
*        If (INT( MOD(IERR,2048)/1024) Orthogonalization Failed
*        If (INT( MOD(IERR,4096)/2048) NLOOPS > MAXITER
*
*               The program will also print an informative message to
*               the standard output when NIV is not proper but it will
*               continue by picking initial estimates internally.
*-----------------------------------------------------------------------
*
* Checking user input errors, and setting up the problem to solve.
*
        IERR=0
        IF (LIM.GT.N) IERR=IERR+1
        IF (LIM.LE.0) IERR=IERR+2
C
        HIEND=.false.
C
        IF ((ILOW.LE.0).OR.(ILOW.GT.N)) THEN
*          ..Look for user choice of eigenpairs in ISELEC
           IF (ISELEC(1).LE.0) THEN
*             ..Nothing is given in ISELEC
              IERR=IERR+4
           ELSE
*             ..Find number of eigenpairs wanted, and their
*             ..min/max indices
              NEIG=1
              ILOW=ISELEC(1)
              IHIGH=ISELEC(1)
              DO I=2,LIM
                 IF (ISELEC(I).LE.0) GOTO 20
                 ILOW=MIN(ILOW,ISELEC(I))
                 IHIGH=MAX(IHIGH,ISELEC(I))
                 NEIG=NEIG+1
              END DO
*             ..Check if a very large index is asked for
 20           IF (IHIGH.GT.N) IERR=IERR+8
           ENDIF
        ELSE
*          ..Look for a range between ILOW and IHIGH
*          ..Invalid range. IHIGH>N
           IF (IHIGH.GT.N) IERR=IERR+8
           NEIG=IHIGH-ILOW+1
*          ..Invalid range. IHIGH<ILOW
           IF (NEIG.LE.0) IERR=IERR+16
           IF (NEIG.GT.LIM) THEN
*             ..Not enough Basis space. Increase LIM or decrease NEIG
              IERR=IERR+32
           ELSE
*             ..Fill in the ISELEC with the required indices
              DO I=1,NEIG
                 ISELEC(I)=ILOW+I-1
              END DO
           ENDIF
        ENDIF
C
        IF (IERR.NE.0) RETURN
C
        NUME=IHIGH
*       ..Identify if few of the highest eigenpairs are wanted.
        IF ((ILOW+IHIGH-1).GT.N) THEN
           HIEND=.true.
           NUME=N-ILOW+1
*          ..Change the problem to a minimum eipenpairs one
*          ..by picking the corresponding eigenpairs on the
*          ..opposite side of the spectrum.
           DO I=1,NEIG
              ISELEC(I)=N-ISELEC(I)+1
           END DO
        ENDIF
*       ..duplications in ISELEC
        IF (NEIG.GT.NUME) IERR=IERR+64
*       ..Not enough Basis space. Increase LIM or decrease NUME
        IF ((NUME.GT.LIM).OR.((NUME.EQ.LIM).AND.(NUME.NE.N)))
     :     IERR=IERR+128
*       ..Size of Block out of bounds
        IF ( (MBLOCK.LT.1).OR.(MBLOCK.GT.NEIG) ) IERR=IERR+256
C
*       ..Check for enough workspace for Dvdson
        IF ((IWRSZ.LT.(LIM*(2*N+LIM+9)+LIM*(LIM+1)/2)+nume))
     *      IERR=IERR+512
C
        IF (IERR.NE.0) RETURN
C
        IF (NIV.GT.LIM) THEN
*          ..Check number of initial estimates NIV is lower than LIM.
           PRINT*,'WARNING: Too many initial estimates.?'
           PRINT*,'The routine will pick the appropriate number'
        ELSEIF ((NIV.LT.NUME).AND.(NIV.GT.0)) THEN
*          ..check if enough initial estimates.
*          ..(NIV<1 => program chooses)
           PRINT*,'WARNING: Not enough initial estimates'
           PRINT*,'The routine will pick the appropriate number'
        ENDIF
*
* Assigning space for the real work arrays
*
        iBasis    =1
        ieigval   =iBasis  +N*LIM
        iAB       =ieigval +LIM
        iS        =iAB     +N*LIM
        iSvec     =iS      +LIM*(LIM+1)/2
        iscra1    =iSvec   +LIM*LIM
        ioldval   =iscra1  +8*LIM
C
        IF (HIEND) CALL DSCAL(N,-1.D0,DIAG,1)
C
        iSTART=NIV
        CALL SETUP(OP,N,LIM,NUME,HIEND,DIAG,
     :             WORK(iBasis),WORK(iAB),WORK(iS),iSTART,array,
     *             iarray,ndim)
        NLOOPS=1
        NMV=ISTART
C
        CALL DVDRVR(OP,N,HIEND,LIM,MBLOCK,DIAG,
     :             NUME,iSTART,NEIG,ISELEC,
     :             CRITE,CRITC,CRITR,ORTHO,MAXITER,
     :             WORK(ieigval),WORK(iBasis),WORK(iAB),
     :             WORK(iS),WORK(iSvec),work(iscra1),
     :             WORK(ioldval),
     :             NLOOPS,NMV,IERR,array,iarray,ndim)
        IF (HIEND) THEN
           CALL DSCAL(N,-1.D0,DIAG,1)
           CALL DSCAL(NUME,-1.D0,WORK(ieigval:),1)
        endif
*
* -Copy the eigenvalues after the eigenvectors
* -Next, copy the difference of eigenvalues between the last two steps
* -Next, copy the residuals for the first NUME estimates
*
        CALL DCOPY(NUME,WORK(ieigval),1,WORK(iBasis+N*NUME),1)
        CALL DCOPY(NUME,WORK(ioldval),1,WORK(iBasis+(N+1)*NUME),1)
        CALL DCOPY(NUME,WORK(iscra1),1,WORK(iBasis+(N+2)*NUME),1)
C
 100    RETURN
        END
*=======================================================================
        SUBROUTINE SETUP(OP,N,LIM,NUME,HIEND,DIAG,
     :                   BASIS,AB,S,NIV,array,iarray,ndim)
*=======================================================================
*       Subroutine for setting up (i) the initial BASIS if not provided,
*       (ii) the product of the matrix A with the Basis into matrix AB,
*       and (iii) the small matrix S=B^TAB. If no initial estimates are
*       available, the BASIS =(e_i1,e_i2,...,e_iNUME), where i1,i2,...,
*       iNUME are the indices of the NUME lowest diagonal elements, and
*       e_i the i-th unit vector. (ii) and (iii) are handled by ADDABS.
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION DIAG(N),BASIS(N*LIM),AB(N*LIM),array(ndim),
     *  iarray(2*ndim)
        DIMENSION S(LIM*(LIM+1)/2),MINELEM(LIM)
        logical hiend
        external op
*-----------------------------------------------------------------------
*   on entry
*   --------
*   OP          The block matrix-vector operation, passed to ADDABS
*   N           the order of the matrix A
*   LIM         The limit on the size of the expanding Basis
*   NUME        Largest index of the wanted eigenvalues.
*   HIEND       Logical. True only if the highest eigenpairs are needed.
*   DIAG        Array of size N with the diagonal elements of A
*   MINELEM     Array keeping the indices of the NUME lowest diagonals.
*
*   on exit
*   -------
*   BASIS       The starting basis.
*   AB, S       The starting D=AB, and small matrix S=B^TAB
*   NIV         The starting dimension of BASIS.
*-----------------------------------------------------------------------
C
        IF ((NIV.GT.LIM).OR.(NIV.LT.NUME)) THEN
*
*          ..Initial estimates are not available. Give as estimates unit
*          ..vectors corresponding to the NUME minimum diagonal elements
*          ..First find the indices of these NUME elements (in MINELEM).
*          ..Array AB is used temporarily as a scratch array.
*
           CALL DINIT(N,-1.D0,AB,1)
           DO I=1,NUME
*             ..imin= the first not gotten elem( NUME<=N )
              DO J=1,N
                 IF (AB(J).LT.0) GOTO 30
              END DO
 30           IMIN=J
              DO J=IMIN+1,N
                 IF ((AB(J).LT.0).AND.
     :              (DIAG(J).LT.DIAG(IMIN))) IMIN=J
              END DO
              MINELEM(I)=IMIN
              AB(IMIN)=1.D0
           END DO
*
*          ..Build the Basis. B_i=e_(MINELEM(i))
*
           CALL DINIT(N*LIM,0.D0,BASIS,1)
           DO J=1,NUME
              I=(J-1)*N+MINELEM(J)
              BASIS(I)=1
           END DO
C
           NIV=NUME
        ENDIF
*
* Find the matrix AB by matrix-vector multiplies, as well as the
* small matrix S = B^TAB.
*
        KPASS=0
        CALL ADDABS(OP,N,LIM,HIEND,KPASS,NIV,BASIS,AB,S,array,iarray,
     *  ndim)
C
        RETURN
        END
*=======================================================================
        SUBROUTINE DVDRVR(OP,N,HIEND,LIM,MBLOCK,DIAG,
     :                    NUME,NIV,NEIG,ISELEC,
     :                    CRITE,CRITC,CRITR,ORTHO,MAXITER,
     :                    EIGVAL,BASIS,AB,S,SVEC,scra1,
     :                    OLDVAL,
     :                    NLOOPS,NMV,IERR,array,iarray,ndim)
*=======================================================================
*       called by DVDSON
*
*       Driver routine implementing Davidson's main loop. On entry it
*       is given the Basis, the work matrix D=AB and the small symmetric
*       matrix to be solved, S=B^TAB (as found by SETUP). In each step
*       the small problem is solved by calling DSPEVX.
*       TSTSEL tests for eigenvalue convergence and selects the next
*       pairs to be considered for targeting (as a block).
*       NEWVEC computes the new vectors (block) to be added in the
*       expanding basis, and tests for residual convergence.
*       ADDABS is the critical step of matrix multiplication. The new
*       vectors of D are found Dnew=ABnew, and the new small problem S,
*       is calculated. The algorithm is repeated.
*       In case of a large expanding basis (KPASS=LIM) the Basis, AB,
*       SVEC and S are collapsed.
*       At the end the current eigenvector estimates are computed as
*       well as the residuals and eigenvalue differences.
*
*       Subroutines called:
*       DSPEVX, MULTBC, TSTSEL, OVFLOW, NEWVEC, ADDABS,
*       DCOPY, DDOT, DAXPY
*-----------------------------------------------------------------------
C
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION DIAG(N)
        DIMENSION S(LIM*(LIM+1)/2)
        DIMENSION SVEC(LIM*LIM),EIGVAL(LIM)
        DIMENSION ISELEC(NEIG)
        DIMENSION BASIS(N*LIM),AB(N*LIM)
        DIMENSION SCRA1(8*LIM),ISCRA2(LIM),INCV(LIM)
        DIMENSION ICV(NUME),OLDVAL(NUME),array(ndim),iarray(2*ndim)
        LOGICAL RESTART,FIRST,DONE,HIEND,TSTSEL
        external op
C
*-----------------------------------------------------------------------
*
*   on entry
*   -------
*
*   OP          The user specified block-matrix-vector routine
*   N           The order of the matrix A
*   HIEND       Logical. True only if the highest eigenpairs are needed.
*   LIM         The limit on the size of the expanding Basis
*   MBLOCK      Number of vectors to be targeted in each iteration.
*   DIAG        Array of size N with the diagonal elements of A
*   NUME        The largest index of the eigenvalues wanted.
*   NIV         Starting dimension of expanding basis.
*   NEIG        Number of eigenvalues wanted.
*   ISELEC      Array containg the indices of those NEIG eigenpairs.
*   CRITE       Convergence thresholds for eigenvalues, coefficients
*   CRITC,CRITR and residuals.
*   BASIS       Array with the basis vectors.
*   AB          Array with the vectors D=AB
*   S           Array keeping the symmetric matrix of the small problem.
*   SVEC        Array for holding the eigenvectors of S
*   SCRA1       Srcatch array used by DSPEVX.
*   ISCRA2      Integer Srcatch array used by TSTSEL.
*   INCV        Srcatch array used in DSPEVX. Also used in TSTSEL and
*               NEWVEC where it holds the Indices of uNConVerged pairs
*   ICV         It contains "1" to the locations of ConVerged eigenpairs
*   OLDVAL      Array keeping the previous' step eigenvalue estimates.
*
*   on exit
*   -------
*
*   EIGVAL      Array containing the NUME lowest eigenvalues of the
*               the matrix A (or -A if the highest are sought).
*   Basis       On exit Basis stores the NUME corresponding eigenvectors
*   OLDVAL      On exit it stores the final differences of eigenvalues.
*   SCRA1       On exit it stores the NUME corresponding residuals.
*   NLOOPS      Number of loops taken by the algorithm
*   NMV         Number of matrix-vector products performed.
*
*-----------------------------------------------------------------------
        DO I=1,NUME
           EIGVAL(I)=1.D30
           ICV(I)=0
        END DO
        FIRST =.true.
        KPASS =NIV
        NNCV  =KPASS
C
10      CONTINUE
*       (iterations for kpass=NUME,LIM)
*
* Diagonalize the matrix S. Find only the NUME smallest eigenpairs
*
           CALL DCOPY(NUME,EIGVAL,1,OLDVAL,1)
           CALL MKMAT(KPASS,S,SVEC)
           CALL QLDIAG(KPASS,SVEC,EIGVAL)
           IERR=0
*          IF (IERR.NE.0) GOTO 60
*
* TeST for convergence on the absolute difference of eigenvalues between
* successive steps. Also SELect the unconverged eigenpairs and sort them
* by the largest magnitude in the last added NNCV rows of Svec.
*
           DONE=TSTSEL(KPASS,NUME,NEIG,ISELEC,SVEC,EIGVAL,ICV,
     :            CRITE,CRITC,SCRA1,ISCRA2,OLDVAL,NNCV,INCV)
           IF ((DONE).OR.(KPASS.GE.N)) GOTO 30
C
           IF (KPASS.EQ.LIM) THEN
* Maximum size for expanding basis. Collapse basis, D, and S, Svec
* Consider the basis vectors found in TSTSEL for the newvec.
*
              CALL MULTBC(N,LIM,NUME,SVEC,SCRA1,BASIS)
              CALL MULTBC(N,LIM,NUME,SVEC,SCRA1,AB)
              CALL OVFLOW(NUME,LIM,S,SVEC,EIGVAL)
              KPASS=NUME
           ENDIF
*
* Compute and add the new vectors. NNCV is set to the number of new
* vectors that have not converged. If none, DONE=true, exit.
*
           CALL NEWVEC(N,NUME,LIM,MBLOCK,KPASS,CRITR,ORTHO,NNCV,INCV,
     :                 DIAG,SVEC,EIGVAL,AB,BASIS,ICV,RESTART,DONE)
C
*          ..An infinite loop is avoided since after a collapsing Svec=I
*          ..=> Res=Di-lBi which is just computed and it is orthogonal.
*          ..The following is to prevent an improbable infinite loop.
           IF (.NOT.RESTART) THEN
              FIRST=.true.
           ELSEIF (FIRST) THEN
              FIRST=.false.
              CALL MULTBC(N,KPASS+NNCV,NUME,SVEC,SCRA1,BASIS)
              CALL MULTBC(N,KPASS+NNCV,NUME,SVEC,SCRA1,AB)
              CALL OVFLOW(NUME,KPASS+NNCV,S,SVEC,EIGVAL)
              KPASS=NUME
              GOTO 10
           ELSE
              IERR=IERR+1024
              GOTO 30
           ENDIF
C
           IF (DONE) GOTO 30
*
* Add new columns in D and S, from the NNCV new vectors.
*
           CALL ADDABS(OP,N,LIM,HIEND,KPASS,NNCV,BASIS,AB,S,array,
     *                 iarray,ndim)
C
           NMV=NMV+NNCV
           KPASS=KPASS+NNCV
           NLOOPS=NLOOPS+1
C
        IF (NLOOPS.LE.MAXITER) GOTO 10
        IERR=IERR+2048
        NLOOPS=NLOOPS-1
        KPASS=KPASS-NNCV
 30     CONTINUE
*
* Calculate final results. EIGVAL contains the eigenvalues, BASIS the
* eigenvectors, OLDVAL the eigenvalue differences, and SCRA1 residuals.
*
        DO I=1,NUME
           OLDVAL(I)=ABS(OLDVAL(I)-EIGVAL(I))
        END DO
C
        CALL MULTBC(N,KPASS,NUME,SVEC,SCRA1,BASIS)
        CALL MULTBC(N,KPASS,NUME,SVEC,SCRA1,AB)
*
* i=1,NUME residual(i)= DCi-liBCi= newDi-linewBi
* temporarily stored in AB(NUME*N+1)
*
        DO I=1,NUME
           CALL DCOPY(N,AB((I-1)*N+1),1,AB(NUME*N+1),1)
           CALL DAXPY(N,-EIGVAL(I),BASIS((I-1)*N+1),1,AB(NUME*N+1),1)
           SCRA1(I)=DDOT(N,AB(NUME*N+1),1,AB(NUME*N+1),1)
           SCRA1(I)=SQRT(SCRA1(I))
        END DO
C
 60     RETURN
        END
*=======================================================================
       SUBROUTINE ADDABS(OP,N,LIM,HIEND,KPASS,NNCV,BASIS,AB,S,array,
     * iarray,ndim)
*=======================================================================
*       Called by: DVDRVR, SETUP
*
*       Calculates the new column in the D matrix and the new column
*       in the S matrix. The new D column is D(new)=AB(new). S has a
*       new row and column, but being symmetric only the new column is
*       stored. S(i,kpass+1)=B(i)^T D(kpass+1) for all i.
*
*       subroutines called:
*       OP, DDOT, DSCAL
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION BASIS(N*LIM),AB(N*LIM),array(ndim),iarray(2*ndim)
        DIMENSION S(LIM*(LIM+1)/2)
        LOGICAL HIEND
        EXTERNAL OP
*-----------------------------------------------------------------------
*   on entry
*   -------
*   N           The order of the matrix A
*   kpass       The current dimension of the expanding sub-basis
*   NNCV        Number of new basis vectors.
*   Basis       the basis vectors, including the new NNCV ones.
*   on exit
*   -------
*   AB          The new matrix D=AB. (with new NNCV columns)
*   S           The small matrix with NNCV new columns at the last part
*-----------------------------------------------------------------------
*
* The user specified matrix-vector routine is called with the new
* basis vector B(*,kpass+1) and the result is assigned to AB(idstart)
C
        IDSTART=KPASS*N+1
        CALL OP(N,NNCV,BASIS(IDSTART),AB(IDSTART),
     1          array,iarray,iarray(ndim+1),ndim)
*
* If highest pairs are sought, use the negative of the matrix
*
        IF (HIEND) CALL DSCAL(N*NNCV,-1.D0,AB(IDSTART:),1)
*
* The new S is calculated by adding the new last columns
* S(new)=B^T D(new).
*
        ISSTART=KPASS*(KPASS+1)/2
        DO IV=1,NNCV
           IBSTART=1
           DO IBV=1,KPASS+IV
               SS=DDOT(N,BASIS(IBSTART),1,AB(IDSTART),1)
               S(ISSTART + IBV)=SS
               IBSTART=IBSTART+N
           END DO
           ISSTART=ISSTART+KPASS+IV
           IDSTART=IDSTART+N
        END DO
C
        RETURN
        END
*=======================================================================
        LOGICAL FUNCTION TSTSEL(KPASS,NUME,NEIG,ISELEC,SVEC,EIGVAL,ICV,
     :                       CRITE,CRITC,ROWLAST,IND,OLDVAL,NNCV,INCV)
*=======================================================================
*
*       Called by: DVDRVR
C
*       It first checks if the wanted eigenvalues have reached
*       convergence and updates OLDVAL. Second, for each wanted and non
*       converged eigenvector, it finds the largest absolute coefficient
*       of the NNCV last added vectors (from SVEC) and if not coverged,
*       places it in ROWLAST. IND has the corresponding indices.
*       Third, it sorts ROWLAST in decreasing order and places the
*       corresponding indices in the array INCV. The index array INCV
*       and the number of unconverged pairs NNCV, are passed to DVDRVR.
*       Later in NEWVEC only the first MBLOCK of NNCV pairs will be
*       targeted, since if ROWLAST(i) > ROWLAST(j)
*       then approximately RESIDUAL(i) > RESIDUAL(j)
*
*       Subroutines called
*       IDAMAX
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        LOGICAL DONE
        DIMENSION SVEC(KPASS*NUME),EIGVAL(NUME)
        DIMENSION ICV(NUME)
        DIMENSION ROWLAST(NEIG),IND(NEIG),OLDVAL(NUME)
        DIMENSION INCV(NEIG),ISELEC(NEIG)
*-----------------------------------------------------------------------
*
*   on entry
*   -------
*   KPASS       current dimension of the expanding Basis
*   NUME        Largest index of the wanted eigenvalues.
*   NEIG        number of wanted eigenvalues of original matrix
*   ISELEC      index array of the wanted eigenvalues.
*   SVEC        the eigenvectors of the small system
*   EIGVAL      The NUME lowest eigenvalues of the small problem
*   ICV         Index of converged eigenpairs.ICV(i)=1 iff eigenpair i
*               has converged, and ICV(i)=0 if eigenpair i has not.
*   CRITE,CRITC Convergence thresholds for eigenvalues and coefficients
*   ROWLAST     scratch array, keeping the largest absolute coefficient
*               of the NNCV last rows of Svec.
*   IND         scratch array, temporary keeping the indices of Rowlast
*   OLDVAL      The previous iteration's eigenvalues.
*
*   on exit
*   -------
*   NNCV         Number of non converged eigenvectors (to be targeted)
*   INCV         Index to these columns in decreasing order of magnitude
*   TSTSEL       true if convergence has been reached
*
*-----------------------------------------------------------------------
C
        DONE=.False.
*
* Test all wanted eigenvalues for convergence under CRITE
*
        NNCE=0
        DO I=1,NEIG
           IVAL=ISELEC(I)
           IF (ABS(OLDVAL(IVAL)-EIGVAL(IVAL)).GE.CRITE) NNCE=NNCE+1
        END DO
        IF (NNCE.EQ.0) THEN
           TSTSEL=.TRUE.
           RETURN
        ENDIF
*
* Find the maximum element of the last NNCV coefficients of unconverged
* eigenvectors. For those unconverged coefficients, put their indices
* to IND and find their number NNCV
*
        ICNT=0
        DO I=1,NEIG
           IF (ICV(ISELEC(I)).EQ.0) THEN
*             ..Find coefficient and test for convergence
              ICUR=KPASS*ISELEC(I)
              TMAX=ABS( SVEC(ICUR) )
              DO L=1,NNCV-1
                 TMAX=MAX( TMAX, ABS(SVEC(ICUR-L)) )
              END DO
              IF (TMAX.LT.CRITC) THEN
*                ..this  coefficient converged
                 ICV(ISELEC(I))=1
              ELSE
*                ..Not converged. Add it to the list.
                 ICNT=ICNT+1
                 IND(ICNT)=ISELEC(I)
                 ROWLAST(ICNT)=TMAX
              ENDIF
           ENDIF
        END DO
C
        NNCV=ICNT
        IF (NNCV.EQ.0) DONE=.TRUE.
*
* Sort the ROWLAST elements interchanging their indices as well
*
        DO I=1,NNCV
           INDX=IDAMAX(NNCV-I+1,ROWLAST(I),1)
           INCV(I)=IND(INDX+I-1)
C
           TEMP=ROWLAST(INDX+I-1)
           ROWLAST(INDX+I-1)=ROWLAST(I)
           ROWLAST(I)=TEMP
           ITEMP=IND(INDX+I-1)
           IND(INDX+I-1)=IND(I)
           IND(I)=ITEMP
        END DO
C
        TSTSEL=DONE
        RETURN
        END
*=======================================================================
        SUBROUTINE MULTBC(N,K,M,C,TEMP,B)
*=======================================================================
*       called by: DVDRVR
*
*       Multiplies B(N,K)*C(K,M) and stores it in B(N,M)
*       Used for collapsing the expanding basis to current estimates,
*       when basis becomes too large, or for returning the results back
C
*       Subroutines called
*       DINIT, DGEMV, DCOPY
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION B(N*K),C(K*M),TEMP(M)
*-----------------------------------------------------------------------
        DO IROW=1,N
*              CALL DINIT(M,0.d0,TEMP,1)
           CALL DGEMV('Transp',K,M, 1.D0, C,K,B(IROW),N, 0.D0 ,TEMP,1)
           CALL DCOPY(M,TEMP,1,B(IROW),N)
        END DO
C
        RETURN
        END
C
*=======================================================================
        SUBROUTINE OVFLOW(NUME,LIM,S,SVEC,EIGVAL)
*=======================================================================
*       Called by: DVDRVR
*       Called when the upper limit (LIM) has been reached for the basis
*       expansion. The new S is computed as S'(i,j)=l(i)delta(i,j) where
*       l(i) eigenvalues, and delta of Kronecker, i,j=1,NUME. The new
*       eigenvectors of the small matrix are the unit vectors.
*
*       Subroutines called:
*       DCOPY, DINIT
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION SVEC(LIM*NUME),S((LIM*(LIM+1))/2)
        DIMENSION EIGVAL(LIM)
*-----------------------------------------------------------------------
*   on entry
*   -------
*   NUME        The largest index of eigenvalues wanted.
*   SVEC        the kpass eigenvectors of the smaller system solved
*   EIGVAL      the eigenvalues of this small system
*   on exit
*   -------
*   S           The new small matrix to be solved.
*-----------------------------------------------------------------------
*
* calculation of the new upper S=diag(l1,...,l_NUME) and
* its matrix Svec of eigenvectors (e1,...,e_NUME)
*
        CALL DINIT((NUME*(NUME+1))/2,0.d0,S,1)
        CALL DINIT(NUME*NUME,0.d0,SVEC,1)
        IND=0
        ICUR=0
        DO I=1,NUME
           S(IND+I)=EIGVAL(I)
           SVEC(ICUR+I)=1
           ICUR=ICUR+NUME
           IND=IND+I
        END DO
C
        RETURN
        END
*=======================================================================
        SUBROUTINE NEWVEC(N,NUME,LIM,MBLOCK,KPASS,CRITR,ORTHO,NNCV,INCV,
     :                    DIAG,SVEC,EIGVAL,AB,BASIS,ICV,RESTART,DONE)
*=======================================================================
*
*       Called by: DVDRVR
*
*       It calculates the new expansion vectors of the basis.
*       For each one of the vectors in INCV starting with the largest
*       megnitude one, calculate its residual Ri= DCi-liBCi and check
*       the ||Ri|| for convergence. If it is converged do not add it
*       but look for the immediate larger coefficient and its vector.
*       The above procedure continues until MBLOCK vectors have been
*       added to the basis, or the upper limit has been encountered.
*       Thus only  the required MBLOCK residuals are computed. Then,
*       calculate the first order correction on the added residuals
*       Ri(j) = Ri(j)/(li-Ajj) and orthonormalizes the new vectors
*       to the basis and to themselves.
*
*       Subroutines called:
*       ORTHNRM, DDOT, DGEMV
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION INCV(NUME)
        DIMENSION ICV(NUME)
        DIMENSION DIAG(N)
        DIMENSION BASIS(N*LIM),AB(N*LIM)
        DIMENSION SVEC(LIM*NUME)
        DIMENSION EIGVAL(LIM)
        LOGICAL RESTART,DONE
*-----------------------------------------------------------------------
*   on entry
*   --------
*   N           The order of the matrix A
*   NUME        The largest index of the eigenvalues wanted.
*   LIM         The limit on the size of the expanding Basis
*   MBLOCK      Maximum number of vectora to enter the basis
*   KPASS       the current dimension of the expanding basis
*   CRITR       Convergence threshold for residuals
*   ORTHO       Orthogonality threshold to be passed to ORTHNRM
*   NNCV        Number of Non ConVerged pairs (MBLOCK will be targeted)
*   INCV        Index to the corresponding SVEC columns of these pairs.
*   DIAG        Array of size N with the diagonal elements of A
*   SVEC,EIGVAL Arrays holding the eigenvectors and eigenvalues of S
*   AB          Array with the vectors D=AB
*   BASIS       the expanding basis having kpass vectors
*   ICV         Index of converged eigenpairs (ICV(i)=1 <=>i converged)
C
*   on exit
*   -------
*   NNCV        The number of vectors finally added to the basis.
*   BASIS       The new basis incorporating the new vectors at the end
*   ICV         Index of converged eigenpairs (updated)
*   DONE        logical, if covergance has been reached.
*   RESTART     logical, if because of extreme loss of orthogonality
*               the Basis should be collapsed to current approximations.
*-----------------------------------------------------------------------
        DONE    = .FALSE.
        NEWSTART= KPASS*N+1
        NADDED  = 0
        ICVC    = 0
        LIMADD  = MIN( LIM, MBLOCK+KPASS )
        ICUR    = NEWSTART
*
* Compute RESIDUALS for the MBLOCK of the NNCV not converged vectors.
*
        DO I=1,NNCV
           INDX=INCV(I)
*          ..Compute  Newv=BASIS*Svec_indx , then
*          ..Compute  Newv=AB*Svec_indx - eigval*Newv and then
*          ..compute the norm of the residual of Newv
           CALL DGEMV('N',N,KPASS,1.D0,BASIS,N,SVEC((INDX-1)*KPASS+1),1,
     :                 0.d0,BASIS(ICUR),1)
           CALL DGEMV('N',N,KPASS,1.D0,AB,N,SVEC((INDX-1)*KPASS+1),1,
     :                 -EIGVAL(INDX),BASIS(ICUR),1)
           SS = DNRM2(N,BASIS(ICUR),1)
*
*          ..Check for convergence of this residual
*
           IF (SS.LT.CRITR) THEN
*             ..Converged,do not add. Go for next non converged one
              ICVC=ICVC+1
              ICV( INDX ) = 1
              IF (ICVC.LT.NNCV) CYCLE
*             ..All have converged.
              DONE=.TRUE.
              RETURN
           ELSE
*             ..Not converged. Add it in the basis
              NADDED=NADDED+1
              INCV(NADDED)=INDX
              IF ((NADDED+KPASS).EQ.LIMADD) GOTO 20
*             ..More to be added in the block
              ICUR=ICUR+N
           ENDIF
        END DO
C
 20     NNCV=NADDED
*
* Diagonal preconditioning: newvect(i)=newvect(i)/(l-Aii)
* If (l-Aii) is very small (or zero) divide by 10.D-6
*
        ICUR=NEWSTART-1
        DO I=1,NNCV
           DO IROW=1,N
              DG=EIGVAL(INCV(I))-DIAG(IROW)
              IF (ABS(DG).GT.(1.D-13)) THEN
                  BASIS(ICUR+IROW)=BASIS(ICUR+IROW) / DG
              ELSE
                  BASIS(ICUR+IROW)=BASIS(ICUR+IROW) /1.D-13
              ENDIF
           END DO
           ICUR=ICUR+N
        END DO
*
* ORTHONORMALIZATION
*
        CALL ORTHNRM(N,LIM,ORTHO,KPASS,NNCV,AB(NEWSTART),
     :               BASIS,RESTART)
C
 99     RETURN
        END
*=======================================================================
        SUBROUTINE ORTHNRM(N,LIM,ORTHO,KPASS,NNCV,SCRA1,
     :                     BASIS,RESTART)
*=======================================================================
*
*       It orthogonalizes the new NNCV basis vectors starting from the
*       kpass+1, to the previous vectors of the basis and to themselves.
*       A Gram-Schmidt method is followed after which the residuals
*       should be orthogonal to the BASIS. Because of machine arithmetic
*       errors this orthogonality may be lost, and a reorthogonalization
*       procedure is adopted whenever orthogonality loss is above a
*       ORTHO. If after some reorthogonalizations the procedure does not
*       converge to orthogonality, the basis is collapsed to the
*       current eigenvector approximations.
*
*       Subroutines called:
*       DAXPY, DDOT, DSCAL
*-----------------------------------------------------------------------
        IMPLICIT DOUBLE PRECISION(A-H,O-Z)
        DIMENSION BASIS(N*LIM)
        DIMENSION SCRA1(N)
        LOGICAL RESTART
*-----------------------------------------------------------------------
*   on entry
*   --------
*   N           The order of the matrix A
*   LIM         The limit on the size of the expanding Basis
*   ORTHO       The orthogonality threshold
*   KPASS       The number of basis vectors already in Basis
*   NNCV        The number of new vectors in the basis
*   SCRA1       Scratch vector of size N
*   BASIS       the expanding basis having kpass vectors
*
*   on exit
*   -------
*   BASIS       the new basis orthonormalized
*   RESTART     Logical, if true the algoritm will collapse BASIS.
*-----------------------------------------------------------------------
*
* ORTHOGONALIZATION
*
        RESTART=.false.
        ICUR=KPASS*N+1
*
*       .. do iv=1,nncv
        IV = 1
 30     CONTINUE
C
           DPREV=1.D+7
 5         DCUR=0.D0
           IBSTART=1
           DO I=1,KPASS+IV-1
              SCRA1(I)=DDOT(N,BASIS(IBSTART),1,BASIS(ICUR),1)
              DCUR=MAX(DCUR,ABS(SCRA1(I)))
              IBSTART=IBSTART+N
           END DO
           IBSTART=1
           DO I=1,KPASS+IV-1
              CALL DAXPY(N,-SCRA1(I),BASIS(IBSTART),1,BASIS(ICUR),1)
              IBSTART=IBSTART+N
           END DO
C
           IF (DCUR.GE.ORTHO) THEN
              IF (DCUR.GT.DPREV) THEN
                 RESTART=.true.
*                ..Adjust the number of added vectors.
                 NNCV=IV-1
                 RETURN
              ELSE
                 DPREV=DCUR
                 GOTO 5
              ENDIF
           ENDIF
*
* NORMALIZATION
*
           SCRA1(1)=DDOT(N,BASIS(ICUR),1,BASIS(ICUR),1)
           SCRA1(1)=SQRT(SCRA1(1))
           IF (SCRA1(1).LT.1D-14) THEN
              CALL DCOPY(N,BASIS( N*(NNCV-1)+1),1,BASIS(ICUR),1)
              NNCV=NNCV-1
           ELSE
              CALL DSCAL(N,1/SCRA1(1),BASIS(ICUR:),1)
              ICUR=ICUR+N
              IV = IV +1
           ENDIF
        IF (IV.LE.NNCV) GOTO 30
C
        RETURN
        END
*=======================================================================
        SUBROUTINE DINIT( N, A, X, INCX )
*=======================================================================
*       PURPOSE ... INITIALIZES DOUBLE PRECISION VECTOR TO
*                   A CONSTANT VALUE 'A'
*=======================================================================
        INTEGER N, INCX
        DOUBLE PRECISION A, X (*)
        INTEGER XADDR, I
C
        IF  ( INCX .EQ. 1 )  THEN
            DO I = 1, N
                X(I) = A
            END DO
        ELSE
            XADDR = 1
            IF  ( INCX .LT. 0 )  THEN
                XADDR = (-N+1)*INCX + 1
            ENDIF
            DO I = 1, N
                X(XADDR) = A
                XADDR = XADDR + INCX
            END DO
        ENDIF
C
        RETURN
        END
*=======================================================================
       SUBROUTINE MKMAT(N,A,B)
*=======================================================================
*      copy upper triangular packed array A into lower triangle of B
*=======================================================================
       INTEGER N, I, J, IPT
       DOUBLE PRECISION A(N*(N+1)/2), B(N,N)
C
       IPT=0
       DO I=1,N
          DO J=1,I
             IPT=IPT+1
             B(I,J)=A(IPT)
          END DO
       END DO
C
       RETURN
       END
