! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
      module cdenprop_defs
      use precisn_gbl
      use blas_lapack_gbl, only: blasint
      implicit none
      !*********************************************************!
      !
      !Module containing parameter and derived type definitions
      !
      !*********************************************************!

      integer, parameter :: idp = wp
      integer, parameter :: ir_max=8 !Maximum no of IR per point group
      integer, parameter :: tsym_max=1000 !Maximum number of target symmetries.

      integer, parameter :: maxnuc=20
      integer, parameter :: maxprop=9
      integer, parameter :: maxprop_par=9
!     ---------------------------

!     D2h group multiplication table
!     Note the change in dimensions from 8,8 to maxsym, maxsym
!     to achieve consistency via the global parameters.
      integer, dimension(ir_max,ir_max), parameter :: IPD2H=RESHAPE( (/ &
     &        1,2,3,4,5,6,7,8, &
     &        2,1,4,3,6,5,8,7, &
     &        3,4,1,2,7,8,5,6, &
     &        4,3,2,1,8,7,6,5, &
     &        5,6,7,8,1,2,3,4, &
     &        6,5,8,7,2,1,4,3, &
     &        7,8,5,6,3,4,1,2, &
     &        8,7,6,5,4,3,2,1/) , (/ ir_max, ir_max /) )


      character(len=5), dimension(8), parameter :: CD2H_symb=(/ &
     & ' AG  ', ' B3U ', ' B2U ', ' B1G ', ' B1U ', ' B2G ', ' B3G ', ' AU  ' /)
      character(len=5), dimension(4), parameter :: CC2V_symb=(/ &
     & ' A1  ', ' B1  ', ' B2  ', ' A2  '/)


      character(len=10), dimension(8), parameter ::spin_symb=(/&
     & 'Singlet   ', 'Doublet   ', 'Triplet   ', 'Quadruplet',      &
     & 'Quintuplet', 'Sextuplet ', 'Septuplet ', 'Octuplet  ' /)

!
!     Define the symbols for the elements in the periodic table
!
      character(len=2), dimension(103) :: ASYMB=(/&
     &     'H ', 'He', 'Li', 'Be', 'B ', 'C ', 'N ',                   &
     &     'O ', 'F ', 'Ne', 'Na', 'Mg', 'Al', 'Si', 'P ', 'S ', 'Cl', &
     &     'Ar', 'K ', 'Ca', 'Sc', 'Ti', 'V ', 'Cr', 'Mn', 'Fe', 'Co', &
     &     'Ni', 'Cu', 'Zn', 'Ga', 'Ge', 'As', 'Se', 'Br', 'Kr', 'Rb', &
     &     'Sr', 'Y ', 'Zr', 'Nb', 'Mo', 'Tc', 'Ru', 'Rh', 'Pd', 'Ag', &
     &     'Cd', 'In', 'Sn', 'Sb', 'Te', 'I ', 'Xe', 'Cs', 'Ba', 'La', &
     &     'Ce', 'Pr', 'Nd', 'Pm', 'Sm', 'Eu', 'Gd', 'Tb', 'Dy', 'Ho', &
     &     'Er', 'Tm', 'Yb', 'Lu', 'Hf', 'Ta', 'W ', 'Re', 'Os', 'Ir', &
     &     'Pt', 'Au', 'Hg', 'Tl', 'Pb', 'Bi', 'Po', 'At', 'Rn', 'Fr', &
     &     'Ra', 'Ac', 'Th', 'Pa', 'U ', 'Np', 'Pu', 'Am', 'Cm', 'Bk', &
     &     'Cf', 'Es', 'Fm', 'Md', 'No', 'Lr'/)
!
!     Define the masses of the elements in the periodic table
!
      real(kind=idp), dimension(103) ::AMASS=(/&
     &      1.0078246_idp, 4.002601_idp, 7.01600_idp,                            &
     &      9.01218_idp, 11.009307_idp, 12.000000_idp, 14.0030738_idp,           &
     &     15.9949141_idp, 18.9984022_idp, 19.992441_idp, 22.9898_idp,           &
     &     23.98504_idp, 26.98153_idp, 27.976929_idp, 30.973764_idp,             &
     &     31.9720727_idp, 34.9688531_idp, 39.962386_idp, 38.96371_idp,          &
     &     39.96259_idp, 44.95592_idp, 48._idp, 50.9440_idp, 51.9405_idp,        &
     &     54.9380_idp, 55.9349_idp, 58.9332_idp, 57.9353_idp, 62.9296_idp,      &
     &     63.9291_idp, 68.9257_idp, 73.9219_idp, 74.9216_idp, 79.9165_idp,      &
     &     78.91839_idp, 83.91151_idp, 84.9117_idp, 87.9056_idp, 88.9059_idp,    &
     &     89.9043_idp, 92.9060_idp, 97.9055_idp, 98._idp, 101.9037_idp,         &
     &    102.9048_idp, 107.90389_idp, 106.90509_idp, 113.9036_idp, 114.9041_idp,&
     &    120._idp, 120.9038_idp, 129.9067_idp, 126.90466_idp, 131.90416_idp,    &
     &    132.9051_idp, 137.9050_idp, 138.9061_idp, 139.9053_idp, 140.9074_idp,  &
     &    141.9075_idp, 145._idp, 151.9195_idp, 152.9209_idp, 157.9241_idp,      &
     &    159.9250_idp, 163.9288_idp, 164.9303_idp, 165.9304_idp, 168.9344_idp,  &
     &    173.9390_idp, 174.9409_idp, 179.9468_idp, 180.9480_idp, 183.9510_idp,  &
     &    186.9560_idp, 192._idp, 192.9633_idp, 194.9648_idp, 196.9666_idp,      &
     &    201.970625_idp, 204.9745_idp, 207.9766_idp, 208.9804_idp, 209._idp,    &
     &    210._idp, 222._idp, 223._idp, 226._idp, 227._idp, 232._idp, 231._idp,  &
     &    238._idp,237._idp, 244._idp, 243._idp, 247._idp, 247._idp, 251._idp,   &
     &    252._idp, 257._idp, 258._idp, 259._idp, 260._idp/)

      type CSFheader
         CHARACTER(LEN=120) :: NAME
         integer :: MGVN
         double precision :: S, SZ, R, PIN
         integer :: NORB, NSRB,NOCSF,NELT, lcdof, IDIAG,NSYM, SYMTYP, lndof
         integer, dimension (:) :: npflg(6)
         double precision ::  thres
         integer :: NCTARG,NTGSYM
         integer, allocatable, dimension (:) :: iphz, nctgt, notgt, mcont, gucont, nob, ndtrf, nodo, numtgt
         integer, allocatable, dimension (:) :: itarget_overall_phase, idtarg
         integer, allocatable, dimension (:,:) ::  itarget_symmetry_order
         integer :: iposit
         integer, allocatable, dimension (:) :: nob0, nobl, nob0l,no_l2_virtuals
         integer :: l2nocsf,last_continuum_csf
      contains
         procedure :: dealloc=>dealloc_CSFheader
      end type CSFheader

      type CSFbody
         ! MPI shared memory windows
         integer :: icdo_window = -1
         integer :: indo_window = -1
         integer :: ndo_window  = -1
         integer :: cdo_window  = -1
         ! pointers to MPI shared memory
         integer,   pointer :: icdo(:) => null()
         integer,   pointer :: indo(:) => null()
         integer,   pointer :: ndo(:)  => null()
         real(idp), pointer :: cdo(:)  => null()
      contains
         procedure :: dealloc => dealloc_CSFbody
         final :: finalize_CSFbody
      end type CSFbody

      type CIvect
         integer :: nset, nrec
         CHARACTER(len=120) :: name
         integer :: nnuc,nocsf,nstat,mgvn
         real(kind=idp) :: S, SZ
         integer :: nelt
         real(kind=idp) :: e0
         character(len=8), dimension(:) :: cname(maxnuc)
         real(kind=idp), dimension(:) :: charge(maxnuc),xnuc(maxnuc), ynuc(maxnuc), znuc(maxnuc)
         real(kind=idp), allocatable, dimension(:) ::  ei
         integer , allocatable, dimension (:) :: iphz
         real(kind=idp), allocatable, dimension(:,:) :: CV
      !!!DESCRIPTORS FOR THE DISTRIBUTED SCALAPACK MATRIX, CV (if used)
         integer(blasint) :: mat_dimen,mat_dimen_r,mat_dimen_c
         integer(blasint) :: blacs_context
         integer(blasint) :: myrow,mycol
         integer(blasint) :: nprow,npcol
         integer(blasint) :: scal_block_size
         integer(blasint) :: local_row_dimen,local_col_dimen
         integer(blasint) :: descr_CV_mat(50)
         integer(blasint) :: lda
         logical :: CV_is_scalapack = .false.
      !!!DESCRIPTORS FOR THE DISTRIBUTED SCALAPACK MATRIX, CV (if used)
      contains
         procedure :: dealloc=>dealloc_CIvect
!         final :: destroy_CIvect !Use when intel finally fully implement 2003 standard
!         procedure :: destroy => destroy_CIvect
         procedure :: init_CV
         procedure :: final_CV
         procedure :: A_B_matmul
         procedure :: redistribute
         procedure :: gather_vectors
         procedure :: set_CV_element
         procedure :: add_to_CV_element
         procedure :: local_to_global
         procedure :: global_to_local
      end type CIvect

      type property_integrals
         integer :: no_of_integrals,no_of_properties
         integer, allocatable, dimension(:) :: lp, mp, nilmq, qp
         character(len=8), allocatable, dimension(:) :: property_name
         integer, allocatable, dimension(:) :: nob,mob, mpob
         integer, allocatable, dimension(:,:) :: indexv,inverted_indexv,istart
         real(kind=idp), allocatable, dimension(:,:) :: xintegrals
      end type property_integrals

      contains

!     Destructors for allocatable arrays in type definitions
!     ------------------------------------------------------------------
      subroutine dealloc_CIvect(this)
      implicit none

      class(CIvect), intent(inout) :: this

      if (allocated(this%ei)) deallocate(this%ei)
      if (allocated(this%iphz)) deallocate(this%iphz)
      if (allocated(this%CV)) deallocate(this%CV)

      end subroutine dealloc_CIvect

      subroutine dealloc_CSFheader(this)
      implicit none

      class(CSFheader), intent(inout) :: this

      if (allocated(this%iphz)) deallocate(this%iphz)
      if (allocated(this%nctgt)) deallocate(this%nctgt)
      if (allocated(this%notgt)) deallocate(this%notgt)
      if (allocated(this%numtgt)) deallocate(this%numtgt)
      if (allocated(this%itarget_symmetry_order)) deallocate(this%itarget_symmetry_order)
      if (allocated(this%idtarg)) deallocate(this%idtarg)
      if (allocated(this%mcont)) deallocate(this%mcont)
      if (allocated(this%gucont)) deallocate(this%gucont)
      if (allocated(this%nob)) deallocate(this%nob)
      if (allocated(this%ndtrf)) deallocate(this%ndtrf)
      if (allocated(this%nodo)) deallocate(this%nodo)
      if (allocated(this%nob0)) deallocate(this%nob0)
      if (allocated(this%nob0l)) deallocate(this%nob0l)
      if (allocated(this%nobl)) deallocate(this%nobl)
      if (allocated(this%no_l2_virtuals)) deallocate(this%no_l2_virtuals)

      end subroutine dealloc_CSFheader


    !> \brief  Memory cleanup for CSFbody
    !> \author J Benda
    !> \date   2010
    !>
    !> Performs automatic deallocation of members (if not already done).
    !>
    subroutine dealloc_CSFbody (this)

        use mpi_gbl, only: shared_communicator
        use mpi_memory_gbl, only: mpi_memory_deallocate_integer, mpi_memory_deallocate_real

        class(CSFbody), intent(inout) :: this

        if (associated(this % icdo)) then
            call mpi_memory_deallocate_integer(this % icdo, size(this % icdo), this % icdo_window, shared_communicator)
            nullify (this % icdo)
        end if

        if (associated(this % indo)) then
            call mpi_memory_deallocate_integer(this % indo, size(this % indo), this % indo_window, shared_communicator)
            nullify (this % indo)
        end if

        if (associated(this % ndo)) then
            call mpi_memory_deallocate_integer(this % ndo, size(this % ndo), this % ndo_window, shared_communicator)
            nullify (this % ndo)
        end if

        if (associated(this % cdo)) then
            call mpi_memory_deallocate_real(this % cdo, size(this % cdo), this % cdo_window, shared_communicator)
            nullify (this % cdo)
        end if

    end subroutine dealloc_CSFbody


    !> \brief  Class destructor for CSFbody
    !> \author J Benda
    !> \date   2020
    !>
    !> Performs automatic deallocation of members (if not already done).
    !>
    subroutine finalize_CSFbody (this)

        type(CSFbody), intent(inout) :: this

        call this % dealloc

    end subroutine finalize_CSFbody


      subroutine init_CV(this,mat_dimen_r,mat_dimen_c)
      use mpi_gbl, only: nprocs, mpi_xermsg
      use const_gbl, only: stdout
      implicit none
      class(CIvect), intent(inout) :: this
      integer, intent(in) :: mat_dimen_r, mat_dimen_c

      integer :: ido, ifail
      integer(blasint) :: info
      integer(blasint), external :: numroc !SCALAPACK function

         write(stdout,'(/," CIvect%init_CV start")')
#if defined(scalapack) && defined(usempi)
         if (this%CV_is_scalapack) then !SCALAPACK ARRAY

            this%mat_dimen_r = mat_dimen_r
            this%mat_dimen_c = mat_dimen_c

            this%mat_dimen = 0
            if (this%mat_dimen_r .eq. mat_dimen_c) this%mat_dimen = this%mat_dimen_r

!We assume that BLACS has been initialized like this somewhere else.
!            do ido=1,int( sqrt( real(nprocs) ) + 1 )
!              if(mod(nprocs,ido) .eq. 0) this%nprow = ido
!            end do
!
!            this%npcol = nprocs/this%nprow
!
!            call blacs_get( -1, 0, this%blacs_context )
!            call blacs_gridinit(  this%blacs_context , 'r', this%nprow, this%npcol )
            call blacs_gridinfo(  this%blacs_context , this%nprow, this%npcol, this%myrow, this%mycol )

            write(stdout,"('context = ',i4,' nprocs = ',i4,' matdimen = ',2i8,&
                &' nrow = ',i8,' ncol = ',i8,' myrow = ',i8,' mycol = ',i8)") &
                this % blacs_context, nprocs, this % mat_dimen_r, this % mat_dimen_c, this % nprow, &
                this % npcol, this % myrow, this % mycol

            this%scal_block_size = min ( this%mat_dimen_r/this%nprow, this%mat_dimen_c/this%npcol )
            this%scal_block_size = min(this%scal_block_size, 64_blasint)
            this%scal_block_size = max(this%scal_block_size, 1_blasint)

            this%local_row_dimen = numroc(this%mat_dimen_r,this%scal_block_size,this%myrow,0_blasint,this%nprow)
            this%local_col_dimen = numroc(this%mat_dimen_c,this%scal_block_size,this%mycol,0_blasint,this%npcol)

            this%lda = max (1_blasint,this%local_row_dimen)

            write(stdout,"('block_size = ',i4,' local_row_size = ',i8,' local_col_size = ',i8,' lda = ',i8)") &
                this % scal_block_size, this % local_row_dimen, this % local_col_dimen, this % lda

            if (this % myrow >= 0 .and. this % mycol >= 0) then
               call descinit (this % descr_CV_mat, this % mat_dimen_r, this % mat_dimen_c, this % scal_block_size, &
                              this % scal_block_size, 0_blasint, 0_blasint, this % blacs_context, this % lda, info)

               if (info /= 0) then
                  call mpi_xermsg('CIvect', 'init_CV', 'Error in getting description for A', int(info), 1)
               end if
            else
               this % descr_CV_mat(:) = -1
            end if

            if(allocated(this%CV)) deallocate(this%CV)
            allocate(this%CV(this%lda,this%local_col_dimen),stat=ifail)
            if (ifail /= 0) then
               call mpi_xermsg('CIvect', 'init_CV', 'Error in LOCAL this%CV allocation', ifail, 1)
            end if

            this%CV = 0.0_idp

         else !STANDARD ARRAY
#endif
            this%mat_dimen_r = mat_dimen_r
            this%mat_dimen_c = mat_dimen_c

            this % local_row_dimen = mat_dimen_r
            this % local_col_dimen = mat_dimen_c

            this%mat_dimen = 0
            if (this%mat_dimen_r .eq. mat_dimen_c) this%mat_dimen = this%mat_dimen_r

            if(allocated(this%CV)) deallocate(this%CV)
            allocate(this%CV(mat_dimen_r,mat_dimen_c),stat=ifail)
            if (ifail /= 0) then
               call mpi_xermsg('CIvect', 'init_CV', 'Error in this%CV allocation', ifail, 1)
            end if

            this%CV = 0.0_idp
#if defined(scalapack) && defined(usempi)
         endif
#endif
         write(stdout,'(/," CIvect%init_CV finished")')

      end subroutine init_CV


      subroutine final_CV(this)
      use const_gbl, only: stdout
      implicit none
      class(CIvect), intent(inout) :: this

         if (allocated(this%CV)) deallocate(this%CV)

         this%mat_dimen = 0
         this%mat_dimen_r = 0
         this%mat_dimen_c = 0
         this%local_row_dimen = 0
         this%local_col_dimen = 0
         this%lda = 0

      end subroutine final_CV

      !>  this%CV = matmul(A%CV,B%CV)
      subroutine A_B_matmul(this,A,B,TRANSA,TRANSB)
      use maths, only: maths_dmatrix_multiply_blas95
      class(CIvect) :: this
      class(CIvect), intent(in) :: A, B
      character(len=1), intent(in) :: TRANSA,TRANSB
      logical :: ta, tb
      real(kind=idp), parameter :: alpha = 1.0_idp, beta = 0.0_idp
      integer(blasint), parameter :: one = 1
      integer(blasint) :: m, n, k
#if defined(scalapack) && defined(usempi)
         if (this%CV_is_scalapack) then !SCALAPACK ARRAY
            if (A%CV_is_scalapack .and. B%CV_is_scalapack) then
               m = this%mat_dimen_r
               n = this%mat_dimen_c
               if ((TRANSA .eq. 'T') .or. (TRANSA .eq. 't')) then
                  k = A%mat_dimen_r
               else
                  k = A%mat_dimen_c
               endif
               call pdgemm (transa, transb, m, n, k, alpha, A % CV, one, one, A % descr_CV_mat(1:9), B % CV, one, one, &
                            B % descr_CV_mat(1:9), beta, this % CV, one, one, this % descr_CV_mat(1:9))
            else
               print *,'A_B_matmul: at least one of A,B are not in SCALAPACK format!!!'
               stop
            endif
         else !STANDARD ARRAY
#endif
            ta = (TRANSA == 'T' .or. TRANSA == 't')
            tb = (TRANSB == 'T' .or. TRANSB == 't')
            call maths_dmatrix_multiply_blas95(A % CV, B % CV, this % CV, ta, tb)
#if defined(scalapack) && defined(usempi)
         endif
#endif

      end subroutine A_B_matmul

      subroutine gather_vectors(this,vec,nreq,rdest,cdest)
      implicit none
      class(CIvect), intent(in) :: this
      real(kind=idp), allocatable :: vec(:,:)
      integer, intent(in) :: nreq,rdest,cdest

      integer(blasint) :: err, ig, jg, il, jl, iprow, ipcol
#if defined(scalapack) && defined(usempi)
         if (this%CV_is_scalapack) then !SCALAPACK ARRAY

            call blacs_barrier(this%blacs_context, 'a')

            if (nreq > this%mat_dimen_c) then
               print *,'nreq outside of allowed range',nreq,this%mat_dimen_c
               stop
            endif

            if (allocated(vec)) deallocate(vec)
            allocate(vec(this%mat_dimen_r,nreq),stat=err)
            if (err .ne. 0) then
               print *,'allocation error in gather_vectors'
               stop
            endif

            vec = 0.0_idp
            do jg=1,nreq
               do ig=1, this%mat_dimen_r
                  call infog2l(ig, jg, this%descr_CV_mat(1:9), this%nprow, this%npcol, this%myrow, this%mycol, il, jl, iprow, ipcol)
                  if (this%myrow==iprow .and. this%mycol==ipcol) then
                     vec(ig,jg) = this%CV(il,jl)
                  endif
               enddo
            enddo

            !Gather the vectors on process with coordinates (rdest, cdest) if (rdest=-1, cdest=-1) then the vector is gathered on all processes
            call dgsum2d(this%blacs_context, 'all', ' ', this%mat_dimen_r, nreq, vec, this%mat_dimen_r, rdest, cdest )

         else !STANDARD ARRAY
#endif
            if (allocated(vec)) deallocate(vec)
            allocate(vec(size(this%CV,1),nreq),stat=err)
            if (err .ne. 0) then
               print *,'allocation error in gather_vectors standard',size(this%CV,1),nreq
               stop
            endif

            vec(:,1:nreq) = this%CV(:,1:nreq)
#if defined(scalapack) && defined(usempi)
         endif
#endif
      end subroutine gather_vectors


        !> \brief   Redistribute matrix between two BLACS contexts
        !> \authors J Benda
        !> \date    2019
        !>
        !> Redistribute matrix, present in one BLACS context, to another BLACS context. The source and target
        !> matrix sizes do not have to be equal. In that case, only the largest common submatrix will be redistributed.
        !>
        !> \param[inout] this         Distributed matrix to write.
        !> \param[in]    that         Distributed matrix to read.
        !> \param[in]    context_opt  BLACS context that contains both all source and all destination MPI processes.
        !>                            If not provided, the subroutine will use this % blacs_context instead.
        !>
        subroutine redistribute (this, that, context_opt)

            class(CIvect), intent(inout)        :: this
            type(CIvect),  intent(in)           :: that
            integer(blasint), intent(in), optional :: context_opt

            integer(blasint) :: context, m, n, one = 1

            m = min(this % mat_dimen_r, that % mat_dimen_r)
            n = min(this % mat_dimen_c, that % mat_dimen_c)

#if defined(scalapack) && defined(usempi)
            if (this % CV_is_scalapack) then
                if (present(context_opt)) then
                    context = context_opt
                else
                    context = this % blacs_context
                end if

                call pdgemr2d(m, n, that % CV, one, one, that % descr_CV_mat(1:9), &
                                    this % CV, one, one, this % descr_CV_mat(1:9), context)
            else
#endif
                this % CV(1:m,1:n) = that % CV(1:m,1:n)
#if defined(scalapack) && defined(usempi)
            end if
#endif
        end subroutine redistribute


        !> \brief   Convert local matrix indices to global ones
        !> \authors J Benda
        !> \date    2019
        !>
        !> In case of distributed matrix, get the real coordinates (in the global matrix).
        !> Trivially returns the same indices in non-distributed case.
        !>
        !> \param[in]  this   Distributed matrix to read.
        !> \param[in]  i_loc  Row index in the local portion of the matrix.
        !> \param[in]  j_loc  Column index in the local portion of the matrix.
        !> \param[out] i      Global row index.
        !> \param[out] j      Global column index.
        !>
        subroutine local_to_global (this, i_loc, j_loc, i, j)

            class(CIvect), intent(in)  :: this
            integer,       intent(in)  :: i_loc, j_loc
            integer,       intent(out) :: i, j
#if defined (usempi) && defined(scalapack)
            integer(blasint), external :: indxl2g
#endif
            if (i_loc < 1 .or. this % local_row_dimen < i_loc .or. &
                j_loc < 1 .or. this % local_col_dimen < j_loc) then
                i = -1
                j = -1
                return
            end if

#if defined(usempi) && defined(scalapack)
            if (this % CV_is_scalapack) then
                i = indxl2g(i_loc, this % scal_block_size, this % myrow, 0_blasint, this % nprow)
                j = indxl2g(j_loc, this % scal_block_size, this % mycol, 0_blasint, this % npcol)
            else
#endif
                i = i_loc
                j = j_loc
#if defined(usempi) && defined(scalapack)
            end if
#endif
        end subroutine local_to_global


        !> \brief   Convert global matrix indices to local ones
        !> \authors J Benda
        !> \date    2019
        !>
        !> In case of distributed matrix, get the local coordinates corresponding to
        !> the global ones. Trivially returns the same indices in non-distributed case.
        !> If the specified global matrix row/column is not located at the current processor,
        !> -1 is returned in the corresponding index. Only when both `i_loc` and
        !> `j_loc` are positive, the element specified by the `i`,`j` pair is located
        !> at the current processor.
        !>
        !> \param[in]  this   Distributed matrix to read.
        !> \param[in]  i      Global row index.
        !> \param[in]  j      Global column index.
        !> \param[out] i_loc  Row index in the local portion of the matrix.
        !> \param[out] j_loc  Column index in the local portion of the matrix.
        !>
        subroutine global_to_local (this, i, j, i_loc, j_loc)

            class(CIvect), intent(in)  :: this
            integer,       intent(in)  :: i, j
            integer,       intent(out) :: i_loc, j_loc
#if defined(usempi) && defined(scalapack)
            integer(blasint) :: proc_row, proc_col, zero = 0, gi, gj, li, lj
            external :: infog1l
#endif
            if (i < 1 .or. this % mat_dimen_r < i .or. &
                j < 1 .or. this % mat_dimen_c < j) then
                i_loc = -1
                j_loc = -1
                return
            end if

#if defined(usempi) && defined(scalapack)
            if (this % CV_is_scalapack) then
                gi = i; call infog1l(gi, this % scal_block_size, this % nprow, this % myrow, zero, li, proc_row)
                gj = j; call infog1l(gj, this % scal_block_size, this % npcol, this % mycol, zero, lj, proc_col)
                i_loc = li; if (proc_row /= this % myrow) i_loc = -1
                j_loc = lj; if (proc_col /= this % mycol) j_loc = -1
            else
#endif
                i_loc = i
                j_loc = j
#if defined(usempi) && defined(scalapack)
            end if
#endif
        end subroutine global_to_local


      subroutine set_CV_element(this,val,i,j)
      implicit none
      class(CIvect), intent(inout) :: this
      real(kind=idp), intent(in) :: val
      integer, intent(in) :: i,j

      integer(blasint) :: i_loc, j_loc, proc_row, proc_col
#if defined(scalapack) && defined(usempi)
         if (this%CV_is_scalapack) then !SCALAPACK ARRAY
            !Figure out which proc it belongs to and the local matrix index
            call infog2l(int(i, blasint), int(j, blasint), this % descr_CV_mat(1:9), &
                         this % nprow, this % npcol, this % myrow, this % mycol, i_loc, j_loc, proc_row, proc_col)
            if ((this%myrow .eq. proc_row) .and. (this%mycol .eq. proc_col)) then
               if ((i_loc > this%local_row_dimen) .or. (j_loc > this%local_col_dimen)) then
                  print *,'error inserting element',i,j,i_loc,j_loc,this%local_row_dimen,this%local_col_dimen
               endif
               this%CV(i_loc, j_loc) = val
            endif
         else !STANDARD ARRAY
#endif
            this%CV(i,j) = val
#if defined(scalapack) && defined(usempi)
         endif
#endif
      end subroutine set_CV_element

      subroutine add_to_CV_element(this,val,i,j)
      implicit none
      class(CIvect), intent(inout) :: this
      real(kind=idp), intent(in) :: val
      integer, intent(in) :: i,j

      integer(blasint) :: i_loc, j_loc, proc_row, proc_col
#if defined(scalapack) && defined(usempi)
         if (this%CV_is_scalapack) then !SCALAPACK ARRAY
            !Figure out which proc it belongs to and the local matrix index
            call infog2l(int(i, blasint), int(j, blasint), this % descr_CV_mat(1:9), &
                         this % nprow, this % npcol, this % myrow, this % mycol, i_loc, j_loc, proc_row, proc_col)
            if ((this%myrow .eq. proc_row) .and. (this%mycol .eq. proc_col)) then
               if ((i_loc > this%local_row_dimen) .or. (j_loc > this%local_col_dimen)) then
                  print *,'error adding into element',i,j,i_loc,j_loc,this%local_row_dimen,this%local_col_dimen
               endif
               this%CV(i_loc, j_loc) = this%CV(i_loc, j_loc) + val
            endif
         else !STANDARD ARRAY
#endif
            this%CV(i,j) = this%CV(i,j) + val
#if defined(scalapack) && defined(usempi)
         endif
#endif
      end subroutine add_to_CV_element


      end module
