
////////////////////////////////////////////////////////////////////////
//
//                 Cambridge Crystallographic Data Centre
//                                CCDC 
//
////////////////////////////////////////////////////////////////////////
//
// For information on free software for rendering this file to create very
// high quality bitmap images, please visit the POV-Ray website:
// www.povray.org.
//
// This POV-Ray output mechanism has been tested with POV-Ray version 3.7.
//
// The CCDC is keen to receive feedback on, and suggestions for improvements
// to, this POV-Ray output mechanism. Please visit www.ccdc.cam.ac.uk,
// or e-mail feedback to support@ccdc.cam.ac.uk. Thank you.
//
////////////////////////////////////////////////////////////////////////
//
// If this POVRay file has been generated from an entry in the Cambridge 
// Structural Database, then it will include bibliographic, chemical, crystal, 
// experimental, refinement or atomic coordinate data resulting from the CCDC's 
// data processing and validation procedures.
//
////////////////////////////////////////////////////////////////////////

#version 3.7;
global_settings {
    assumed_gamma 2.2
    max_trace_level 5
}

// -----------------------------------------------------------

#macro rotate_view_for_animation()
    // If using the [filename.pov].ini file, with animation enabled, this will produce
    // a cyclic animation of the scene rotating, otherwise this will not affect the image:
    rotate <0,clock*360,0>
#end

#macro ccdc_perspective_camera( camera_position, field_of_view )
    camera {
        perspective
        location camera_position
        up    <0,1,0>
        right  -x * (image_width/image_height)
        look_at <0,0,-100>
        // Convert the vertical field of view to the horizontal field of view
        angle degrees(2 * atan2(tan(radians(field_of_view / 2)) * image_width, image_height))

        rotate_view_for_animation()
    }
#end

#macro ccdc_directional_light_source( light_position, light_diffuse_colour, light_specular_colour )
    // The scalar multiplier applied to light_position seems to be needed for correct brightness
    light_source {
        5 * light_position
        light_diffuse_colour
        parallel
        shadowless
        rotate_view_for_animation()
    }
    light_source {
        5 * light_position
        light_specular_colour
        parallel
        rotate_view_for_animation()
    }
#end

#macro ccdc_ambient_light_source( light_colour )
    global_settings { ambient_light light_colour * 10 }
#end

#macro ccdc_background_colour( background_colour )
    background { background_colour }
#end

#macro ccdc_background_gradient( top_right_colour, top_left_colour, bottom_left_colour, bottom_right_colour )
    // TODO - adjust to use all four colours
    background { bottom_left_colour }
#end

#macro ccdc_orient_world( world_orientation )
    transform { world_orientation }
#end

#macro ccdc_orient_structure( structure_orientation )
    transform { structure_orientation }
#end

#macro ccdc_set_standard_mercury_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.2
            roughness 0.02
        }
    }
#end

#macro ccdc_set_shiny_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.8
            roughness 0.02
        }
    }
#end

#macro ccdc_set_matt_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.0
            roughness 0.02
        }
    }
#end

#macro ccdc_set_metallic_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.2
            diffuse 0.2
            specular 1.0
            roughness 0.02 
            metallic
            reflection { 0.5, 1.0
               fresnel on
               metallic 0.8
            }
        }
    }
#end

#macro ccdc_set_iridescent_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.1
            diffuse 0.1
            reflection 0.2
            specular 1
            roughness 0.02
            irid {
              0.35
              thickness 0.5
              turbulence 0.5
            }
        }
    }
#end

#macro ccdc_set_solid_material_properties( object_color )
    // change the method call here to select different properties settings
    ccdc_set_shiny_solid_material_properties( object_color )
#end

#macro ccdc_set_wireframe_material_properties( object_color )
    no_shadow
    pigment { object_color }
#end

#macro ccdc_set_point_size( point_size )
    // TODO
#end

#macro ccdc_draw_circle( centre, circle_radius, transformation, circle_color )
    torus {
        circle_radius, 0.03
        transform { transformation }
        translate centre
        ccdc_set_wireframe_material_properties( circle_color )
    }
#end

#macro ccdc_draw_wireframe_sphere( centre, sphere_radius, sphere_color )
    union {
        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )

    }
#end

#macro ccdc_draw_solid_sphere( position, sphere_radius, sphere_color )
    sphere {
        position, sphere_radius
        ccdc_set_solid_material_properties( sphere_color )
    }
#end

#macro ccdc_draw_solid_polygon( vertices, origin, transformation, polygon_color )
    #local nvertices = dimension_size( vertices, 1 );
    polygon {
        nvertices + 1
        #local i = 0;
        #while ( i < nvertices )
            vertices[ i ]
            #local i = i + 1;
        #end
        // Repeat the first point, to close the polygon:
        vertices[ 0 ]
        translate origin
        transform { transformation inverse }
        ccdc_set_solid_material_properties( polygon_color )
    }
#end

#macro ccdc_draw_solid_torus( centre, ring_radius, torus_radius, transformation, torus_color )
    torus {
        ring_radius, torus_radius
        transform { transformation }
        translate centre
        ccdc_set_solid_material_properties( torus_color )
    }
#end

#macro ccdc_draw_disk( centre, disk_radius, disk_normal, disk_color )
    // TODO
#end

#macro ccdc_draw_line_segment( line_begin, line_end, line_color )
    cylinder {
        line_begin, line_end, 0.03
        ccdc_set_wireframe_material_properties( line_color )
    }
#end

#macro ccdc_draw_stippled_line_segment( line_begin, line_end, stipple, stipple_scale_factor, line_color )
    // TODO - adjust to use stipple and stipple_scale_factor
    /* For example:

        AACRUB delocalised bonds:
        61680 = F0F0
        
        AABHTZ contacts:
        43690 = AAAA
    */
    // The following is approximately correct when drawing contacts
    #declare nsteps = 20;
    #declare increment = 1 / nsteps;
    #declare scalar1 = 0.0;
    #while ( scalar1 < 1.0 )
        #declare point1 = ( scalar1 * line_begin ) + ( 1 - scalar1 ) * line_end;
        #declare scalar2 = scalar1 + ( increment / 2.0 );
        #declare point2 = ( scalar2 * line_begin ) + ( 1 - scalar2 ) * line_end;
        cylinder {
            point1, point2, 0.03
            ccdc_set_wireframe_material_properties( line_color )
        }
        #declare scalar1 = scalar1 + increment;
    #end
#end

#macro ccdc_draw_wireframe_arc( arc_centre, arc_end_1, arc_end_2,
                                stipple, stipple_scale_factor, line_color )
    // TODO
#end

#macro ccdc_draw_wireframe_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_colour )
    union {
        #local circle_radius = 1.0;
        #local origin = < 0, 0, 0 >;

        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )

        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >

        translate centre
    }
#end

#macro ccdc_draw_open_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        open
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_closed_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_solid_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_color )
    sphere {
        < 0, 0, 0 >, 1
        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( ellipsoid_color )
    }
#end

#macro ccdc_draw_ellipsoid_axes( centre, axis1, axis2, axis3, axes_color )
    union {
        cylinder {
            < 0.05, 0, 0 >, < -0.05, 0, 0 >, 1
        }
        cylinder {
            < 0, 0.05, 0 >, < 0, -0.05, 0 >, 1
        }
        cylinder {
            < 0, 0, 0.05 >, < 0, 0, -0.05 >, 1
        }
        #local axis_height = 0.01;
        scale < vlength( axis1 ) + axis_height, vlength( axis2 ) + axis_height, vlength( axis3 ) + axis_height >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( axes_color )
    }
#end

#macro ccdc_draw_point( point_location, point_color )
    // TODO
#end

#macro ccdc_draw_solid_triangle( corner1, corner2, corner3,
                                 color1, color2, color3,
                                 normal1, normal2, normal3 )
   smooth_triangle {
        corner1, normal1,
        corner2, normal2,
        corner3, normal3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_solid_triangle_without_normals( corner1, corner2, corner3,
                                                 color1, color2, color3 )
   triangle {
        corner1,
        corner2,
        corner3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_text( position, message, text_color, text_offset )
    // TODO - Adjust so that the text orientation is correct
    text { ttf font_name, message, 0.1, 0
        ccdc_set_wireframe_material_properties( color text_color )
        /*
        finish {
            ambient 0.2
            diffuse 0.6
            phong 0.3
            phong_size 100
        }
        */
        scale < font_scale, font_scale, font_scale >
        // rotate so text always faces the camera when animated
        rotate <0,clock*360,0>
        translate text_offset
        transform { world_rotation inverse }
        transform { structure_rotation inverse }
        translate position
    }
#end

#macro ccdc_draw_static_text( message )
    // TODO
#end

// -----------------------------------------------------------

// You can edit the file "ccdc_macro_overrides.inc" in this directory
// to override the implementations of any or all the above POVRay macros:
#include "ccdc_macro_overrides.inc"

#declare font_scale = 0.578879;

#declare font_name = "arial.ttf";

ccdc_perspective_camera( < 0, 0, 33167.3 >, 0.05 )
ccdc_directional_light_source( < 1, 1, 1 >, rgb < 0.698039, 0.698039, 0.698039 >, rgb < 1, 1, 1 > )
ccdc_directional_light_source( < -1, 0.2, 1 >, rgb < 0.501961, 0.501961, 0.501961 >, rgb < 0.501961, 0.501961, 0.501961 > )
ccdc_ambient_light_source( rgb < 0.301961, 0.301961, 0.301961 > )
ccdc_background_colour( rgb < 1, 1, 1 > )
union {
    #declare world_orientation = transform { matrix <
        0.117139, -0.963475, -0.240822,
        -0.858084, 0.0238875, -0.512954,
        0.499971, 0.266732, -0.823944,
        0, 0, 0 > };
    #declare world_rotation = transform { matrix <
        0.117139, -0.963475, -0.240822,
        -0.858084, 0.0238875, -0.512954,
        0.499971, 0.266732, -0.823944,
        0, 0, 0 > };
    union {
        // jr286_2
        #declare structure_orientation = transform { matrix <
            -0.974519, -0.0504268, 0.218565,
            0.0422676, 0.915662, 0.399719,
            -0.220288, 0.398772, -0.890199,
            11.1371, -18.1377, 1.21358 > };
        #declare structure_rotation = transform { matrix <
            -0.974519, -0.0504268, 0.218565,
            0.0422676, 0.915662, 0.399719,
            -0.220288, 0.398772, -0.890199,
            0, 0, 0 > };
        // jr286_2 atoms
        // atom Cr1B
        ccdc_draw_solid_sphere( < 15.2598, 19.6988, 10.2593 >, 0.5, rgb < 1, 0.647059, 0 > )
        // atom N3B
        ccdc_draw_solid_sphere( < 15.5326, 17.662, 10.3779 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom N2B
        ccdc_draw_solid_sphere( < 17.2852, 19.888, 10.3721 >, 0.1, rgb < 0, 0, 1 > )
        // atom N1B
        ccdc_draw_solid_sphere( < 15.1743, 21.7402, 10.0377 >, 0.1, rgb < 0, 0, 0 > )
        // atom N4B
        ccdc_draw_solid_sphere( < 15.0938, 19.7571, 12.3075 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom N5B
        ccdc_draw_solid_sphere( < 13.2117, 19.5863, 10.2506 >, 0.1, rgb < 1, 0, 0 > )
        // atom N6B
        ccdc_draw_solid_sphere( < 15.1584, 19.5467, 8.22547 >, 0.1, rgb < 1, 1, 0 > )
        // atom N7B
        ccdc_draw_solid_sphere( < 13.1558, 21.0715, 12.115 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N8B
        ccdc_draw_solid_sphere( < 13.2979, 18.1097, 8.38521 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom C23B
        ccdc_draw_solid_sphere( < 14.5688, 16.9191, 10.913 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C22B
        ccdc_draw_solid_sphere( < 14.3227, 15.6019, 10.5321 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C21B
        ccdc_draw_solid_sphere( < 15.064, 15.0656, 9.51014 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C19B
        ccdc_draw_solid_sphere( < 16.1434, 15.7899, 8.97399 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C20B
        ccdc_draw_solid_sphere( < 16.4052, 17.0862, 9.48403 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C18B
        ccdc_draw_solid_sphere( < 16.9712, 15.2728, 7.95595 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C17B
        ccdc_draw_solid_sphere( < 18.0044, 15.9934, 7.47839 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C16B
        ccdc_draw_solid_sphere( < 18.3194, 17.2411, 8.04641 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C15B
        ccdc_draw_solid_sphere( < 17.5746, 17.7879, 9.07452 >, 0.1, rgb < 0, 0.501961, 0 > )
        // atom C14B
        ccdc_draw_solid_sphere( < 18.1155, 18.9559, 9.81296 >, 0.1, rgb < 0, 0, 1 > )
        // atom C13B
        ccdc_draw_solid_sphere( < 19.4935, 19.0573, 9.97146 >, 0.1, rgb < 0, 0, 1 > )
        // atom C12B
        ccdc_draw_solid_sphere( < 20.0318, 20.1104, 10.663 >, 0.1, rgb < 0, 0, 1 > )
        // atom C11B
        ccdc_draw_solid_sphere( < 19.1975, 21.0514, 11.2132 >, 0.1, rgb < 0, 0, 1 > )
        // atom C10B
        ccdc_draw_solid_sphere( < 17.8251, 20.9283, 11.0752 >, 0.1, rgb < 0, 0, 1 > )
        // atom C9B
        ccdc_draw_solid_sphere( < 16.9698, 21.9903, 11.6426 >, 0.1, rgb < 0, 0, 0 > )
        // atom C8B
        ccdc_draw_solid_sphere( < 17.3983, 22.6684, 12.7715 >, 0.1, rgb < 0, 0, 0 > )
        // atom C7B
        ccdc_draw_solid_sphere( < 16.7806, 23.852, 13.2128 >, 0.1, rgb < 0, 0, 0 > )
        // atom C6B
        ccdc_draw_solid_sphere( < 15.7762, 24.4109, 12.4966 >, 0.1, rgb < 0, 0, 0 > )
        // atom C4B
        ccdc_draw_solid_sphere( < 15.2715, 23.7469, 11.3515 >, 0.1, rgb < 0, 0, 0 > )
        // atom C5B
        ccdc_draw_solid_sphere( < 15.8027, 22.4919, 10.9913 >, 0.1, rgb < 0, 0, 0 > )
        // atom C3B
        ccdc_draw_solid_sphere( < 14.2457, 24.2937, 10.559 >, 0.1, rgb < 0, 0, 0 > )
        // atom C2B
        ccdc_draw_solid_sphere( < 13.7672, 23.5955, 9.48609 >, 0.1, rgb < 0, 0, 0 > )
        // atom C1B
        ccdc_draw_solid_sphere( < 14.221, 22.292, 9.2914 >, 0.1, rgb < 0, 0, 0 > )
        // atom C24B
        ccdc_draw_solid_sphere( < 16.0394, 19.1564, 13.0677 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom C25B
        ccdc_draw_solid_sphere( < 16.1529, 19.3928, 14.4106 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom C26B
        ccdc_draw_solid_sphere( < 15.2599, 20.2678, 15.0092 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom C27B
        ccdc_draw_solid_sphere( < 14.2554, 20.8303, 14.2697 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom C28B
        ccdc_draw_solid_sphere( < 14.1788, 20.5391, 12.9059 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // atom C29B
        ccdc_draw_solid_sphere( < 12.3903, 22.1897, 12.6878 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C30B
        ccdc_draw_solid_sphere( < 12.4766, 20.2678, 11.1729 >, 0.1, rgb < 1, 0, 0 > )
        // atom C31B
        ccdc_draw_solid_sphere( < 11.0967, 20.1993, 11.2076 >, 0.1, rgb < 1, 0, 0 > )
        // atom C32B
        ccdc_draw_solid_sphere( < 10.4514, 19.4243, 10.2735 >, 0.1, rgb < 1, 0, 0 > )
        // atom C33B
        ccdc_draw_solid_sphere( < 11.1675, 18.7458, 9.32327 >, 0.1, rgb < 1, 0, 0 > )
        // atom C34B
        ccdc_draw_solid_sphere( < 12.5512, 18.8332, 9.3282 >, 0.1, rgb < 1, 0, 0 > )
        // atom C35B
        ccdc_draw_solid_sphere( < 12.624, 16.9179, 7.82603 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C36B
        ccdc_draw_solid_sphere( < 14.3198, 18.6697, 7.63237 >, 0.1, rgb < 1, 1, 0 > )
        // atom C37B
        ccdc_draw_solid_sphere( < 14.5127, 18.3052, 6.28767 >, 0.1, rgb < 1, 1, 0 > )
        // atom C38B
        ccdc_draw_solid_sphere( < 15.5162, 18.8846, 5.57225 >, 0.1, rgb < 1, 1, 0 > )
        // atom C39B
        ccdc_draw_solid_sphere( < 16.3177, 19.861, 6.14931 >, 0.1, rgb < 1, 1, 0 > )
        // atom C40B
        ccdc_draw_solid_sphere( < 16.1051, 20.1748, 7.46873 >, 0.1, rgb < 1, 1, 0 > )
        // jr286_2 bonds
        // bond Cr1B-N3B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 15.3962, 18.6804, 10.3186 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 15.3962, 18.6804, 10.3186 >, < 15.5326, 17.662, 10.3779 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond Cr1B-N2B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 16.2725, 19.7934, 10.3157 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 16.2725, 19.7934, 10.3157 >, < 17.2852, 19.888, 10.3721 >, 0.1, rgb < 0, 0, 1 > )
        // bond Cr1B-N1B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 15.2171, 20.7195, 10.1485 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 15.2171, 20.7195, 10.1485 >, < 15.1743, 21.7402, 10.0377 >, 0.1, rgb < 0, 0, 0 > )
        // bond Cr1B-N4B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 15.1768, 19.728, 11.2834 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 15.1768, 19.728, 11.2834 >, < 15.0938, 19.7571, 12.3075 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond Cr1B-N5B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 14.2358, 19.6425, 10.255 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 14.2358, 19.6425, 10.255 >, < 13.2117, 19.5863, 10.2506 >, 0.1, rgb < 1, 0, 0 > )
        // bond Cr1B-N6B
        ccdc_draw_closed_cylinder( < 15.2598, 19.6988, 10.2593 >, < 15.2091, 19.6228, 9.24237 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 15.2091, 19.6228, 9.24237 >, < 15.1584, 19.5467, 8.22547 >, 0.1, rgb < 1, 1, 0 > )
        // bond N3B-C23B
        ccdc_draw_closed_cylinder( < 15.5326, 17.662, 10.3779 >, < 14.5688, 16.9191, 10.913 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond N3B-C20B
        ccdc_draw_closed_cylinder( < 15.5326, 17.662, 10.3779 >, < 16.4052, 17.0862, 9.48403 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond N2B-C14B
        ccdc_draw_closed_cylinder( < 17.2852, 19.888, 10.3721 >, < 18.1155, 18.9559, 9.81296 >, 0.1, rgb < 0, 0, 1 > )
        // bond N2B-C10B
        ccdc_draw_closed_cylinder( < 17.2852, 19.888, 10.3721 >, < 17.8251, 20.9283, 11.0752 >, 0.1, rgb < 0, 0, 1 > )
        // bond N1B-C5B
        ccdc_draw_closed_cylinder( < 15.1743, 21.7402, 10.0377 >, < 15.8027, 22.4919, 10.9913 >, 0.1, rgb < 0, 0, 0 > )
        // bond N1B-C1B
        ccdc_draw_closed_cylinder( < 15.1743, 21.7402, 10.0377 >, < 14.221, 22.292, 9.2914 >, 0.1, rgb < 0, 0, 0 > )
        // bond N4B-C24B
        ccdc_draw_closed_cylinder( < 15.0938, 19.7571, 12.3075 >, < 16.0394, 19.1564, 13.0677 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond N4B-C28B
        ccdc_draw_closed_cylinder( < 15.0938, 19.7571, 12.3075 >, < 14.1788, 20.5391, 12.9059 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond N5B-C30B
        ccdc_draw_closed_cylinder( < 13.2117, 19.5863, 10.2506 >, < 12.4766, 20.2678, 11.1729 >, 0.1, rgb < 1, 0, 0 > )
        // bond N5B-C34B
        ccdc_draw_closed_cylinder( < 13.2117, 19.5863, 10.2506 >, < 12.5512, 18.8332, 9.3282 >, 0.1, rgb < 1, 0, 0 > )
        // bond N6B-C36B
        ccdc_draw_closed_cylinder( < 15.1584, 19.5467, 8.22547 >, < 14.3198, 18.6697, 7.63237 >, 0.1, rgb < 1, 1, 0 > )
        // bond N6B-C40B
        ccdc_draw_closed_cylinder( < 15.1584, 19.5467, 8.22547 >, < 16.1051, 20.1748, 7.46873 >, 0.1, rgb < 1, 1, 0 > )
        // bond N7B-C28B
        ccdc_draw_closed_cylinder( < 13.1558, 21.0715, 12.115 >, < 13.6673, 20.8053, 12.5105 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 13.6673, 20.8053, 12.5105 >, < 14.1788, 20.5391, 12.9059 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond N7B-C29B
        ccdc_draw_closed_cylinder( < 13.1558, 21.0715, 12.115 >, < 12.7731, 21.6306, 12.4014 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 12.7731, 21.6306, 12.4014 >, < 12.3903, 22.1897, 12.6878 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N7B-C30B
        ccdc_draw_closed_cylinder( < 13.1558, 21.0715, 12.115 >, < 12.8162, 20.6696, 11.644 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 12.8162, 20.6696, 11.644 >, < 12.4766, 20.2678, 11.1729 >, 0.1, rgb < 1, 0, 0 > )
        // bond N8B-C34B
        ccdc_draw_closed_cylinder( < 13.2979, 18.1097, 8.38521 >, < 12.9245, 18.4715, 8.8567 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 12.9245, 18.4715, 8.8567 >, < 12.5512, 18.8332, 9.3282 >, 0.1, rgb < 1, 0, 0 > )
        // bond N8B-C35B
        ccdc_draw_closed_cylinder( < 13.2979, 18.1097, 8.38521 >, < 12.961, 17.5138, 8.10562 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 12.961, 17.5138, 8.10562 >, < 12.624, 16.9179, 7.82603 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N8B-C36B
        ccdc_draw_closed_cylinder( < 13.2979, 18.1097, 8.38521 >, < 13.8089, 18.3897, 8.00879 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 13.8089, 18.3897, 8.00879 >, < 14.3198, 18.6697, 7.63237 >, 0.1, rgb < 1, 1, 0 > )
        // bond C23B-C22B
        ccdc_draw_closed_cylinder( < 14.5688, 16.9191, 10.913 >, < 14.3227, 15.6019, 10.5321 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C22B-C21B
        ccdc_draw_closed_cylinder( < 14.3227, 15.6019, 10.5321 >, < 15.064, 15.0656, 9.51014 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C21B-C19B
        ccdc_draw_closed_cylinder( < 15.064, 15.0656, 9.51014 >, < 16.1434, 15.7899, 8.97399 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C19B-C20B
        ccdc_draw_closed_cylinder( < 16.1434, 15.7899, 8.97399 >, < 16.4052, 17.0862, 9.48403 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C19B-C18B
        ccdc_draw_closed_cylinder( < 16.1434, 15.7899, 8.97399 >, < 16.9712, 15.2728, 7.95595 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C20B-C15B
        ccdc_draw_closed_cylinder( < 16.4052, 17.0862, 9.48403 >, < 17.5746, 17.7879, 9.07452 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C18B-C17B
        ccdc_draw_closed_cylinder( < 16.9712, 15.2728, 7.95595 >, < 18.0044, 15.9934, 7.47839 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C17B-C16B
        ccdc_draw_closed_cylinder( < 18.0044, 15.9934, 7.47839 >, < 18.3194, 17.2411, 8.04641 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C16B-C15B
        ccdc_draw_closed_cylinder( < 18.3194, 17.2411, 8.04641 >, < 17.5746, 17.7879, 9.07452 >, 0.1, rgb < 0, 0.501961, 0 > )
        // bond C15B-C14B
        ccdc_draw_closed_cylinder( < 17.5746, 17.7879, 9.07452 >, < 17.8451, 18.3719, 9.44374 >, 0.1, rgb < 0, 0.501961, 0 > )
        ccdc_draw_closed_cylinder( < 17.8451, 18.3719, 9.44374 >, < 18.1155, 18.9559, 9.81296 >, 0.1, rgb < 0, 0, 1 > )
        // bond C14B-C13B
        ccdc_draw_closed_cylinder( < 18.1155, 18.9559, 9.81296 >, < 19.4935, 19.0573, 9.97146 >, 0.1, rgb < 0, 0, 1 > )
        // bond C13B-C12B
        ccdc_draw_closed_cylinder( < 19.4935, 19.0573, 9.97146 >, < 20.0318, 20.1104, 10.663 >, 0.1, rgb < 0, 0, 1 > )
        // bond C12B-C11B
        ccdc_draw_closed_cylinder( < 20.0318, 20.1104, 10.663 >, < 19.1975, 21.0514, 11.2132 >, 0.1, rgb < 0, 0, 1 > )
        // bond C11B-C10B
        ccdc_draw_closed_cylinder( < 19.1975, 21.0514, 11.2132 >, < 17.8251, 20.9283, 11.0752 >, 0.1, rgb < 0, 0, 1 > )
        // bond C10B-C9B
        ccdc_draw_closed_cylinder( < 17.8251, 20.9283, 11.0752 >, < 17.3975, 21.4593, 11.3589 >, 0.1, rgb < 0, 0, 1 > )
        ccdc_draw_closed_cylinder( < 17.3975, 21.4593, 11.3589 >, < 16.9698, 21.9903, 11.6426 >, 0.1, rgb < 0, 0, 0 > )
        // bond C9B-C8B
        ccdc_draw_closed_cylinder( < 16.9698, 21.9903, 11.6426 >, < 17.3983, 22.6684, 12.7715 >, 0.1, rgb < 0, 0, 0 > )
        // bond C9B-C5B
        ccdc_draw_closed_cylinder( < 16.9698, 21.9903, 11.6426 >, < 15.8027, 22.4919, 10.9913 >, 0.1, rgb < 0, 0, 0 > )
        // bond C8B-C7B
        ccdc_draw_closed_cylinder( < 17.3983, 22.6684, 12.7715 >, < 16.7806, 23.852, 13.2128 >, 0.1, rgb < 0, 0, 0 > )
        // bond C7B-C6B
        ccdc_draw_closed_cylinder( < 16.7806, 23.852, 13.2128 >, < 15.7762, 24.4109, 12.4966 >, 0.1, rgb < 0, 0, 0 > )
        // bond C6B-C4B
        ccdc_draw_closed_cylinder( < 15.7762, 24.4109, 12.4966 >, < 15.2715, 23.7469, 11.3515 >, 0.1, rgb < 0, 0, 0 > )
        // bond C4B-C5B
        ccdc_draw_closed_cylinder( < 15.2715, 23.7469, 11.3515 >, < 15.8027, 22.4919, 10.9913 >, 0.1, rgb < 0, 0, 0 > )
        // bond C4B-C3B
        ccdc_draw_closed_cylinder( < 15.2715, 23.7469, 11.3515 >, < 14.2457, 24.2937, 10.559 >, 0.1, rgb < 0, 0, 0 > )
        // bond C3B-C2B
        ccdc_draw_closed_cylinder( < 14.2457, 24.2937, 10.559 >, < 13.7672, 23.5955, 9.48609 >, 0.1, rgb < 0, 0, 0 > )
        // bond C2B-C1B
        ccdc_draw_closed_cylinder( < 13.7672, 23.5955, 9.48609 >, < 14.221, 22.292, 9.2914 >, 0.1, rgb < 0, 0, 0 > )
        // bond C24B-C25B
        ccdc_draw_closed_cylinder( < 16.0394, 19.1564, 13.0677 >, < 16.1529, 19.3928, 14.4106 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond C25B-C26B
        ccdc_draw_closed_cylinder( < 16.1529, 19.3928, 14.4106 >, < 15.2599, 20.2678, 15.0092 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond C26B-C27B
        ccdc_draw_closed_cylinder( < 15.2599, 20.2678, 15.0092 >, < 14.2554, 20.8303, 14.2697 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond C27B-C28B
        ccdc_draw_closed_cylinder( < 14.2554, 20.8303, 14.2697 >, < 14.1788, 20.5391, 12.9059 >, 0.1, rgb < 0.933333, 0.509804, 0.933333 > )
        // bond C30B-C31B
        ccdc_draw_closed_cylinder( < 12.4766, 20.2678, 11.1729 >, < 11.0967, 20.1993, 11.2076 >, 0.1, rgb < 1, 0, 0 > )
        // bond C31B-C32B
        ccdc_draw_closed_cylinder( < 11.0967, 20.1993, 11.2076 >, < 10.4514, 19.4243, 10.2735 >, 0.1, rgb < 1, 0, 0 > )
        // bond C32B-C33B
        ccdc_draw_closed_cylinder( < 10.4514, 19.4243, 10.2735 >, < 11.1675, 18.7458, 9.32327 >, 0.1, rgb < 1, 0, 0 > )
        // bond C33B-C34B
        ccdc_draw_closed_cylinder( < 11.1675, 18.7458, 9.32327 >, < 12.5512, 18.8332, 9.3282 >, 0.1, rgb < 1, 0, 0 > )
        // bond C36B-C37B
        ccdc_draw_closed_cylinder( < 14.3198, 18.6697, 7.63237 >, < 14.5127, 18.3052, 6.28767 >, 0.1, rgb < 1, 1, 0 > )
        // bond C37B-C38B
        ccdc_draw_closed_cylinder( < 14.5127, 18.3052, 6.28767 >, < 15.5162, 18.8846, 5.57225 >, 0.1, rgb < 1, 1, 0 > )
        // bond C38B-C39B
        ccdc_draw_closed_cylinder( < 15.5162, 18.8846, 5.57225 >, < 16.3177, 19.861, 6.14931 >, 0.1, rgb < 1, 1, 0 > )
        // bond C39B-C40B
        ccdc_draw_closed_cylinder( < 16.3177, 19.861, 6.14931 >, < 16.1051, 20.1748, 7.46873 >, 0.1, rgb < 1, 1, 0 > )
        ccdc_orient_structure( structure_orientation )
        #declare structure_orientation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        #declare structure_rotation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
    }
    // jr286_2 finished
    ccdc_orient_world( world_orientation )
}
