
////////////////////////////////////////////////////////////////////////
//
//                 Cambridge Crystallographic Data Centre
//                                CCDC 
//
////////////////////////////////////////////////////////////////////////
//
// For information on free software for rendering this file to create very
// high quality bitmap images, please visit the POV-Ray website:
// www.povray.org.
//
// This POV-Ray output mechanism has been tested with POV-Ray version 3.7.
//
// The CCDC is keen to receive feedback on, and suggestions for improvements
// to, this POV-Ray output mechanism. Please visit www.ccdc.cam.ac.uk,
// or e-mail feedback to support@ccdc.cam.ac.uk. Thank you.
//
////////////////////////////////////////////////////////////////////////
//
// If this POVRay file has been generated from an entry in the Cambridge 
// Structural Database, then it will include bibliographic, chemical, crystal, 
// experimental, refinement or atomic coordinate data resulting from the CCDC's 
// data processing and validation procedures.
//
////////////////////////////////////////////////////////////////////////

#version 3.7;
global_settings {
    assumed_gamma 2.2
    max_trace_level 5
}

// -----------------------------------------------------------

#macro rotate_view_for_animation()
    // If using the [filename.pov].ini file, with animation enabled, this will produce
    // a cyclic animation of the scene rotating, otherwise this will not affect the image:
    rotate <0,clock*360,0>
#end

#macro ccdc_perspective_camera( camera_position, field_of_view )
    camera {
        perspective
        location camera_position
        up    <0,1,0>
        right  -x * (image_width/image_height)
        look_at <0,0,-100>
        // Convert the vertical field of view to the horizontal field of view
        angle degrees(2 * atan2(tan(radians(field_of_view / 2)) * image_width, image_height))

        rotate_view_for_animation()
    }
#end

#macro ccdc_directional_light_source( light_position, light_diffuse_colour, light_specular_colour )
    // The scalar multiplier applied to light_position seems to be needed for correct brightness
    light_source {
        5 * light_position
        light_diffuse_colour
        parallel
        shadowless
        rotate_view_for_animation()
    }
    light_source {
        5 * light_position
        light_specular_colour
        parallel
        rotate_view_for_animation()
    }
#end

#macro ccdc_ambient_light_source( light_colour )
    global_settings { ambient_light light_colour * 10 }
#end

#macro ccdc_background_colour( background_colour )
    background { background_colour }
#end

#macro ccdc_background_gradient( top_right_colour, top_left_colour, bottom_left_colour, bottom_right_colour )
    // TODO - adjust to use all four colours
    background { bottom_left_colour }
#end

#macro ccdc_orient_world( world_orientation )
    transform { world_orientation }
#end

#macro ccdc_orient_structure( structure_orientation )
    transform { structure_orientation }
#end

#macro ccdc_set_standard_mercury_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.2
            roughness 0.02
        }
    }
#end

#macro ccdc_set_shiny_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.8
            roughness 0.02
        }
    }
#end

#macro ccdc_set_matt_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.0
            roughness 0.02
        }
    }
#end

#macro ccdc_set_metallic_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.2
            diffuse 0.2
            specular 1.0
            roughness 0.02 
            metallic
            reflection { 0.5, 1.0
               fresnel on
               metallic 0.8
            }
        }
    }
#end

#macro ccdc_set_iridescent_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.1
            diffuse 0.1
            reflection 0.2
            specular 1
            roughness 0.02
            irid {
              0.35
              thickness 0.5
              turbulence 0.5
            }
        }
    }
#end

#macro ccdc_set_solid_material_properties( object_color )
    // change the method call here to select different properties settings
    ccdc_set_shiny_solid_material_properties( object_color )
#end

#macro ccdc_set_wireframe_material_properties( object_color )
    no_shadow
    pigment { object_color }
#end

#macro ccdc_set_point_size( point_size )
    // TODO
#end

#macro ccdc_draw_circle( centre, circle_radius, transformation, circle_color )
    torus {
        circle_radius, 0.03
        transform { transformation }
        translate centre
        ccdc_set_wireframe_material_properties( circle_color )
    }
#end

#macro ccdc_draw_wireframe_sphere( centre, sphere_radius, sphere_color )
    union {
        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )

    }
#end

#macro ccdc_draw_solid_sphere( position, sphere_radius, sphere_color )
    sphere {
        position, sphere_radius
        ccdc_set_solid_material_properties( sphere_color )
    }
#end

#macro ccdc_draw_solid_polygon( vertices, origin, transformation, polygon_color )
    #local nvertices = dimension_size( vertices, 1 );
    polygon {
        nvertices + 1
        #local i = 0;
        #while ( i < nvertices )
            vertices[ i ]
            #local i = i + 1;
        #end
        // Repeat the first point, to close the polygon:
        vertices[ 0 ]
        translate origin
        transform { transformation inverse }
        ccdc_set_solid_material_properties( polygon_color )
    }
#end

#macro ccdc_draw_solid_torus( centre, ring_radius, torus_radius, transformation, torus_color )
    torus {
        ring_radius, torus_radius
        transform { transformation }
        translate centre
        ccdc_set_solid_material_properties( torus_color )
    }
#end

#macro ccdc_draw_disk( centre, disk_radius, disk_normal, disk_color )
    // TODO
#end

#macro ccdc_draw_line_segment( line_begin, line_end, line_color )
    cylinder {
        line_begin, line_end, 0.03
        ccdc_set_wireframe_material_properties( line_color )
    }
#end

#macro ccdc_draw_stippled_line_segment( line_begin, line_end, stipple, stipple_scale_factor, line_color )
    // TODO - adjust to use stipple and stipple_scale_factor
    /* For example:

        AACRUB delocalised bonds:
        61680 = F0F0
        
        AABHTZ contacts:
        43690 = AAAA
    */
    // The following is approximately correct when drawing contacts
    #declare nsteps = 20;
    #declare increment = 1 / nsteps;
    #declare scalar1 = 0.0;
    #while ( scalar1 < 1.0 )
        #declare point1 = ( scalar1 * line_begin ) + ( 1 - scalar1 ) * line_end;
        #declare scalar2 = scalar1 + ( increment / 2.0 );
        #declare point2 = ( scalar2 * line_begin ) + ( 1 - scalar2 ) * line_end;
        cylinder {
            point1, point2, 0.03
            ccdc_set_wireframe_material_properties( line_color )
        }
        #declare scalar1 = scalar1 + increment;
    #end
#end

#macro ccdc_draw_wireframe_arc( arc_centre, arc_end_1, arc_end_2,
                                stipple, stipple_scale_factor, line_color )
    // TODO
#end

#macro ccdc_draw_wireframe_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_colour )
    union {
        #local circle_radius = 1.0;
        #local origin = < 0, 0, 0 >;

        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )

        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >

        translate centre
    }
#end

#macro ccdc_draw_open_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        open
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_closed_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_solid_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_color )
    sphere {
        < 0, 0, 0 >, 1
        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( ellipsoid_color )
    }
#end

#macro ccdc_draw_ellipsoid_axes( centre, axis1, axis2, axis3, axes_color )
    union {
        cylinder {
            < 0.05, 0, 0 >, < -0.05, 0, 0 >, 1
        }
        cylinder {
            < 0, 0.05, 0 >, < 0, -0.05, 0 >, 1
        }
        cylinder {
            < 0, 0, 0.05 >, < 0, 0, -0.05 >, 1
        }
        #local axis_height = 0.01;
        scale < vlength( axis1 ) + axis_height, vlength( axis2 ) + axis_height, vlength( axis3 ) + axis_height >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( axes_color )
    }
#end

#macro ccdc_draw_point( point_location, point_color )
    // TODO
#end

#macro ccdc_draw_solid_triangle( corner1, corner2, corner3,
                                 color1, color2, color3,
                                 normal1, normal2, normal3 )
   smooth_triangle {
        corner1, normal1,
        corner2, normal2,
        corner3, normal3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_solid_triangle_without_normals( corner1, corner2, corner3,
                                                 color1, color2, color3 )
   triangle {
        corner1,
        corner2,
        corner3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_text( position, message, text_color, text_offset )
    // TODO - Adjust so that the text orientation is correct
    text { ttf font_name, message, 0.1, 0
        ccdc_set_wireframe_material_properties( color text_color )
        /*
        finish {
            ambient 0.2
            diffuse 0.6
            phong 0.3
            phong_size 100
        }
        */
        scale < font_scale, font_scale, font_scale >
        // rotate so text always faces the camera when animated
        rotate <0,clock*360,0>
        translate text_offset
        transform { world_rotation inverse }
        transform { structure_rotation inverse }
        translate position
    }
#end

#macro ccdc_draw_static_text( message )
    // TODO
#end

// -----------------------------------------------------------

// You can edit the file "ccdc_macro_overrides.inc" in this directory
// to override the implementations of any or all the above POVRay macros:
#include "ccdc_macro_overrides.inc"

#declare font_scale = 0.374927;

#declare font_name = "arial.ttf";

ccdc_perspective_camera( < 0, 0, 21481.7 >, 0.05 )
ccdc_directional_light_source( < 1, 1, 1 >, rgb < 0.698039, 0.698039, 0.698039 >, rgb < 1, 1, 1 > )
ccdc_directional_light_source( < -1, 0.2, 1 >, rgb < 0.501961, 0.501961, 0.501961 >, rgb < 0.501961, 0.501961, 0.501961 > )
ccdc_ambient_light_source( rgb < 0.301961, 0.301961, 0.301961 > )
ccdc_background_colour( rgb < 1, 1, 1 > )
union {
    #declare world_orientation = transform { matrix <
        -0.934016, 0.0725175, -0.349794,
        0.326854, 0.568612, -0.754882,
        0.144155, -0.819403, -0.554796,
        0, 0, 0 > };
    #declare world_rotation = transform { matrix <
        -0.934016, 0.0725175, -0.349794,
        0.326854, 0.568612, -0.754882,
        0.144155, -0.819403, -0.554796,
        0, 0, 0 > };
    union {
        // jr281
        #declare structure_orientation = transform { matrix <
            1, 0, 0,
            0, -1, 0,
            0, 0, -1,
            -4.11713, 8.11459, 4.24599 > };
        #declare structure_rotation = transform { matrix <
            1, 0, 0,
            0, -1, 0,
            0, 0, -1,
            0, 0, 0 > };
        // jr281 atoms
        // atom Cr1
        ccdc_draw_solid_sphere( < 5.20976, 9.2887, 3.35401 >, 0.5, rgb < 1, 0.647059, 0 > )
        // atom O36
        ccdc_draw_solid_sphere( < 10.5827, 7.96076, 0.701444 >, 0.1, rgb < 0.796078, 0, 0 > )
        // atom N1
        ccdc_draw_solid_sphere( < 5.1131, 11.0318, 2.25956 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N2
        ccdc_draw_solid_sphere( < 3.38574, 9.80953, 4.14292 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N3
        ccdc_draw_solid_sphere( < 5.13691, 7.59697, 4.53984 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N4
        ccdc_draw_solid_sphere( < 4.32952, 8.17867, 1.84498 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N5
        ccdc_draw_solid_sphere( < 6.99351, 8.7682, 2.51645 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom N6
        ccdc_draw_solid_sphere( < 6.28177, 10.3139, 4.78007 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // atom C1
        ccdc_draw_solid_sphere( < 6.22923, 11.6888, 1.95228 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C2
        ccdc_draw_solid_sphere( < 6.33431, 12.5356, 0.845671 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C3
        ccdc_draw_solid_sphere( < 5.28028, 12.6018, -0.0208522 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C4
        ccdc_draw_solid_sphere( < 4.07694, 11.9248, 0.272526 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C5
        ccdc_draw_solid_sphere( < 2.97885, 11.8912, -0.620352 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C6
        ccdc_draw_solid_sphere( < 1.82274, 11.2725, -0.264996 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C7
        ccdc_draw_solid_sphere( < 1.67358, 10.7633, 1.03102 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C8
        ccdc_draw_solid_sphere( < 2.72343, 10.7511, 1.94693 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C9
        ccdc_draw_solid_sphere( < 3.98225, 11.2458, 1.51077 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C10
        ccdc_draw_solid_sphere( < 2.4412, 10.4062, 3.36183 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C11
        ccdc_draw_solid_sphere( < 1.20137, 10.7666, 3.87213 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C12
        ccdc_draw_solid_sphere( < 0.903032, 10.5052, 5.18553 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C13
        ccdc_draw_solid_sphere( < 1.85077, 9.90215, 5.98052 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C14
        ccdc_draw_solid_sphere( < 3.09421, 9.55105, 5.45038 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C15
        ccdc_draw_solid_sphere( < 4.04847, 8.8293, 6.33052 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C16
        ccdc_draw_solid_sphere( < 3.99131, 9.05841, 7.69069 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C17
        ccdc_draw_solid_sphere( < 4.64156, 8.21703, 8.62179 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C18
        ccdc_draw_solid_sphere( < 5.28626, 7.10754, 8.2033 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C19
        ccdc_draw_solid_sphere( < 5.39919, 6.82576, 6.8175 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C20
        ccdc_draw_solid_sphere( < 6.02334, 5.66083, 6.33529 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C21
        ccdc_draw_solid_sphere( < 6.12727, 5.4525, 4.99583 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C22
        ccdc_draw_solid_sphere( < 5.71923, 6.46905, 4.12844 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C23
        ccdc_draw_solid_sphere( < 4.85817, 7.74816, 5.87626 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C24
        ccdc_draw_solid_sphere( < 3.18357, 7.54263, 2.07218 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C25
        ccdc_draw_solid_sphere( < 2.25304, 7.27512, 1.06143 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C26
        ccdc_draw_solid_sphere( < 2.47497, 7.78715, -0.183905 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C27
        ccdc_draw_solid_sphere( < 3.69602, 8.41887, -0.480903 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C28
        ccdc_draw_solid_sphere( < 4.66853, 8.50135, 0.551134 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C29
        ccdc_draw_solid_sphere( < 3.98206, 8.97974, -1.74637 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C30
        ccdc_draw_solid_sphere( < 5.20983, 9.50094, -1.99833 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C31
        ccdc_draw_solid_sphere( < 6.21848, 9.4521, -1.01727 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C32
        ccdc_draw_solid_sphere( < 5.98137, 8.94594, 0.248344 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C33
        ccdc_draw_solid_sphere( < 7.14034, 8.72213, 1.15628 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C34
        ccdc_draw_solid_sphere( < 8.34194, 8.41076, 0.571407 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C35
        ccdc_draw_solid_sphere( < 9.45789, 8.1705, 1.36842 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C37
        ccdc_draw_solid_sphere( < 11.763, 7.70779, 1.47558 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C38
        ccdc_draw_solid_sphere( < 9.31142, 8.19616, 2.74539 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C39
        ccdc_draw_solid_sphere( < 8.07595, 8.49511, 3.29247 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C40
        ccdc_draw_solid_sphere( < 8.00372, 8.59904, 4.77602 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C41
        ccdc_draw_solid_sphere( < 8.81951, 7.78936, 5.53886 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C42
        ccdc_draw_solid_sphere( < 9.0328, 8.02673, 6.91018 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C43
        ccdc_draw_solid_sphere( < 8.49268, 9.12797, 7.49809 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C44
        ccdc_draw_solid_sphere( < 7.61485, 9.97044, 6.77449 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C45
        ccdc_draw_solid_sphere( < 7.0534, 11.1455, 7.32288 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C46
        ccdc_draw_solid_sphere( < 6.19664, 11.8968, 6.57567 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C47
        ccdc_draw_solid_sphere( < 5.78557, 11.4278, 5.32701 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // atom C48
        ccdc_draw_solid_sphere( < 7.29087, 9.63601, 5.43387 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // jr281 bonds
        // bond Cr1-N1
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 5.16143, 10.1603, 2.80679 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 5.16143, 10.1603, 2.80679 >, < 5.1131, 11.0318, 2.25956 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond Cr1-N2
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 4.29775, 9.54912, 3.74847 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 4.29775, 9.54912, 3.74847 >, < 3.38574, 9.80953, 4.14292 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond Cr1-N3
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 5.17333, 8.44284, 3.94692 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 5.17333, 8.44284, 3.94692 >, < 5.13691, 7.59697, 4.53984 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond Cr1-N4
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 4.76964, 8.73369, 2.5995 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 4.76964, 8.73369, 2.5995 >, < 4.32952, 8.17867, 1.84498 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond Cr1-N5
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 6.10163, 9.02845, 2.93523 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 6.10163, 9.02845, 2.93523 >, < 6.99351, 8.7682, 2.51645 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond Cr1-N6
        ccdc_draw_closed_cylinder( < 5.20976, 9.2887, 3.35401 >, < 5.74576, 9.80132, 4.06704 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 5.74576, 9.80132, 4.06704 >, < 6.28177, 10.3139, 4.78007 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        // bond O36-C35
        ccdc_draw_closed_cylinder( < 10.5827, 7.96076, 0.701444 >, < 10.0203, 8.06563, 1.03493 >, 0.1, rgb < 0.796078, 0, 0 > )
        ccdc_draw_closed_cylinder( < 10.0203, 8.06563, 1.03493 >, < 9.45789, 8.1705, 1.36842 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond O36-C37
        ccdc_draw_closed_cylinder( < 10.5827, 7.96076, 0.701444 >, < 11.1729, 7.83427, 1.08851 >, 0.1, rgb < 0.796078, 0, 0 > )
        ccdc_draw_closed_cylinder( < 11.1729, 7.83427, 1.08851 >, < 11.763, 7.70779, 1.47558 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N1-C1
        ccdc_draw_closed_cylinder( < 5.1131, 11.0318, 2.25956 >, < 5.67116, 11.3603, 2.10592 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 5.67116, 11.3603, 2.10592 >, < 6.22923, 11.6888, 1.95228 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N1-C9
        ccdc_draw_closed_cylinder( < 5.1131, 11.0318, 2.25956 >, < 4.54767, 11.1388, 1.88517 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 4.54767, 11.1388, 1.88517 >, < 3.98225, 11.2458, 1.51077 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N2-C10
        ccdc_draw_closed_cylinder( < 3.38574, 9.80953, 4.14292 >, < 2.91347, 10.1079, 3.75238 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 2.91347, 10.1079, 3.75238 >, < 2.4412, 10.4062, 3.36183 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N2-C14
        ccdc_draw_closed_cylinder( < 3.38574, 9.80953, 4.14292 >, < 3.23998, 9.68029, 4.79665 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 3.23998, 9.68029, 4.79665 >, < 3.09421, 9.55105, 5.45038 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N3-C22
        ccdc_draw_closed_cylinder( < 5.13691, 7.59697, 4.53984 >, < 5.42807, 7.03301, 4.33414 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 5.42807, 7.03301, 4.33414 >, < 5.71923, 6.46905, 4.12844 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N3-C23
        ccdc_draw_closed_cylinder( < 5.13691, 7.59697, 4.53984 >, < 4.99754, 7.67256, 5.20805 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 4.99754, 7.67256, 5.20805 >, < 4.85817, 7.74816, 5.87626 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N4-C24
        ccdc_draw_closed_cylinder( < 4.32952, 8.17867, 1.84498 >, < 3.75655, 7.86065, 1.95858 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 3.75655, 7.86065, 1.95858 >, < 3.18357, 7.54263, 2.07218 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N4-C28
        ccdc_draw_closed_cylinder( < 4.32952, 8.17867, 1.84498 >, < 4.49903, 8.34001, 1.19806 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 4.49903, 8.34001, 1.19806 >, < 4.66853, 8.50135, 0.551134 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N5-C33
        ccdc_draw_closed_cylinder( < 6.99351, 8.7682, 2.51645 >, < 7.06692, 8.74517, 1.83637 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 7.06692, 8.74517, 1.83637 >, < 7.14034, 8.72213, 1.15628 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N5-C39
        ccdc_draw_closed_cylinder( < 6.99351, 8.7682, 2.51645 >, < 7.53473, 8.63165, 2.90446 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 7.53473, 8.63165, 2.90446 >, < 8.07595, 8.49511, 3.29247 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N6-C47
        ccdc_draw_closed_cylinder( < 6.28177, 10.3139, 4.78007 >, < 6.03367, 10.8709, 5.05354 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 6.03367, 10.8709, 5.05354 >, < 5.78557, 11.4278, 5.32701 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond N6-C48
        ccdc_draw_closed_cylinder( < 6.28177, 10.3139, 4.78007 >, < 6.78632, 9.97498, 5.10697 >, 0.1, rgb < 0.631373, 0.631373, 1 > )
        ccdc_draw_closed_cylinder( < 6.78632, 9.97498, 5.10697 >, < 7.29087, 9.63601, 5.43387 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C1-C2
        ccdc_draw_closed_cylinder( < 6.22923, 11.6888, 1.95228 >, < 6.33431, 12.5356, 0.845671 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C2-C3
        ccdc_draw_closed_cylinder( < 6.33431, 12.5356, 0.845671 >, < 5.28028, 12.6018, -0.0208522 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C3-C4
        ccdc_draw_closed_cylinder( < 5.28028, 12.6018, -0.0208522 >, < 4.07694, 11.9248, 0.272526 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C4-C5
        ccdc_draw_closed_cylinder( < 4.07694, 11.9248, 0.272526 >, < 2.97885, 11.8912, -0.620352 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C4-C9
        ccdc_draw_closed_cylinder( < 4.07694, 11.9248, 0.272526 >, < 3.98225, 11.2458, 1.51077 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C5-C6
        ccdc_draw_closed_cylinder( < 2.97885, 11.8912, -0.620352 >, < 1.82274, 11.2725, -0.264996 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C6-C7
        ccdc_draw_closed_cylinder( < 1.82274, 11.2725, -0.264996 >, < 1.67358, 10.7633, 1.03102 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C7-C8
        ccdc_draw_closed_cylinder( < 1.67358, 10.7633, 1.03102 >, < 2.72343, 10.7511, 1.94693 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C8-C9
        ccdc_draw_closed_cylinder( < 2.72343, 10.7511, 1.94693 >, < 3.98225, 11.2458, 1.51077 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C8-C10
        ccdc_draw_closed_cylinder( < 2.72343, 10.7511, 1.94693 >, < 2.4412, 10.4062, 3.36183 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C10-C11
        ccdc_draw_closed_cylinder( < 2.4412, 10.4062, 3.36183 >, < 1.20137, 10.7666, 3.87213 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C11-C12
        ccdc_draw_closed_cylinder( < 1.20137, 10.7666, 3.87213 >, < 0.903032, 10.5052, 5.18553 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C12-C13
        ccdc_draw_closed_cylinder( < 0.903032, 10.5052, 5.18553 >, < 1.85077, 9.90215, 5.98052 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C13-C14
        ccdc_draw_closed_cylinder( < 1.85077, 9.90215, 5.98052 >, < 3.09421, 9.55105, 5.45038 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C14-C15
        ccdc_draw_closed_cylinder( < 3.09421, 9.55105, 5.45038 >, < 4.04847, 8.8293, 6.33052 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C15-C16
        ccdc_draw_closed_cylinder( < 4.04847, 8.8293, 6.33052 >, < 3.99131, 9.05841, 7.69069 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C15-C23
        ccdc_draw_closed_cylinder( < 4.04847, 8.8293, 6.33052 >, < 4.85817, 7.74816, 5.87626 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C16-C17
        ccdc_draw_closed_cylinder( < 3.99131, 9.05841, 7.69069 >, < 4.64156, 8.21703, 8.62179 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C17-C18
        ccdc_draw_closed_cylinder( < 4.64156, 8.21703, 8.62179 >, < 5.28626, 7.10754, 8.2033 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C18-C19
        ccdc_draw_closed_cylinder( < 5.28626, 7.10754, 8.2033 >, < 5.39919, 6.82576, 6.8175 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C19-C20
        ccdc_draw_closed_cylinder( < 5.39919, 6.82576, 6.8175 >, < 6.02334, 5.66083, 6.33529 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C19-C23
        ccdc_draw_closed_cylinder( < 5.39919, 6.82576, 6.8175 >, < 4.85817, 7.74816, 5.87626 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C20-C21
        ccdc_draw_closed_cylinder( < 6.02334, 5.66083, 6.33529 >, < 6.12727, 5.4525, 4.99583 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C21-C22
        ccdc_draw_closed_cylinder( < 6.12727, 5.4525, 4.99583 >, < 5.71923, 6.46905, 4.12844 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C24-C25
        ccdc_draw_closed_cylinder( < 3.18357, 7.54263, 2.07218 >, < 2.25304, 7.27512, 1.06143 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C25-C26
        ccdc_draw_closed_cylinder( < 2.25304, 7.27512, 1.06143 >, < 2.47497, 7.78715, -0.183905 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C26-C27
        ccdc_draw_closed_cylinder( < 2.47497, 7.78715, -0.183905 >, < 3.69602, 8.41887, -0.480903 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C27-C28
        ccdc_draw_closed_cylinder( < 3.69602, 8.41887, -0.480903 >, < 4.66853, 8.50135, 0.551134 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C27-C29
        ccdc_draw_closed_cylinder( < 3.69602, 8.41887, -0.480903 >, < 3.98206, 8.97974, -1.74637 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C28-C32
        ccdc_draw_closed_cylinder( < 4.66853, 8.50135, 0.551134 >, < 5.98137, 8.94594, 0.248344 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C29-C30
        ccdc_draw_closed_cylinder( < 3.98206, 8.97974, -1.74637 >, < 5.20983, 9.50094, -1.99833 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C30-C31
        ccdc_draw_closed_cylinder( < 5.20983, 9.50094, -1.99833 >, < 6.21848, 9.4521, -1.01727 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C31-C32
        ccdc_draw_closed_cylinder( < 6.21848, 9.4521, -1.01727 >, < 5.98137, 8.94594, 0.248344 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C32-C33
        ccdc_draw_closed_cylinder( < 5.98137, 8.94594, 0.248344 >, < 7.14034, 8.72213, 1.15628 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C33-C34
        ccdc_draw_closed_cylinder( < 7.14034, 8.72213, 1.15628 >, < 8.34194, 8.41076, 0.571407 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C34-C35
        ccdc_draw_closed_cylinder( < 8.34194, 8.41076, 0.571407 >, < 9.45789, 8.1705, 1.36842 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C35-C38
        ccdc_draw_closed_cylinder( < 9.45789, 8.1705, 1.36842 >, < 9.31142, 8.19616, 2.74539 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C38-C39
        ccdc_draw_closed_cylinder( < 9.31142, 8.19616, 2.74539 >, < 8.07595, 8.49511, 3.29247 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C39-C40
        ccdc_draw_closed_cylinder( < 8.07595, 8.49511, 3.29247 >, < 8.00372, 8.59904, 4.77602 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C40-C41
        ccdc_draw_closed_cylinder( < 8.00372, 8.59904, 4.77602 >, < 8.81951, 7.78936, 5.53886 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C40-C48
        ccdc_draw_closed_cylinder( < 8.00372, 8.59904, 4.77602 >, < 7.29087, 9.63601, 5.43387 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C41-C42
        ccdc_draw_closed_cylinder( < 8.81951, 7.78936, 5.53886 >, < 9.0328, 8.02673, 6.91018 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C42-C43
        ccdc_draw_closed_cylinder( < 9.0328, 8.02673, 6.91018 >, < 8.49268, 9.12797, 7.49809 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C43-C44
        ccdc_draw_closed_cylinder( < 8.49268, 9.12797, 7.49809 >, < 7.61485, 9.97044, 6.77449 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C44-C45
        ccdc_draw_closed_cylinder( < 7.61485, 9.97044, 6.77449 >, < 7.0534, 11.1455, 7.32288 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C44-C48
        ccdc_draw_closed_cylinder( < 7.61485, 9.97044, 6.77449 >, < 7.29087, 9.63601, 5.43387 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C45-C46
        ccdc_draw_closed_cylinder( < 7.0534, 11.1455, 7.32288 >, < 6.19664, 11.8968, 6.57567 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        // bond C46-C47
        ccdc_draw_closed_cylinder( < 6.19664, 11.8968, 6.57567 >, < 5.78557, 11.4278, 5.32701 >, 0.1, rgb < 0.466667, 0.466667, 0.466667 > )
        ccdc_draw_solid_sphere( < 2.8763, 11.3081, 0.645983 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 5.16935, 11.8381, 1.13666 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 4.95938, 8.96649, -0.740566 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 3.43428, 7.95063, 0.810821 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 5.54402, 6.62554, 5.44886 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 4.70416, 7.96437, 7.25668 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 6.70385, 10.7317, 6.03567 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_draw_solid_sphere( < 8.20907, 8.85826, 6.15525 >, 0.25, rgbt < 1, 0, 0, 0.5 > )
        ccdc_orient_structure( structure_orientation )
        #declare structure_orientation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        #declare structure_rotation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
    }
    // jr281 finished
    ccdc_draw_stippled_line_segment( < -0.682857, 0.163954, 3.43517 >, < -1.24084, -3.19352, 3.6 >, 43690, 1, rgb < 0, 1, 0 > )
    ccdc_draw_text( < -0.961846, -1.51479, 3.51758 >, "3.408", rgb < 0, 1, 0 >, < 0, 0, 2 > )
    ccdc_draw_stippled_line_segment( < 0.842249, -0.851905, 4.98655 >, < 1.05222, -3.72351, 3.10933 >, 43690, 1, rgb < 0, 1, 0 > )
    ccdc_draw_text( < 0.947234, -2.28771, 4.04794 >, "3.437", rgb < 0, 1, 0 >, < 0, 0, 2 > )
    ccdc_draw_stippled_line_segment( < 4.09194, -0.743672, -1.90926 >, < 1.42689, 1.48904, -1.20287 >, 43690, 1, rgb < 0, 1, 0 > )
    ccdc_draw_text( < 2.75941, 0.372685, -1.55607 >, "3.548", rgb < 0, 1, 0 >, < 0, 0, 2 > )
    ccdc_draw_stippled_line_segment( < 2.58672, -2.61716, -1.78968 >, < 0.587025, 0.150221, -3.01069 >, 43690, 1, rgb < 0, 1, 0 > )
    ccdc_draw_text( < 1.58687, -1.23347, -2.40018 >, "3.626", rgb < 0, 1, 0 >, < 0, 0, 2 > )
    ccdc_draw_stippled_line_segment( < 0.842249, -0.851905, 4.98655 >, < -0.682857, 0.163954, 3.43517 >, 43690, 1, rgb < 0, 1, 0 > )
    ccdc_draw_text( < 0.0796961, -0.343975, 4.21086 >, "2.401", rgb < 0, 1, 0 >, < 0, 0, 2 > )
    ccdc_orient_world( world_orientation )
}
