
////////////////////////////////////////////////////////////////////////
//
//                 Cambridge Crystallographic Data Centre
//                                CCDC 
//
////////////////////////////////////////////////////////////////////////
//
// For information on free software for rendering this file to create very
// high quality bitmap images, please visit the POV-Ray website:
// www.povray.org.
//
// This POV-Ray output mechanism has been tested with POV-Ray version 3.7.
//
// The CCDC is keen to receive feedback on, and suggestions for improvements
// to, this POV-Ray output mechanism. Please visit www.ccdc.cam.ac.uk,
// or e-mail feedback to support@ccdc.cam.ac.uk. Thank you.
//
////////////////////////////////////////////////////////////////////////
//
// If this POVRay file has been generated from an entry in the Cambridge 
// Structural Database, then it will include bibliographic, chemical, crystal, 
// experimental, refinement or atomic coordinate data resulting from the CCDC's 
// data processing and validation procedures.
//
////////////////////////////////////////////////////////////////////////

#version 3.7;
global_settings {
    assumed_gamma 2.2
    max_trace_level 5
}

// -----------------------------------------------------------

#macro rotate_view_for_animation()
    // If using the [filename.pov].ini file, with animation enabled, this will produce
    // a cyclic animation of the scene rotating, otherwise this will not affect the image:
    rotate <0,clock*360,0>
#end

#macro ccdc_perspective_camera( camera_position, field_of_view )
    camera {
        perspective
        location camera_position
        up    <0,1,0>
        right  -x * (image_width/image_height)
        look_at <0,0,-100>
        // Convert the vertical field of view to the horizontal field of view
        angle degrees(2 * atan2(tan(radians(field_of_view / 2)) * image_width, image_height))

        rotate_view_for_animation()
    }
#end

#macro ccdc_directional_light_source( light_position, light_diffuse_colour, light_specular_colour )
    // The scalar multiplier applied to light_position seems to be needed for correct brightness
    light_source {
        5 * light_position
        light_diffuse_colour
        parallel
        shadowless
        rotate_view_for_animation()
    }
    light_source {
        5 * light_position
        light_specular_colour
        parallel
        rotate_view_for_animation()
    }
#end

#macro ccdc_ambient_light_source( light_colour )
    global_settings { ambient_light light_colour * 10 }
#end

#macro ccdc_background_colour( background_colour )
    background { background_colour }
#end

#macro ccdc_background_gradient( top_right_colour, top_left_colour, bottom_left_colour, bottom_right_colour )
    // TODO - adjust to use all four colours
    background { bottom_left_colour }
#end

#macro ccdc_orient_world( world_orientation )
    transform { world_orientation }
#end

#macro ccdc_orient_structure( structure_orientation )
    transform { structure_orientation }
#end

#macro ccdc_set_standard_mercury_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.2
            roughness 0.02
        }
    }
#end

#macro ccdc_set_shiny_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.8
            roughness 0.02
        }
    }
#end

#macro ccdc_set_matt_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            specular 0.0
            roughness 0.02
        }
    }
#end

#macro ccdc_set_metallic_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.2
            diffuse 0.2
            specular 1.0
            roughness 0.02 
            metallic
            reflection { 0.5, 1.0
               fresnel on
               metallic 0.8
            }
        }
    }
#end

#macro ccdc_set_iridescent_solid_material_properties( object_color )
    no_shadow
    texture {
        pigment { object_color }
        finish {
            ambient 0.1
            diffuse 0.1
            reflection 0.2
            specular 1
            roughness 0.02
            irid {
              0.35
              thickness 0.5
              turbulence 0.5
            }
        }
    }
#end

#macro ccdc_set_solid_material_properties( object_color )
    // change the method call here to select different properties settings
    ccdc_set_shiny_solid_material_properties( object_color )
#end

#macro ccdc_set_wireframe_material_properties( object_color )
    no_shadow
    pigment { object_color }
#end

#macro ccdc_set_point_size( point_size )
    // TODO
#end

#macro ccdc_draw_circle( centre, circle_radius, transformation, circle_color )
    torus {
        circle_radius, 0.03
        transform { transformation }
        translate centre
        ccdc_set_wireframe_material_properties( circle_color )
    }
#end

#macro ccdc_draw_wireframe_sphere( centre, sphere_radius, sphere_color )
    union {
        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( centre, sphere_radius, transformation, sphere_color )

    }
#end

#macro ccdc_draw_solid_sphere( position, sphere_radius, sphere_color )
    sphere {
        position, sphere_radius
        ccdc_set_solid_material_properties( sphere_color )
    }
#end

#macro ccdc_draw_solid_polygon( vertices, origin, transformation, polygon_color )
    #local nvertices = dimension_size( vertices, 1 );
    polygon {
        nvertices + 1
        #local i = 0;
        #while ( i < nvertices )
            vertices[ i ]
            #local i = i + 1;
        #end
        // Repeat the first point, to close the polygon:
        vertices[ 0 ]
        translate origin
        transform { transformation inverse }
        ccdc_set_solid_material_properties( polygon_color )
    }
#end

#macro ccdc_draw_solid_torus( centre, ring_radius, torus_radius, transformation, torus_color )
    torus {
        ring_radius, torus_radius
        transform { transformation }
        translate centre
        ccdc_set_solid_material_properties( torus_color )
    }
#end

#macro ccdc_draw_disk( centre, disk_radius, disk_normal, disk_color )
    // TODO
#end

#macro ccdc_draw_line_segment( line_begin, line_end, line_color )
    cylinder {
        line_begin, line_end, 0.03
        ccdc_set_wireframe_material_properties( line_color )
    }
#end

#macro ccdc_draw_stippled_line_segment( line_begin, line_end, stipple, stipple_scale_factor, line_color )
    // TODO - adjust to use stipple and stipple_scale_factor
    /* For example:

        AACRUB delocalised bonds:
        61680 = F0F0
        
        AABHTZ contacts:
        43690 = AAAA
    */
    // The following is approximately correct when drawing contacts
    #declare nsteps = 20;
    #declare increment = 1 / nsteps;
    #declare scalar1 = 0.0;
    #while ( scalar1 < 1.0 )
        #declare point1 = ( scalar1 * line_begin ) + ( 1 - scalar1 ) * line_end;
        #declare scalar2 = scalar1 + ( increment / 2.0 );
        #declare point2 = ( scalar2 * line_begin ) + ( 1 - scalar2 ) * line_end;
        cylinder {
            point1, point2, 0.03
            ccdc_set_wireframe_material_properties( line_color )
        }
        #declare scalar1 = scalar1 + increment;
    #end
#end

#macro ccdc_draw_wireframe_arc( arc_centre, arc_end_1, arc_end_2,
                                stipple, stipple_scale_factor, line_color )
    // TODO
#end

#macro ccdc_draw_wireframe_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_colour )
    union {
        #local circle_radius = 1.0;
        #local origin = < 0, 0, 0 >;

        #local transformation = transform { matrix <
            0, -1, 0,
            1, 0, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0, 1,
            0, -1, 0,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, 0.707107, 0.707107,
            0, -0.707107, 0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, -0.707107, -0.5,
            0.707107, 0, 0.707107,
            -0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, -0.707107, 0,
            0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            1, 0, 0,
            0, -0.707107, 0.707107,
            0, -0.707107, -0.707107,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.5, 0.707107, 0.5,
            -0.707107, 0, 0.707107,
            0.5, -0.707107, 0.5,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )
        #local transformation = transform { matrix <
            0.707107, 0.707107, 0,
            -0.707107, 0.707107, 0,
            0, 0, 1,
            0, 0, 0 > };
        ccdc_draw_circle( origin, circle_radius, transformation, ellipsoid_colour )

        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >

        translate centre
    }
#end

#macro ccdc_draw_open_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        open
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_closed_cylinder( centre_line_begin, centre_line_end, cylinder_radius, cylinder_color )
    cylinder {
        centre_line_begin, centre_line_end, cylinder_radius
        ccdc_set_solid_material_properties( cylinder_color )
    }
#end

#macro ccdc_draw_solid_ellipsoid( centre, axis1, axis2, axis3, ellipsoid_color )
    sphere {
        < 0, 0, 0 >, 1
        scale < vlength( axis1 ), vlength( axis2 ), vlength( axis3 ) >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( ellipsoid_color )
    }
#end

#macro ccdc_draw_ellipsoid_axes( centre, axis1, axis2, axis3, axes_color )
    union {
        cylinder {
            < 0.05, 0, 0 >, < -0.05, 0, 0 >, 1
        }
        cylinder {
            < 0, 0.05, 0 >, < 0, -0.05, 0 >, 1
        }
        cylinder {
            < 0, 0, 0.05 >, < 0, 0, -0.05 >, 1
        }
        #local axis_height = 0.01;
        scale < vlength( axis1 ) + axis_height, vlength( axis2 ) + axis_height, vlength( axis3 ) + axis_height >
        #declare axis1_norm = vnormalize( axis1 );
        #declare axis2_norm = vnormalize( axis2 );
        #declare axis3_norm = vnormalize( axis3 );
        matrix < axis1_norm.x,  axis1_norm.y,  axis1_norm.z,
                 axis2_norm.x,  axis2_norm.y,  axis2_norm.z,
                 axis3_norm.x,  axis3_norm.y,  axis3_norm.z,
                 0, 0, 0 >
        translate centre
        ccdc_set_solid_material_properties( axes_color )
    }
#end

#macro ccdc_draw_point( point_location, point_color )
    // TODO
#end

#macro ccdc_draw_solid_triangle( corner1, corner2, corner3,
                                 color1, color2, color3,
                                 normal1, normal2, normal3 )
   smooth_triangle {
        corner1, normal1,
        corner2, normal2,
        corner3, normal3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_solid_triangle_without_normals( corner1, corner2, corner3,
                                                 color1, color2, color3 )
   triangle {
        corner1,
        corner2,
        corner3
        ccdc_set_solid_material_properties( color1 )
        // TODO POVRay does not appear to support different colours for 
        //      different vertices in a smooth_triangle
        //      See section 3.5.8 of the POVRay help.
   }
#end

#macro ccdc_draw_text( position, message, text_color, text_offset )
    // TODO - Adjust so that the text orientation is correct
    text { ttf font_name, message, 0.1, 0
        ccdc_set_wireframe_material_properties( color text_color )
        /*
        finish {
            ambient 0.2
            diffuse 0.6
            phong 0.3
            phong_size 100
        }
        */
        scale < font_scale, font_scale, font_scale >
        // rotate so text always faces the camera when animated
        rotate <0,clock*360,0>
        translate text_offset
        transform { world_rotation inverse }
        transform { structure_rotation inverse }
        translate position
    }
#end

#macro ccdc_draw_static_text( message )
    // TODO
#end

// -----------------------------------------------------------

// You can edit the file "ccdc_macro_overrides.inc" in this directory
// to override the implementations of any or all the above POVRay macros:
#include "ccdc_macro_overrides.inc"

#declare font_scale = 0.265657;

#declare font_name = "arial.ttf";

ccdc_perspective_camera( < 0, 0, 15221 >, 0.05 )
ccdc_directional_light_source( < 1, 1, 1 >, rgb < 0.701961, 0.701961, 0.701961 >, rgb < 1, 1, 1 > )
ccdc_directional_light_source( < -1, 0.2, 1 >, rgb < 0.501961, 0.501961, 0.501961 >, rgb < 0.501961, 0.501961, 0.501961 > )
ccdc_ambient_light_source( rgb < 0.301961, 0.301961, 0.301961 > )
ccdc_background_colour( rgb < 1, 1, 1 > )
union {
    #declare world_orientation = transform { matrix <
        -0.386174, -0.899724, -0.203387,
        -0.831118, 0.435031, -0.346396,
        0.400141, 0.0352693, -0.915775,
        0, -0.300749, 0 > };
    #declare world_rotation = transform { matrix <
        -0.386174, -0.899724, -0.203387,
        -0.831118, 0.435031, -0.346396,
        0.400141, 0.0352693, -0.915775,
        0, 0, 0 > };
    union {
        // jr272
        #declare structure_orientation = transform { matrix <
            0, -1, 0,
            0, 0, -1,
            1, 0, 0,
            -18.546, 6.87465, 5.17828 > };
        #declare structure_rotation = transform { matrix <
            0, -1, 0,
            0, 0, -1,
            1, 0, 0,
            0, 0, 0 > };
        // jr272 atoms
        // atom Cr1
        ccdc_draw_solid_sphere( < 6.47008, 3.99384, 17.7934 >, 0.7, rgb < 1, 0.647059, 0 > )
        // atom N2
        ccdc_draw_solid_sphere( < 5.56821, 3.85137, 15.9836 >, 0.1, rgb < 1, 0, 1 > )
        // atom N1
        ccdc_draw_solid_sphere( < 7.38033, 2.18166, 17.4251 >, 0.1, rgb < 1, 0, 0 > )
        // atom N5
        ccdc_draw_solid_sphere( < 7.48517, 4.19135, 19.4931 >, 0.1, rgb < 0, 0, 1 > )
        // atom N6
        ccdc_draw_solid_sphere( < 8.11526, 5.08377, 17.1737 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom N4
        ccdc_draw_solid_sphere( < 5.19484, 3.06436, 19.1534 >, 0.1, rgb < 0, 0, 0 > )
        // atom N3
        ccdc_draw_solid_sphere( < 5.43532, 5.74377, 18.1956 >, 0.1, rgb < 1, 1, 0 > )
        // atom C5
        ccdc_draw_solid_sphere( < 7.99742, 1.98505, 16.2128 >, 0.1, rgb < 1, 0, 0 > )
        // atom C9
        ccdc_draw_solid_sphere( < 7.56607, 2.68228, 15.0446 >, 0.1, rgb < 1, 0, 0 > )
        // atom C34
        ccdc_draw_solid_sphere( < 9.0799, 5.2471, 18.1236 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom C4
        ccdc_draw_solid_sphere( < 9.09375, 1.07986, 16.1207 >, 0.1, rgb < 1, 0, 0 > )
        // atom C10
        ccdc_draw_solid_sphere( < 6.24038, 3.35632, 14.9029 >, 0.1, rgb < 1, 0, 1 > )
        // atom C1
        ccdc_draw_solid_sphere( < 7.68682, 1.37307, 18.4404 >, 0.1, rgb < 1, 0, 0 > )
        // atom C33
        ccdc_draw_solid_sphere( < 8.72654, 4.70619, 19.45 >, 0.1, rgb < 0, 0, 1 > )
        // atom C8
        ccdc_draw_solid_sphere( < 8.37849, 2.60349, 13.9222 >, 0.1, rgb < 1, 0, 0 > )
        // atom C20
        ccdc_draw_solid_sphere( < 4.0708, 5.68135, 18.0122 >, 0.1, rgb < 1, 1, 0 > )
        // atom C24
        ccdc_draw_solid_sphere( < 4.03694, 2.44717, 18.8677 >, 0.1, rgb < 0, 0, 0 > )
        // atom C6
        ccdc_draw_solid_sphere( < 9.85743, 1.01078, 14.9349 >, 0.1, rgb < 1, 0, 0 > )
        // atom C14
        ccdc_draw_solid_sphere( < 4.3395, 4.41225, 15.8082 >, 0.1, rgb < 1, 0, 1 > )
        // atom C2
        ccdc_draw_solid_sphere( < 8.65744, 0.365889, 18.3713 >, 0.1, rgb < 1, 0, 0 > )
        // atom C15
        ccdc_draw_solid_sphere( < 3.53803, 4.89237, 16.9623 >, 0.1, rgb < 1, 1, 0 > )
        // atom C12
        ccdc_draw_solid_sphere( < 4.49052, 4.1068, 13.4407 >, 0.1, rgb < 1, 0, 1 > )
        // atom C11
        ccdc_draw_solid_sphere( < 5.68845, 3.4522, 13.6296 >, 0.1, rgb < 1, 0, 1 > )
        // atom C28
        ccdc_draw_solid_sphere( < 5.55778, 3.23633, 20.4569 >, 0.1, rgb < 0, 0, 0 > )
        // atom C29
        ccdc_draw_solid_sphere( < 6.92264, 3.77437, 20.6435 >, 0.1, rgb < 0, 0, 1 > )
        // atom C37
        ccdc_draw_solid_sphere( < 9.44814, 6.35862, 15.6553 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom C38
        ccdc_draw_solid_sphere( < 8.30101, 5.65203, 15.966 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom C7
        ccdc_draw_solid_sphere( < 9.53109, 1.80048, 13.8718 >, 0.1, rgb < 1, 0, 0 > )
        // atom C13
        ccdc_draw_solid_sphere( < 3.80536, 4.58314, 14.5389 >, 0.1, rgb < 1, 0, 1 > )
        // atom C3
        ccdc_draw_solid_sphere( < 9.39255, 0.263713, 17.229 >, 0.1, rgb < 1, 0, 0 > )
        // atom C16
        ccdc_draw_solid_sphere( < 2.16924, 4.67668, 16.9393 >, 0.1, rgb < 1, 1, 0 > )
        // atom C17
        ccdc_draw_solid_sphere( < 1.31064, 5.27625, 17.8795 >, 0.1, rgb < 1, 1, 0 > )
        // atom C23
        ccdc_draw_solid_sphere( < 5.92191, 6.63277, 19.0606 >, 0.1, rgb < 1, 1, 0 > )
        // atom C35
        ccdc_draw_solid_sphere( < 10.2464, 5.94254, 17.8677 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom C36
        ccdc_draw_solid_sphere( < 10.4309, 6.50109, 16.6133 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // atom C19
        ccdc_draw_solid_sphere( < 3.20212, 6.3883, 18.8873 >, 0.1, rgb < 1, 1, 0 > )
        // atom C18
        ccdc_draw_solid_sphere( < 1.80271, 6.15427, 18.7911 >, 0.1, rgb < 1, 1, 0 > )
        // atom C25
        ccdc_draw_solid_sphere( < 3.1662, 2.03703, 19.8678 >, 0.1, rgb < 0, 0, 0 > )
        // atom C32
        ccdc_draw_solid_sphere( < 9.50253, 4.75278, 20.5963 >, 0.1, rgb < 0, 0, 1 > )
        // atom C30
        ccdc_draw_solid_sphere( < 7.64577, 3.81953, 21.8238 >, 0.1, rgb < 0, 0, 1 > )
        // atom C27
        ccdc_draw_solid_sphere( < 4.7091, 2.87494, 21.4921 >, 0.1, rgb < 0, 0, 0 > )
        // atom C31
        ccdc_draw_solid_sphere( < 8.95197, 4.28741, 21.7757 >, 0.1, rgb < 0, 0, 1 > )
        // atom C26
        ccdc_draw_solid_sphere( < 3.49544, 2.28006, 21.188 >, 0.1, rgb < 0, 0, 0 > )
        // atom C22
        ccdc_draw_solid_sphere( < 5.13173, 7.43164, 19.8873 >, 0.1, rgb < 1, 1, 0 > )
        // atom C21
        ccdc_draw_solid_sphere( < 3.77115, 7.27352, 19.8306 >, 0.1, rgb < 1, 1, 0 > )
        // jr272 bonds
        // bond Cr1-N2
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 6.01915, 3.9226, 16.8885 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 6.01915, 3.9226, 16.8885 >, < 5.56821, 3.85137, 15.9836 >, 0.1, rgb < 1, 0, 1 > )
        // bond Cr1-N1
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 6.9252, 3.08775, 17.6092 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 6.9252, 3.08775, 17.6092 >, < 7.38033, 2.18166, 17.4251 >, 0.1, rgb < 1, 0, 0 > )
        // bond Cr1-N5
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 6.97763, 4.09259, 18.6433 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 6.97763, 4.09259, 18.6433 >, < 7.48517, 4.19135, 19.4931 >, 0.1, rgb < 0, 0, 1 > )
        // bond Cr1-N6
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 7.29267, 4.5388, 17.4835 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 7.29267, 4.5388, 17.4835 >, < 8.11526, 5.08377, 17.1737 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond Cr1-N4
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 5.83246, 3.5291, 18.4734 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 5.83246, 3.5291, 18.4734 >, < 5.19484, 3.06436, 19.1534 >, 0.1, rgb < 0, 0, 0 > )
        // bond Cr1-N3
        ccdc_draw_closed_cylinder( < 6.47008, 3.99384, 17.7934 >, < 5.9527, 4.8688, 17.9945 >, 0.1, rgb < 1, 0.647059, 0 > )
        ccdc_draw_closed_cylinder( < 5.9527, 4.8688, 17.9945 >, < 5.43532, 5.74377, 18.1956 >, 0.1, rgb < 1, 1, 0 > )
        // bond N2-C10
        ccdc_draw_closed_cylinder( < 5.56821, 3.85137, 15.9836 >, < 6.24038, 3.35632, 14.9029 >, 0.1, rgb < 1, 0, 1 > )
        // bond N2-C14
        ccdc_draw_closed_cylinder( < 5.56821, 3.85137, 15.9836 >, < 4.3395, 4.41225, 15.8082 >, 0.1, rgb < 1, 0, 1 > )
        // bond N1-C5
        ccdc_draw_closed_cylinder( < 7.38033, 2.18166, 17.4251 >, < 7.99742, 1.98505, 16.2128 >, 0.1, rgb < 1, 0, 0 > )
        // bond N1-C1
        ccdc_draw_closed_cylinder( < 7.38033, 2.18166, 17.4251 >, < 7.68682, 1.37307, 18.4404 >, 0.1, rgb < 1, 0, 0 > )
        // bond N5-C33
        ccdc_draw_closed_cylinder( < 7.48517, 4.19135, 19.4931 >, < 8.72654, 4.70619, 19.45 >, 0.1, rgb < 0, 0, 1 > )
        // bond N5-C29
        ccdc_draw_closed_cylinder( < 7.48517, 4.19135, 19.4931 >, < 6.92264, 3.77437, 20.6435 >, 0.1, rgb < 0, 0, 1 > )
        // bond N6-C34
        ccdc_draw_closed_cylinder( < 8.11526, 5.08377, 17.1737 >, < 9.0799, 5.2471, 18.1236 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond N6-C38
        ccdc_draw_closed_cylinder( < 8.11526, 5.08377, 17.1737 >, < 8.30101, 5.65203, 15.966 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond N4-C24
        ccdc_draw_closed_cylinder( < 5.19484, 3.06436, 19.1534 >, < 4.03694, 2.44717, 18.8677 >, 0.1, rgb < 0, 0, 0 > )
        // bond N4-C28
        ccdc_draw_closed_cylinder( < 5.19484, 3.06436, 19.1534 >, < 5.55778, 3.23633, 20.4569 >, 0.1, rgb < 0, 0, 0 > )
        // bond N3-C20
        ccdc_draw_closed_cylinder( < 5.43532, 5.74377, 18.1956 >, < 4.0708, 5.68135, 18.0122 >, 0.1, rgb < 1, 1, 0 > )
        // bond N3-C23
        ccdc_draw_closed_cylinder( < 5.43532, 5.74377, 18.1956 >, < 5.92191, 6.63277, 19.0606 >, 0.1, rgb < 1, 1, 0 > )
        // bond C5-C9
        ccdc_draw_closed_cylinder( < 7.99742, 1.98505, 16.2128 >, < 7.56607, 2.68228, 15.0446 >, 0.1, rgb < 1, 0, 0 > )
        // bond C5-C4
        ccdc_draw_closed_cylinder( < 7.99742, 1.98505, 16.2128 >, < 9.09375, 1.07986, 16.1207 >, 0.1, rgb < 1, 0, 0 > )
        // bond C9-C10
        ccdc_draw_closed_cylinder( < 7.56607, 2.68228, 15.0446 >, < 6.90323, 3.0193, 14.9738 >, 0.1, rgb < 1, 0, 0 > )
        ccdc_draw_closed_cylinder( < 6.90323, 3.0193, 14.9738 >, < 6.24038, 3.35632, 14.9029 >, 0.1, rgb < 1, 0, 1 > )
        // bond C9-C8
        ccdc_draw_closed_cylinder( < 7.56607, 2.68228, 15.0446 >, < 8.37849, 2.60349, 13.9222 >, 0.1, rgb < 1, 0, 0 > )
        // bond C34-C33
        ccdc_draw_closed_cylinder( < 9.0799, 5.2471, 18.1236 >, < 8.90322, 4.97664, 18.7868 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        ccdc_draw_closed_cylinder( < 8.90322, 4.97664, 18.7868 >, < 8.72654, 4.70619, 19.45 >, 0.1, rgb < 0, 0, 1 > )
        // bond C34-C35
        ccdc_draw_closed_cylinder( < 9.0799, 5.2471, 18.1236 >, < 10.2464, 5.94254, 17.8677 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond C4-C6
        ccdc_draw_closed_cylinder( < 9.09375, 1.07986, 16.1207 >, < 9.85743, 1.01078, 14.9349 >, 0.1, rgb < 1, 0, 0 > )
        // bond C4-C3
        ccdc_draw_closed_cylinder( < 9.09375, 1.07986, 16.1207 >, < 9.39255, 0.263713, 17.229 >, 0.1, rgb < 1, 0, 0 > )
        // bond C10-C11
        ccdc_draw_closed_cylinder( < 6.24038, 3.35632, 14.9029 >, < 5.68845, 3.4522, 13.6296 >, 0.1, rgb < 1, 0, 1 > )
        // bond C1-C2
        ccdc_draw_closed_cylinder( < 7.68682, 1.37307, 18.4404 >, < 8.65744, 0.365889, 18.3713 >, 0.1, rgb < 1, 0, 0 > )
        // bond C33-C32
        ccdc_draw_closed_cylinder( < 8.72654, 4.70619, 19.45 >, < 9.50253, 4.75278, 20.5963 >, 0.1, rgb < 0, 0, 1 > )
        // bond C8-C7
        ccdc_draw_closed_cylinder( < 8.37849, 2.60349, 13.9222 >, < 9.53109, 1.80048, 13.8718 >, 0.1, rgb < 1, 0, 0 > )
        // bond C20-C15
        ccdc_draw_closed_cylinder( < 4.0708, 5.68135, 18.0122 >, < 3.53803, 4.89237, 16.9623 >, 0.1, rgb < 1, 1, 0 > )
        // bond C20-C19
        ccdc_draw_closed_cylinder( < 4.0708, 5.68135, 18.0122 >, < 3.20212, 6.3883, 18.8873 >, 0.1, rgb < 1, 1, 0 > )
        // bond C24-C25
        ccdc_draw_closed_cylinder( < 4.03694, 2.44717, 18.8677 >, < 3.1662, 2.03703, 19.8678 >, 0.1, rgb < 0, 0, 0 > )
        // bond C6-C7
        ccdc_draw_closed_cylinder( < 9.85743, 1.01078, 14.9349 >, < 9.53109, 1.80048, 13.8718 >, 0.1, rgb < 1, 0, 0 > )
        // bond C14-C15
        ccdc_draw_closed_cylinder( < 4.3395, 4.41225, 15.8082 >, < 3.93876, 4.65231, 16.3853 >, 0.1, rgb < 1, 0, 1 > )
        ccdc_draw_closed_cylinder( < 3.93876, 4.65231, 16.3853 >, < 3.53803, 4.89237, 16.9623 >, 0.1, rgb < 1, 1, 0 > )
        // bond C14-C13
        ccdc_draw_closed_cylinder( < 4.3395, 4.41225, 15.8082 >, < 3.80536, 4.58314, 14.5389 >, 0.1, rgb < 1, 0, 1 > )
        // bond C2-C3
        ccdc_draw_closed_cylinder( < 8.65744, 0.365889, 18.3713 >, < 9.39255, 0.263713, 17.229 >, 0.1, rgb < 1, 0, 0 > )
        // bond C15-C16
        ccdc_draw_closed_cylinder( < 3.53803, 4.89237, 16.9623 >, < 2.16924, 4.67668, 16.9393 >, 0.1, rgb < 1, 1, 0 > )
        // bond C12-C11
        ccdc_draw_closed_cylinder( < 4.49052, 4.1068, 13.4407 >, < 5.68845, 3.4522, 13.6296 >, 0.1, rgb < 1, 0, 1 > )
        // bond C12-C13
        ccdc_draw_closed_cylinder( < 4.49052, 4.1068, 13.4407 >, < 3.80536, 4.58314, 14.5389 >, 0.1, rgb < 1, 0, 1 > )
        // bond C28-C29
        ccdc_draw_closed_cylinder( < 5.55778, 3.23633, 20.4569 >, < 6.24021, 3.50535, 20.5502 >, 0.1, rgb < 0, 0, 0 > )
        ccdc_draw_closed_cylinder( < 6.24021, 3.50535, 20.5502 >, < 6.92264, 3.77437, 20.6435 >, 0.1, rgb < 0, 0, 1 > )
        // bond C28-C27
        ccdc_draw_closed_cylinder( < 5.55778, 3.23633, 20.4569 >, < 4.7091, 2.87494, 21.4921 >, 0.1, rgb < 0, 0, 0 > )
        // bond C29-C30
        ccdc_draw_closed_cylinder( < 6.92264, 3.77437, 20.6435 >, < 7.64577, 3.81953, 21.8238 >, 0.1, rgb < 0, 0, 1 > )
        // bond C37-C38
        ccdc_draw_closed_cylinder( < 9.44814, 6.35862, 15.6553 >, < 8.30101, 5.65203, 15.966 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond C37-C36
        ccdc_draw_closed_cylinder( < 9.44814, 6.35862, 15.6553 >, < 10.4309, 6.50109, 16.6133 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond C16-C17
        ccdc_draw_closed_cylinder( < 2.16924, 4.67668, 16.9393 >, < 1.31064, 5.27625, 17.8795 >, 0.1, rgb < 1, 1, 0 > )
        // bond C17-C18
        ccdc_draw_closed_cylinder( < 1.31064, 5.27625, 17.8795 >, < 1.80271, 6.15427, 18.7911 >, 0.1, rgb < 1, 1, 0 > )
        // bond C23-C22
        ccdc_draw_closed_cylinder( < 5.92191, 6.63277, 19.0606 >, < 5.13173, 7.43164, 19.8873 >, 0.1, rgb < 1, 1, 0 > )
        // bond C35-C36
        ccdc_draw_closed_cylinder( < 10.2464, 5.94254, 17.8677 >, < 10.4309, 6.50109, 16.6133 >, 0.1, rgb < 0.564706, 0.933333, 0.564706 > )
        // bond C19-C18
        ccdc_draw_closed_cylinder( < 3.20212, 6.3883, 18.8873 >, < 1.80271, 6.15427, 18.7911 >, 0.1, rgb < 1, 1, 0 > )
        // bond C19-C21
        ccdc_draw_closed_cylinder( < 3.20212, 6.3883, 18.8873 >, < 3.77115, 7.27352, 19.8306 >, 0.1, rgb < 1, 1, 0 > )
        // bond C25-C26
        ccdc_draw_closed_cylinder( < 3.1662, 2.03703, 19.8678 >, < 3.49544, 2.28006, 21.188 >, 0.1, rgb < 0, 0, 0 > )
        // bond C32-C31
        ccdc_draw_closed_cylinder( < 9.50253, 4.75278, 20.5963 >, < 8.95197, 4.28741, 21.7757 >, 0.1, rgb < 0, 0, 1 > )
        // bond C30-C31
        ccdc_draw_closed_cylinder( < 7.64577, 3.81953, 21.8238 >, < 8.95197, 4.28741, 21.7757 >, 0.1, rgb < 0, 0, 1 > )
        // bond C27-C26
        ccdc_draw_closed_cylinder( < 4.7091, 2.87494, 21.4921 >, < 3.49544, 2.28006, 21.188 >, 0.1, rgb < 0, 0, 0 > )
        // bond C22-C21
        ccdc_draw_closed_cylinder( < 5.13173, 7.43164, 19.8873 >, < 3.77115, 7.27352, 19.8306 >, 0.1, rgb < 1, 1, 0 > )
        ccdc_orient_structure( structure_orientation )
        #declare structure_orientation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
        #declare structure_rotation = transform { matrix <
            1, 0, 0,
            0, 1, 0,
            0, 0, 1,
            0, 0, 0 > };
    }
    // jr272 finished
    ccdc_orient_world( world_orientation )
}
