function sbas_patch(patch_id,iint,fparms);
%
% Doing sbas on patches
%
% Usage: sbas_patch(patch_id,iint,fparms);
%
% patch_id --- name of the patches  (e.g. 'PATCH_2');
% iint:    --- index of iteration (e.g. 1)
% fparms   --- name of the parameter file (default 'parms_candis.mat')
%
%
% by Kang Wang in Jan. 2018
% Last updated by Kang Wang on 08/10/2018  --- dding weighting option
% Last Updated by Kang Wang on 04/02/2019  --- adding another option to compute average velocity

warning off;
if nargin<3;
    fparms='parms_candis.mat';
end

parms=load(fparms);
currdir=pwd;
dir_patch=[currdir,'/',patch_id];
data=load([dir_patch,'/','sbas_in','_INT',num2str(iint),'.mat']);

day1=data.day1;
day2=data.day2;
day_sar_old=data.day_sar;
dBp=data.dBp;
los_patch=data.los_in_patch;
inc_patch=data.inc_patch;
corr_patch = data.corr_patch;
xpixel_patch=data.xpixel_patch;
ypixel_patch=data.ypixel_patch;

Dt=(day2-day1)/365.25; %time interval of interferograms
Tinv=pinv(Dt);

[npixel_all,nintf]=size(los_patch);

sbas_type=parms.sbas_type;
smooth=parms.smooth;
if (smooth<0);
    error('smooth should be no less than zero...');
end
rng=parms.radar_range;
wavelength=parms.wavelength;
pspline=parms.value_spline;
isweight=parms.is_weight;

dem_flag=parms.dem_flag;
vel_flag=parms.vel_flag;


if (pspline > 1 | pspline <=0);
    error(['Wrong value for spline fitting:',num2str(spline)]);
end


indx_tmp=~isnan(los_patch);
indx_sum=sum(indx_tmp')';

if strcmp(sbas_type,'SBAS');
   indx_good=find(indx_sum==nintf);
   los_in=los_patch(indx_good,:);
   if (isweight>0);
     corr_in=corr_patch(indx_good,:);
   end
   
elseif strcmp(sbas_type,'NSBAS');
   indx_good=find(indx_sum>(floor(parms.nsbas_ratio*nintf))); 
   los_in=los_patch(indx_good,:);
   if (isweight>0);
      corr_in=corr_patch(indx_good,:);
   end
else
    error(['Unrecognized sbas_type ',sbas_type]);
end

if length(inc_patch(:))>1;
  inc=inc_patch(indx_good);
else
  inc=inc_patch;
end
[npixel,nintf]=size(los_in);



[day_sar,indx_sort]=sort(day_sar_old,'ascend');
nsar=length(unique(day_sar));
t_sar=(day_sar-day_sar(1))/365.25;
tt_sar=days2yr(day_sar)-2010; % time of the SAR in unit of year, relative to 2010
dt=diff(t_sar);
xx_sar=t_sar/mean(dt);

flog=fopen([currdir,'/',patch_id,'/','sbas_patch.log'],'a');
fprintf(flog,'%s\n',['Start SBAS on ',time_now]);
fprintf(flog,'%s\n',['Number of SAR acquisitions       : ',num2str(nsar)]);
fprintf(flog,'%s\n',['Number of Interferograms         : ',num2str(nintf)]);
fprintf(flog,'%s\n',['Number of pixels per patch       : ',num2str(npixel_all)]);
fprintf(flog,'%s\n',['Estimating DEM error             : ',num2str(parms.dem_flag)]);
fprintf(flog,'%s\n',['Weighting Interferograms         : ',num2str(isweight)]);
fprintf(flog,'%s\n',['Smooth                           : ',num2str(parms.smooth)]);
fprintf(flog,'%s\n',['Value for Spline Fitting         : ',num2str(pspline)]);
fprintf(flog,'%s\n',['SBAS Type                        : ',sbas_type]);


indx1=zeros(nintf,1);
indx2=zeros(nintf,1);

for k=1:nintf;
    indx1(k)=find(day1(k)==day_sar);
    indx2(k)=find(day2(k)==day_sar);
end

if (dem_flag>0);
    G1=zeros(nintf,nsar);     
    beta=4*pi/(wavelength*rng*sin(mean(inc)*pi/180));
    for k=1:nintf;
      indx1_this_intf=indx1(k);
      indx2_this_intf=indx2(k);
      ii1=indx1_this_intf;
      ii2=indx2_this_intf-1;
      G1(k,ii1:ii2)=1;
      G1(k,nsar)=beta*dBp(k);
    end
    
    H2=zeros(nsar-2,nsar);
    for k=1:nsar-2;
        H2(k,k)=1/dt(k);
        H2(k,k+1)=-1/dt(k+1);     %assuming the velocity is the same
    end
  
else
    G1=zeros(nintf,nsar-1);
    for k=1:nintf;
      indx1_this_intf=indx1(k);
      indx2_this_intf=indx2(k);
      ii1=indx1_this_intf;
      ii2=indx2_this_intf-1;
      G1(k,ii1:ii2)=1;
    end
    
    H2=zeros(nsar-2,nsar-1);
    for k=1:nsar-2;
        H2(k,k)=1/dt(k);
        H2(k,k+1)=-1/dt(k+1);     %assuming the velocity is the same
    end
 
end

if (smooth>0);
  G2=smooth*H2;
  d2=zeros(nsar-2,1); %for smoothing;
else
    G2=[];
    d2=[];
end

dm_disp=floor(npixel/100);

los_out=NaN(npixel,nsar);
vel_out1=NaN(npixel,1);
vel_out2=NaN(npixel,1);
vel_out3=NaN(npixel,1); % d = d0+v*t + A*sin(wt)+B*sin(wt); so it contains the seasonal term
dh_out=NaN(npixel,1);
amp_out=NaN(npixel,1);
phs_out=NaN(npixel,1);

if strcmp(sbas_type,'SBAS');
  G=[G1;G2];
  Ginv=pinv(G);

 for k=1:npixel;
     d1=(los_in(k,:))';
     if (isweight>0);
       corr_pixel = corr_in(k,:);
%       corr_pixel(isnan(corr_pixel))=0;
       w_pixel = corr_pixel./sum(corr_pixel);
       W=diag(w_pixel);
       d=[W*d1;d2];
       xx=pinv([W*G1;G2]);
     else
       d=[d1;d2];
       xx=Ginv*d;
     end
     los_out_this_pixel=zeros(nsar,1);
     for nn=2:nsar;
       los_out_this_pixel(nn)=sum(xx(1:nn-1));
     end
     
     if (pspline < 1); % fit a spline function if necessary
       pp_this_pixel=csaps(xx_sar,los_out_this_pixel,pspline);
       los_out_this_pixel_new=fnval(pp_this_pixel,xx_sar);
       los_out_this_pixel=los_out_this_pixel_new-los_out_this_pixel_new(1);
     end     
     los_out(k,:)=los_out_this_pixel; 

       TT1=1; % annual cycle
       GG=ones(length(los_out_this_pixel),4);

       GG(:,2)=tt_sar;
       GG(:,3) = sin(2*pi*TT1*tt_sar);
       GG(:,4) = cos(2*pi*TT1*tt_sar);
       dd=los_out_this_pixel(:);
       yyout=pinv(GG)*dd;
       amp_out(k)=norm([yyout(3) yyout(4)],2);
       phs_out(k)=atan2(yyout(4),yyout(3));
 

    
     if (dem_flag>0);
         dh_out=xx(nsar);
     end
     
     if (vel_flag>0);
      
       t_sar_new=t_sar(3:nsar-3);
       los_out_this_pixel_new=los_out_this_pixel(3:nsar-3);
%       pp1=polyfit(t_sar_new,los_out_this_pixel_new,1);
%       vel_out1(k)=pp1(1);

       pp1=robustfit(t_sar_new,los_out_this_pixel_new);
       vel_out1(k)=pp1(2);
       vel_out2(k)=Tinv*d1;
       vel_out3(k)=yyout(2);

     end
     
    if (mod(k,dm_disp)==0);
       fprintf(flog,'%s\n',['Percentage Done (%): ',sprintf('%8.3f',100*k/npixel)]);
    end
    
     
  end
elseif strcmp(sbas_type,'NSBAS');
   for k=1:npixel;
      d1=(los_in(k,:))';
      indx_good_this_pixel=~isnan(d1);
      d1_good=d1(indx_good_this_pixel);
      G1_good=G1(indx_good_this_pixel,:);
      day1_good=day1(indx_good_this_pixel);
      day2_good=day2(indx_good_this_pixel);
      Dt_good=(day2_good-day1_good)/365.25;

      if (isweight>0);
        corr_tmp = corr_in(k,:);
        corr_pixel = corr_tmp(indx_good_this_pixel);
        w_pixel = corr_pixel./sum(corr_pixel);
        W=diag(w_pixel);
        d=[W*d1_good;d2];
        G=[W*G1_good;G2];
        xx=pinv(G)*d;
      else
        d=[d1_good;d2];
        G=[G1_good;G2];
        xx=pinv(G)*d;
      end
      
      los_out_this_pixel=zeros(nsar,1);
      for nn=2:nsar;
        los_out_this_pixel(nn)=sum(xx(1:nn-1));
      end
     
      if (pspline < 1); % fit a spline function if necessary
        pp_this_pixel=csaps(xx_sar,los_out_this_pixel,pspline);
        los_out_this_pixel_new=fnval(pp_this_pixel,xx_sar);
        los_out_this_pixel=los_out_this_pixel_new-los_out_this_pixel_new(1);
      end
      los_out(k,:)=los_out_this_pixel;
      
      if (dem_flag>0);
         dh_out=xx(nsar);
      end
     
     if (vel_flag>0);
       t_sar_new=t_sar(3:nsar-3);
       los_out_this_pixel_new=los_out_this_pixel(3:nsar-3);
%      pp1=polyfit(t_sar,los_out_this_pixel,1);
%       pp1=polyfit(t_sar_new,los_out_this_pixel_new,1);
%      vel_out1(k)=pp1(1);

       pp1=robustfit(t_sar_new,los_out_this_pixel_new);
       vel_out1(k)=pp1(2);
       vel_out2(k)=pinv(Dt_good)*d1_good;

     end

    if (mod(k,dm_disp)==0);
       fprintf(flog,'%s\n',['Percentage Done (%): ',sprintf('%8.3f',100*k/npixel)]);
    end

   end
end

if (k==npixel);
      fprintf(flog,'%s\n',['Percentage Done (%): ',sprintf('%8.3f',100*k/npixel)]);
end

fprintf(flog,'%s\n',['End SBAS on ', patch_id,' on ',time_now]);
fclose(flog);

los_out_patch=NaN(npixel_all,nsar);
dh_out_patch=NaN(npixel_all,1);
vel_out1_patch=NaN(npixel_all,1);
vel_out2_patch=NaN(npixel_all,1);
vel_out3_patch=NaN(npixel_all,1);
los_out_patch(indx_good,:)=los_out;
dh_out_patch(indx_good,:)=dh_out;
vel_out1_patch(indx_good,:)=vel_out1;
vel_out2_patch(indx_good,:)=vel_out2;
vel_out3_patch(indx_good,:)=vel_out3;

amp_out_patch=NaN(npixel_all,1);
phs_out_patch=NaN(npixel_all,1);
amp_out_patch(indx_good,:)=amp_out;
phs_out_patch(indx_good,:)=phs_out;

save([currdir,'/',patch_id,'/','sbas_out_INT',num2str(iint),'.mat'],'day_sar',...
    'los_out_patch','dh_out_patch','amp_out_patch','phs_out_patch','vel_out1_patch','vel_out2_patch','vel_out3_patch','xpixel_patch','ypixel_patch');

