function [los_out_patch,dh_out_patch,vel_out_patch]=sbas(los_patch,day1,day2,day_sar_in,smooth,varargin);
% function to the SBAS 
%
%  Usage: [los_out,dh_out,vel_out]=sbas(los_patch,day1,day2,day_sar,smooth,varargin);
%
%  options:
%   'baseline'        ---- vector of interferogram baseline
%   'dem_error'       ----  [0/1] toggle for dem error estimation
%   'incident_angle'  ---- average incident angel (used in DEM error estimation)
%   'radar_range'     ---- average radar range [86400]  (used in DEM error estimation)
%   'wavelength'      ---- radar wavelength [0.0554658] (used in DEM error estimation)
%   'sbas_type'       ---- sbas_type [SBAS/NSBAS]
%   'nsbas_ratio'     ---- threshold ratio of good interferograms for individual pixels [0.7] (used with NSBAS)
%   'smooth'          ---- smoothing paramter (non-negative value)
%   'value_spline'    ---- value for spline fitting (0-1)
%   'patch_id'        ---- string of the patch_id (e.g. 'patch_1')
% 
%  by Kang Wang in Jan. 2018

[npixel_all,nintf]=size(los_patch);
[day_sar,indx_sort]=sort(day_sar_in,'ascend');
nsar=length(unique(day_sar));
t_sar=(day_sar-day_sar(1))/365.25;
dt=diff(t_sar);
xx_sar=t_sar/mean(dt);

dem_error = 0;
smooth = 0;
p_spline = 1;
sbas_type = 'SBAS';
wavelength = 0.0554658;
rng = 86400;
inc = 38.0;
vel_flag = 0;
dem_flag= 0;

if (~isempty(varargin))
  for c=1:floor(length(varargin)/2);
        switch varargin{c*2-1};
           case 'baseline';
              dbp = real(varargin{c*2});
           case 'dem_error';
              dem_flag=real(varargin{c*2});
           case 'incident_angle';
              inc = real(varargin{c*2});
           case 'value_spline';
              pspline = real(varargin{c*2});
           case 'sbas_type'; 
              sbas_type = char(varargin{c*2});
           case 'radar_range';
              rng = char(varargin{c*2});
           case 'wavelength';
              wavelength = char(varargin{c*2});
           case 'velocity';
              vel_flag = real(varargin{c*2});
           case 'nsbas_ratio';
              nsbas_ratio = 0.7;
           case 'patch_id';
              patch_id=char(varargin{c*2});
        end
  end
end


disp(['Start SBAS on ',time_now]);
disp(['Number of SAR acquisitions    :  ',num2str(nsar)]);
disp(['Number of Interferograms      :  ',num2str(nintf)]);
disp(['Number of pixels              :  ',num2str(npixel_all)]);
disp(['Smooth value                  :  ',num2str(smooth)]);
disp(['SBAS type                     :  ',sbas_type]);
disp(['p_spline                      :  ',num2str(pspline)]);
disp(['Radar wavelength [m]          :  ',num2str(wavelength)]);
disp(['Mean Radar range [m]          :  ',num2str(mean(rng))]);
disp(['Mean incident angele [deg]    :  ',num2str(mean(inc))]);




if (smooth<0);
    error('smooth should be no less than zero...');
end


if (pspline > 1 | pspline <=0);
    error(['Wrong value for spline fitting:',num2str(spline)]);
end


indx_tmp=~isnan(los_patch);
indx_sum=sum(indx_tmp')';

if strcmp(sbas_type,'SBAS');
   indx_good=find(indx_sum==nintf);
   los_in=los_patch(indx_good,:);
elseif strcmp(sbas_type,'NSBAS');
   indx_good=find(indx_sum>(floor(nsbas_ratio*nintf))); 
   los_in=los_patch(indx_goood,:);
else
    error(['Unrecognized sbas_type ',sbas_type]);
end

inc=inc_patch(indx_good);
[npixel,nintf]=size(los_in);

if strcmp(parms.dem_error,'y');
    dh_out=NaN(npixel,1);
end


currdir = pwd;
flog=fopen([currdir,'/','sbas_',patch_id,'.log'],'a');
fprintf(flog,'%s\n',['Start SBAS on ',time_now]);
fprintf(flog,'%s\n',['Number of SAR acquisitions       : ',num2str(nsar)]);
fprintf(flog,'%s\n',['Number of Interferograms         : ',num2str(nintf)]);
fprintf(flog,'%s\n',['Number of pixels per patch       : ',num2str(npixel_all)]);
fprintf(flog,'%s\n',['Estimating DEM error             : ',parms.dem_error]);


indx1=zeros(nintf,1);
indx2=zeros(nintf,1);

for k=1:nintf;
    indx1(k)=find(day1(k)==day_sar);
    indx2(k)=find(day2(k)==day_sar);
end

if (dem_flag>0);
    G1=zeros(nintf,nsar);     
    beta=4*pi/(wavelength*rng*sin(inc*pi/180));
    for k=1:nintf;
      indx1_this_intf=indx1(k);
      indx2_this_intf=indx2(k);
      ii1=indx1_this_intf;
      ii2=indx2_this_intf-1;
      G1(k,ii1:ii2)=1;
      G1(k,nsar)=beta*dBp(k);
    end
    
    H2=zeros(nsar-2,nsar);
    for k=1:nsar-2;
        H2(k,k)=1/dt(k);
        H2(k,k+1)=-1/dt(k+1);     %assuming the velocity is the same
    end
  
else
    G1=zeros(nintf,nsar-1);
    for k=1:nintf;
      indx1_this_intf=indx1(k);
      indx2_this_intf=indx2(k);
      ii1=indx1_this_intf;
      ii2=indx2_this_intf-1;
      G1(k,ii1:ii2)=1;
    end
    
    H2=zeros(nsar-2,nsar-1);
    for k=1:nsar-2;
        H2(k,k)=1/dt(k);
        H2(k,k+1)=-1/dt(k+1);     %assuming the velocity is the same
    end
 
end

if (smooth>0);
  G2=smooth*H2;
  d2=zeros(nsar-2,1); %for smoothing;
else
    G2=[];
    d2=[];
end

dm_disp=floor(npixel/100);

los_out=NaN(npixel,nsar);
vel_out=NaN(npixel,1);
dh_out=NaN(npixel,1);
if strcmp(sbas_type,'SBAS');
  G=[G1;G2];
  Ginv=pinv(G);

 for k=1:npixel;
    d1=(los_in(k,:))';
     d=[d1;d2];
     xx=Ginv*d;
     los_out_this_pixel=zeros(nsar,1);
     for nn=2:nsar;
       los_out_this_pixel(nn)=sum(xx(1:nn-1));
     end
     
     if (pspline < 1); % fit a spline function if necessary
       pp_this_pixel=csaps(xx_sar,los_out_this_pixel,pspline);
       los_out_this_pixel_new=fnval(pp_this_pixel,xx_sar);
       los_out_this_pixel=los_out_this_pixel_new-los_out_this_pixel_new(1);
     end     
     los_out(k,:)=los_out_this_pixel; 
     
     if (dem_flag>0);
         dh_out=xx(nsar);
     end
     
     if (vel_flag>0);
       pp1=polyfit(t_sar,los_out_this_pixel,1);
       vel_out(k)=pp1(1);
     end
     
    if (mod(k,dm_disp)==0);
       fprintf(flog,'%s\n',['Percentage Done (%): ',sprintf('%8.3f',100*k/npixel)]);
    end
    
     
  end
elseif strcmp(sbas_type,'NSBAS');
   for k=1:npixel;
      d1=(los_in(k,:))';
      indx_good_this_pixel=~isnan(d1);
      d1_good=d1(indx_good_this_pixel);
      G1_good=G1(indx_good_this_pixel,:);
      G=[G1_good;G2];
      d=[d1_good;d2];
      xx=pinv(G)*d;
      
      los_out_this_pixel=zeros(nsar,1);
      for nn=2:nsar;
        los_out_this_pixel(nn)=sum(xx(1:nn-1));
      end
     
      if (pspline < 1); % fit a spline function if necessary
        pp_this_pixel=csaps(xx_sar,los_out_this_pixel,pspline);
        los_out_this_pixel_new=fnval(pp_this_pixel,xx_sar);
        los_out_this_pixel=los_out_this_pixel_new-los_out_this_pixel_new(1);
      end
      los_out(k,:)=los_out_this_pixel;
      
      if (dem_flag>0);
         dh_out=xx(nsar);
      end
     
     if (vel_flag>0);
       pp1=polyfit(t_sar,los_out_this_pixel,1);
       vel_out(k)=pp1(1);
     end
   end
  
end

if (k==npixel);
      fprintf(flog,'%s\n',['Percentage Done (%): ',sprintf('%8.3f',100*k/npixel)]);
end

fprintf(flog,'%s\n',['End SBAS on ', patch_id,' on ',time_now]);
fclose(flog);

los_out_patch=NaN(npixel_all,nsar);
dh_out_patch=NaN(npixel_all,1);
vel_out_patch=NaN(npixel_all,1);
los_out_patch(indx_good,:)=los_out;
dh_out_patch(indx_good,:)=dh_out;
vel_out_patch(indx_good,:)=vel_out;


