function get_aps_from_era5();
% estimate the APS from ERA5 for each SAR acquisitions
% by Kang Wang on 10/21/2019

p=load('parms_aps.mat');
UTC_sat=p.UTC_sat;
time = str2num(UTC_sat(1:2)) + str2num(UTC_sat(4:5))/60;


t_before = floor(time/1);
t_after = ceil(time/1);
fprintf(['Satellite pass is ' num2str(time) ' UTC \n'])

% the faction it is closer towards the other date.
f_after = (time - 1*t_before)/(1*t_after - 1*t_before);
f_after(isnan(f_after))=1;
f_before = 1-f_after;

date_sar=textread('dates.dat','%s\n');
nsar=length(date_sar);
f_after = repmat(f_after,nsar,1);
f_before = repmat(f_before,nsar,1);


[dem,xmin,xmax,ymin,ymax,smpres,nncols,nnrows] = get_DEM;


lonmin = floor(xmin)-1;
lonmax= ceil(xmax)+1;
latmin = floor(ymin)-1;
latmax = ceil(ymax)+1;

maxdem = ceil(max(max(dem))/100)*100+50; % max height of dem in metres

ndates=length(date_sar);

model_type='era';
fig_test = 0;           % when 1 show the dem as debug figure
save_3D_delays = 0;     % When 1 saves the tropopsheric delays for each x,y and with height

Rd = 287.05;            % [j/kg/K] specific gas constants dry air
Rv = 461.495;           % [j/kg/K] water vapour
k1 = 0.776;             % [K/Pa]
k2 = 0.716;             % [K/Pa]
k3 = 3.75e3;            % [K^2/Pa]
coeff.Rd = Rd;
coeff.Rv = Rv;

%% Defaults
zref = 15000;       % zref for integration- tropopause
zincr = 15;         % z increment spacing for vertical interpolation before integration
vertres = 5;        % vertical resolution for comparing dem to vert profiles of delay

for d = 1:ndates;
    this_date=date_sar{d};
   yr=this_date(1:4);
   mth=this_date(5:6);
   day=this_date(7:8);
   hr_sar=UTC_sat(1:2);
   min_sar=UTC_sat(4:5);
  
  t1=datenum(str2num(yr),str2num(mth),str2num(day),str2num(hr_sar),0,0);
  t2=datenum(str2num(yr),str2num(mth),str2num(day),str2num(hr_sar),60,0);
  
  s1=datestr(t1,'yyyymmdd HH:MM:SS');
  s2=datestr(t2,'yyyymmdd HH:MM:SS');
  
  date1=s1(1:8);
  date2=s2(1:8);
  hr1=s1(10:11);
  hr2=s2(10:11);
  time1=[hr1,':00'];
  time2=[hr2,':00'];
  
  syr1=s1(1:4);
  smonth1=s1(5:6);
  sday1=s1(7:8);
  
  
  syr2=s2(1:4);
  smonth2=s2(5:6);
  sday2=s2(7:8);
  
    
    
    % the save filenames
    outfile = [p.era_datapath,'/',this_date,'_ZWD.xyz'];
    hydroutfile = [p.era_datapath,'/',this_date,'_ZHD.xyz']; 
     
    no_data = 0;
    for kk = 1:2
        if kk == 1
            file = [p.era_datapath,'/','ggap',syr1,smonth1,sday1,hr1,'00.nc'];
            
        end
        if kk == 2
            file = [p.era_datapath,'/','ggap',syr2,smonth2,sday2,hr2,'00.nc'];
        end
        
        
        % check
        if exist(file,'file')~=2
            no_data = no_data+1;
        else
            
            era_data_type=p.era_data_type;
            %% loading the weather model data
            if  strcmpi(model_type,'era')
                 [ Temp,e,Geopot,P,longrid,latgrid,xx,yy,lon0360_flag] =  aps_load_era(file,era_data_type) ;
            elseif  strcmpi(model_type,'era5')
                 [ Temp,e,Geopot,P,longrid,latgrid,xx,yy,lon0360_flag] =  aps_load_era(file,era_data_type) ;
            elseif strcmpi(model_type,'merra') || strcmpi(model_type,'merra2')
                 [ Temp,e,Geopot,P,longrid,latgrid,xx,yy,lon0360_flag] =  aps_load_merra(file,model_type,coeff) ;
            end
            
            
            %% verify and cope with NAN's
            [ Temp,e,Geopot,P,longrid,latgrid] =  aps_weather_model_nan_check( Temp,e,Geopot,P,longrid,latgrid) ;


            % define weather model grid nodes
            latlist = reshape(latgrid(:,:,1),[],1);
            lonlist = reshape(longrid(:,:,1),[],1);
            xlist = reshape(xx,[],1);
            ylist = reshape(yy,[],1);

            %% Limit weather model to those grid points around the user-defined InSAR box
            % Getting the weather model resolution
            lat_res = abs(diff(unique(latgrid)))*1.5;
            lat_res = lat_res(1);
            lon_res = abs(diff(unique(longrid)))*1.5;
            lon_res = lon_res(1);

            % make sure the weather model and the lonlat specified by user
            % is defined in the same way. Weatehr models can be [0 360]
            % degrees. User can be [-180 180] unlikely [0 360]
            if strcmpi(lon0360_flag,'y')
                % fprintf('This needs to be defined better and some checks are needed \n')
                % not all pixels should be shifted.
  
                if xmin<0
                   xmin= xmin+360; 
                end
                if xmax<0
                   xmax = xmax +360; 
                end
            end


            % generation a plot
            if fig_test ==1 & d==1 & kk==1
                fontsize = 15;
                hfig = figure('name',[model_type ' weather model nodes and InSAR region'],'position', [ 200   243   610   603]);
                % for the legend generation
                plot(mean([xmin xmax]),mean([ymax ymin]),'wo','markeredgecolor','k','markerfacecolor','w','markersize',15)
                hold on
                plot(mean([xmin xmax]),mean([ymax ymin]),'r.')
                hold on
                plot(mean([xmin xmax]),mean([ymax ymin]),'r-','linewidth',2)

                imagesc([xmin xmax],[ymax ymin],dem)  
                cc=colorbar;
                view(0,90)

                axis xy
                ylabel(cc,'Topography [m]','fontsize',fontsize)
                hold on
                plot(lonlist,latlist,'wo','markeredgecolor','k','markerfacecolor','w','markersize',15)
                hold on
                plot([xmin xmin xmax xmax xmin],[ymin ymax ymax ymin ymin],'r-','linewidth',2)

                title([model_type ' weather model nodes'],'fontsize',fontsize)
                set(gca,'fontsize',fontsize)
                axis equal
                axis tight 

                legend('location','northoutside',[model_type ' weather model nodes'],['Used ' model_type ' weather model nodes'],'InSAR box')
            end

            % making the weather model grid slightly larger than the InSAR
            % bounding box. Will use the weather model resolution for this
            % to make sure an extra grid node is included.
            ix = find(ymin-lat_res<= latlist & latlist <= ymax+lat_res  & xmin-lon_res<= lonlist & lonlist<= xmax+lon_res) ;
            xlist = xlist(ix);
            ylist = ylist(ix);
            latlist = latlist(ix);
            lonlist = lonlist(ix);
            numy = length(unique(latlist));
            numx = length(unique(lonlist));
            ulatlist = unique(latlist);
            ulonlist = unique(lonlist);
            uxlist = unique(xlist);
            uylist = unique(ylist);

            if fig_test ==1 & d==1 & kk==1
                figure(hfig)
                hold on
                plot(lonlist,latlist,'r.')
                xlim([xmin-4*lon_res  xmax+4*lon_res])
                ylim([ymin-4*lat_res  ymax+4*lat_res])
                str='';
                while ~strcmpi(str,'y') && ~strcmpi(str,'n') 
                    fprintf(['Do the red nodes extend (just) outside the red InSAR rectangle? \n'])  
                    str = input('Continue? [y: for yes, n: no] \n','s');
                end
                if strcmpi(str,'n')
                    error('Check your lon lat crop, otherwize extend area of downlaoded weather model data.')
                end
            end
            
            clear lon_res lat_res ylist xlist %%%SSS 4/16

            % saving the information for support plotting 
            eval([model_type '.' model_type '_lonlat =[lonlist latlist];']);                            % era.era_lonlat = [lonlist latlist];
            eval([model_type '.region =[[xmin xmin xmax xmax xmin]'' [ymin ymax ymax ymin ymin]''];']); % era.region = [[xmin xmin xmax xmax xmin]' [ymin ymax ymax ymin ymin]'];
            deminfo.xmin = xmin;
            deminfo.xmax = xmax;
            deminfo.ymax = ymax;
            deminfo.ymin = ymin;
            deminfo.dem = dem;
            eval([model_type '.deminfo =deminfo;']);                                                    % era.deminfo = deminfo;
            clear deminfo
            % checking if the file already exist. Yes append otherwiuze create it
            if exist('tca_support.mat','file')==2
                eval(['save(''tca_support.mat'',''-append'',''' model_type ''');']);                    % save('tca_support.mat','-append','era')
            else
                eval(['save(''tca_support.mat'',''' model_type ''');']);                                % save('tca_support.mat','era')        
            end
            if fig_test ==1 & d==1 & kk==1
                    if exist(['aps_' model_type],'dir')~=7
                        mkdir(['aps_' model_type]);
                    end
                    print(hfig,'-dpng',['aps_' model_type  filesep model_type '_datapoints.png'])
                    print(hfig,'-depsc',['aps_' model_type filesep model_type '_datapoints.eps'])
            end
            eval(['clear ' model_type]);
         

            % Convert Geopotential to Geopotential Height and then to Geometric Height
            g0 = 9.80665;
            % Calculate Geopotential Height, H
            H = Geopot./g0;

            % map of g with latitude
            g = 9.80616.*(1 - 0.002637.*cosd(2.*latgrid) + 0.0000059.*(cosd(2.*latgrid)).^2);
            % map of Re with latitude
            Rmax = 6378137; 
            Rmin = 6356752;
            Re = sqrt(1./(((cosd(latgrid).^2)./Rmax^2) + ((sind(latgrid).^2)./Rmin^2)));

            % Calculate Geometric Height, Z
            Z = (H.*Re)./(g/g0.*Re - H);
            
            % Find middle of scene to work out glocal and Rlocal for use later
            midx = round(mean(uxlist));
            midy = round(mean(uylist));
            glocal = g(midy,midx,1);
            Rlocal = Re(midy,midx,1);

            cdslices = maxdem/vertres +1;
            cdstack = zeros(numy,numx,cdslices);
            cdstack_dry = zeros(numy,numx,cdslices);
            cdstack_wet = zeros(numy,numx,cdslices);

            XI=(0:zincr:zref)';
            gh = glocal.*(Rlocal./(Rlocal+XI)).^2; %gravity with height for XI height range

            % Interpolate Temp P and e from 0:20:15000 m
            % then integrate using trapz to estimate delay as function of height
            for i = 1:numx;
                for j = 1:numy;
                    xn = uxlist(i);
                    yn = uylist(j);

                    %interpolate at res zincr before doing integration
                    X=double(squeeze(Z(yn,xn,:)));
                    Ye=double(squeeze(e(yn,xn,:)));
                    YeI = interp1(X,Ye,XI,'spline')*100; %change from hPa to Pa

                    YP=double(squeeze(P(yn,xn,:)));
                    YPI = interp1(X,YP,XI,'spline')*100; %change from hPa to Pa

                    YT=double(squeeze(Temp(yn,xn,:)));
                    YTI = interp1(X,YT,XI,'spline');

                    tmp1 = ((k2-(Rd*k1/Rv)).*YeI./YTI + k3.*YeI./(YTI.^2));
                    Lw = (10^-6).*-1*flipud(cumtrapz(flipud(XI),flipud(tmp1)));
                    % L is zenith delay one way in metres
                    % tmp2 = k1.*YPI./YTI;
                    %Ld = (10^-6).*-1*flipud(cumtrapz(flipud(XI),flipud(tmp2)));             % This is using P/T expression (Hanssen, 2001)
                    gm = glocal; 
                    Ld = (10^-6).*((k1*Rd/gm).*(YPI - YPI(zref/zincr +1)));                 % This is P0 expression (Hanssen, 2001)

                
                    % Interpolate important part (i.e. total delay at elevations
                    % less than maxdem) at high res i.e. vertres, and put in cdstack.
                    cdI=(0:vertres:maxdem)';
                    LdI=interp1(XI,Ld,cdI,'spline');
                    LwI=interp1(XI,Lw,cdI,'spline');

                    %cdstack(j,i,:) = LsI;
                    cdstack_dry(j,i,:) = LdI;
                    cdstack_wet(j,i,:) = LwI;
                    
                    if save_3D_delays==1               
                        cdI3D=(0:100:max(XI))';
                        LdI3D=interp1(XI,Ld,cdI3D,'spline');
                        LwI3D=interp1(XI,Lw,cdI3D,'spline');

                        %cdstack(j,i,:) = LsI;
                        cdstack_dry3D(j,i,:) = LdI3D;
                        cdstack_wet3D(j,i,:) = LwI3D;
                        clear LdI3D LwI3D
                    end
                end
            end
            clear uxlist uylist ulonlist ulatlist %%%SSS 4/16
             

            % Interpolate each cdstack layer onto a grid given by the DEM extents
            % in UTM m.
            xsmpres = (xmax-xmin)/nncols;
            ysmpres = (ymax-ymin)/nnrows;
            [xi,yi] = meshgrid(xmin+0.5*xsmpres:xsmpres:xmax-0.5*xsmpres,ymin+0.5*ysmpres:ysmpres:ymax-0.5*ysmpres);

            ix_temp = diff(lonlist);
            ix_temp = find(ix_temp~=0);
            ix_temp = ix_temp(1);
            lonlist_matrix = reshape(lonlist,ix_temp,[]) ;
            latlist_matrix = reshape(latlist,ix_temp,[]) ;
            clear lonlist latlist %%%SSS 4/16
            
            % saving the outputs
            if save_3D_delays==1               
                if kk==1
                    clear hgt lon lat dry1 dry2 wet1 dem_temp hgt_topo %%%SSS 4/16
                    hgt = cdI3D;
                    lon = [lonlist_matrix];
                    lat = latlist_matrix;
                    dry1 = cdstack_dry3D;
                    wet1 = cdstack_wet3D;
                    
                    
                    
                                
                    % also give the station topography
                    dem_temp = dem;
                    dem_temp(isnan(dem_temp))=0;
                    hgt_topo = griddata(xi,yi,dem_temp,lon,lat,'linear'); 
                    clear dem_temp
                    
                    
                elseif kk==2
                    clear dry2 wet2
                    dry2 = cdstack_dry3D;
                    wet2 = cdstack_wet3D;
                end
            end
            
            clear cdstack_interp_dry %%%SSS 4/16
            cdstack_interp_dry = zeros(nnrows,nncols,cdslices);
            parfor n = 1:cdslices
                newslice = interp2(lonlist_matrix,latlist_matrix,cdstack_dry(:,:,n),xi,yi,'linear');          
                cdstack_interp_dry(:,:,n)= flipud(newslice); % flipud due to ypositive downpage for matlab figs, and ypositive uppage for utmy
            end     

            clear cdstack_interp_wet %%%SSS 4/16 
            cdstack_interp_wet = zeros(nnrows,nncols,cdslices);
            parfor n = 1:cdslices 
               newslice = interp2(lonlist_matrix,latlist_matrix,cdstack_wet(:,:,n),xi,yi,'linear');   
               cdstack_interp_wet(:,:,n)= flipud(newslice); % flipud due to ypositive downpage for matlab figs, and ypositive uppage for utmy
            end
            clear lonlist_matrix latlist_matrix %%%SSS 4/16
            % keeping the coordinates in the same grid as the data
            xi = flipud(xi);
            yi = flipud(yi);


            % Pull out delays from cdstack layers that match dem heights
            clear wetcorrection hydrcorrection rounddem %%%SSS 4/16
            wetcorrection = ones(nnrows,nncols);
            hydrcorrection = ones(nnrows,nncols);
            rounddem = round(dem/vertres);
            rounddem(dem < 0)=0;

            % cope with the case that NaN are present. This can be sea-level
            rounddem(isnan(dem))=0;

            for i=1:nnrows
                for j=1:nncols
                    wetcorrection(i,j) = cdstack_interp_wet(i,j,rounddem(i,j)+1);
                end
            end

            for i=1:nnrows
                for j=1:nncols
                    hydrcorrection(i,j) = cdstack_interp_dry(i,j,rounddem(i,j)+1);
                end
            end

            if kk==1
                wetcorrection1 = wetcorrection;
                drycorrection1 = hydrcorrection;
            end
            if kk==2
                wetcorrection2 = wetcorrection;
                drycorrection2 = hydrcorrection;
            end
            clear wetcorrection hydrcorrection

        end
    end


    
    if sum(no_data)==0
        % note that this is a one way Zenith delay and not a slant delay. 
        % Units are in cm


        % saving individual estimates based on the time-stamp
        outfile_wet_before = [p.era_datapath,'/',this_date, '_ZWD_before.xyz'];
        outfile_wet_after = [p.era_datapath,'/',this_date, '_ZWD_after.xyz'];
        outfile_dry_before = [p.era_datapath,'/',this_date, '_ZHD_before.xyz'];
        outfile_dry_after = [p.era_datapath,'/',this_date, '_ZHD_after.xyz'];


        fidwet_before = fopen(outfile_wet_before,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(wetcorrection1,[],1)]';
        tally = fwrite(fidwet_before,data_write,'double');
        fclose(fidwet_before);
        clear data_write tally %%%SSS 4/16
        fidwet_after = fopen(outfile_wet_after,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(wetcorrection2,[],1)]';
        tally = fwrite(fidwet_after,data_write,'double');
        fclose(fidwet_after);
        clear data_write tally %%%SSS 4/16
        fiddry_before = fopen(outfile_dry_before,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(drycorrection1,[],1)]';
        tally = fwrite(fiddry_before,data_write,'double');
        fclose(fiddry_before); 
        clear data_write tally %%%SSS 4/16
        fiddry_after = fopen(outfile_dry_after,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(drycorrection2,[],1)]';
        tally = fwrite(fiddry_after,data_write,'double');
        fclose(fiddry_after);
        clear data_write tally %%%SSS 4/16

        % saving the outputs
        if save_3D_delays==1   
            wet = wet1*f_before(d) +  wet2*f_after;
            dry = dry1*f_before(d) +  dry2*f_after;
            save([weather_model_datapath filesep date_before(d,:) filesep date_before(d,:) '_3D.mat'],'dry','wet','hgt','lon','lat','hgt_topo')
            clear wet dry hgt dry1 dry2 wet1 wet2 %%%SSS 4/16
        end
        

                
        % Output wet correction
        wetcorrection = wetcorrection1*f_before(d) +  wetcorrection2*f_after(d);
        clear wetcorrection1 wetcorrection2
        wetcorrection = wetcorrection*100;                 % delay in [cm]
        fid = fopen(outfile,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(wetcorrection,[],1)]';
        tally = fwrite(fid,data_write,'double');
        fclose(fid);
        clear data_write tally wetcorrection %%%SSS 4/16

        % Output hydrostatic correction
        hydrcorrection = drycorrection1*f_before(d) +  drycorrection2*f_after(d);
        clear drycorrection1 drycorrection2
        hydrcorrection = hydrcorrection*100;                 % delay in [cm]
        fid = fopen(hydroutfile,'w');
        data_write =  [reshape(xi,[],1) reshape(yi,[],1) reshape(hydrcorrection,[],1)]';
        tally = fwrite(fid,data_write,'double');
        fclose(fid);
        clear data_write tally hydrcorrection %%%SSS 4/16

        fprintf([num2str(d) ' completed out of ' num2str(ndates) '\n' ])
        
    else
        fprintf([num2str(d) ' completed out of ' num2str(ndates) ' (NO DATA)\n' ])
    end
end

