function [S_series,E_series,I_series,R_series] = run_SEIR_spatial(IC,beta_c,beta_h,beta_w,sigma,gamma,Tc,Th,Tw,tf,theta)
% USAGE: [S_series,E_series,I_series,R_series] = run_SEIR_spatial(IC,beta_c,beta_h,beta_w,sigma,gamma,Tc,Th,Tw,tf,theta)
% written by Allison Shaw, Contact ashaw@umn.edu for assistance.
% last updated 3 June 2020
%
% runs simulations of a semi-spatial SEIR model, tracking the number of
% susceptibles (S), exposed (E), infected (I), and removed (R) individuals.
% All individuals spend a fraction of each day (T_h) at home and a
%  fraction (T_w) working either from home or from work, and a fraction
%  (2*T_c) commuting to work.
%
% INPUTS:
%   IC = initial number of susceptible, exposed, infected and removed
%   individuals
%   beta_c = infection rate while commuting
%   beta_h = infection rate while at home
%   beta_w = infection rate while at work
%   sigma = rate of moving from exposed to infected
%   gamma = recovery rate in any location
%   Tw = fraction of the day spent working
%   Tc = fraction of the day spent commuting
%   Th = fraction of the day spent not working
%   tf = number of days to simulate for
%   theta = fraction of the population commuting to campus
% 
% OUTPUTS:
%   S_series = count of susceptible individuals for each day
%   E_series = count of exposed individuals for each day
%   I_series = count of infected individuals for each day
%   R_series = count of removed individual for each day

params_h  = [beta_h sigma gamma]; % everyone at home
params_cw = [beta_c sigma gamma]; % commuting time to work
params_ww = [beta_w sigma gamma]; % working at work
params_ch = [beta_h sigma gamma]; % commuting time at home
params_wh = [beta_h sigma gamma]; % working time at home

S_0 = IC(1);
E_0 = IC(2);
I_0 = IC(3);
R_0 = IC(4);


options = odeset('NonNegative',1:4);
[S_series,E_series,I_series,R_series] = deal(NaN(tf,1)); % initialize vectors to store number of S/E/I/R over days

for t=1:tf
    
    % dynamics during non-work day - both
    IC_h = [S_0 E_0 I_0 R_0];
    [~,N_h] = ode45(@SEIR,[0 Th],IC_h,options,params_h);    
    S_h = N_h(end,1);
    E_h = N_h(end,2);
    I_h = N_h(end,3);
    R_h = N_h(end,4);
   
    % dynamics during commute - stay at home
    IC_ch = [S_h*(1-theta) E_h*(1-theta) I_h*(1-theta) R_h*(1-theta)];
    if Tc==0 % if no time spent commuting, there is no change in S E I and R
        N_ch = [S_h*(1-theta) E_h*(1-theta) I_h*(1-theta) R_h*(1-theta)];
    else
        [~,N_ch] = ode45(@SEIR,[0 Tc],IC_ch,options,params_ch);    
    end
    S_ch = N_ch(end,1);
    E_ch = N_ch(end,2);
    I_ch = N_ch(end,3);
    R_ch = N_ch(end,4);
    
    % dynamics during commute - commute to work
    IC_cw = [S_h*theta E_h*theta I_h*theta R_h*theta];
    if Tc==0 % if no time spent commuting, there is no change in S E I and R
        N_cw = [S_h*theta E_h*theta I_h*theta R_h*theta];
    else
        [~,N_cw] = ode45(@SEIR,[0 Tc],IC_cw,options,params_cw);
    end
    S_cw = N_cw(end,1);
    E_cw = N_cw(end,2);
    I_cw = N_cw(end,3);
    R_cw = N_cw(end,4);

    % dynamics during work day - stay at home
    IC_wh = [S_ch E_ch I_ch R_ch];
    if Tw==0 % if no time spent working, there is no change in S E I and R
        N_wh = [S_ch E_ch I_ch R_ch];
    else
        [~,N_wh] = ode45(@SEIR,[0 Tw],IC_wh,options,params_wh);    
    end
    S_wh = N_wh(end,1);
    E_wh = N_wh(end,2);
    I_wh = N_wh(end,3);
    R_wh = N_wh(end,4);
       
    
    % dynamics during work day - commuters
    IC_ww = [S_cw E_cw I_cw R_cw];
    if Tw==0 % if no time spent working, there is no change in S E I and R
        N_ww = [S_cw E_cw I_cw R_cw];
    else
        [~,N_ww] = ode45(@SEIR,[0 Tw],IC_ww,options,params_ww);
    end
    S_ww = N_ww(end,1);
    E_ww = N_ww(end,2);
    I_ww = N_ww(end,3);
    R_ww = N_ww(end,4);
    
    % dynamics during commute part 2 - stay at home
    IC_ch2 = [S_wh E_wh I_wh R_wh];
    if Tc==0 % if no time spent commuting, there is no change in S E I and R
        N_ch2 = [S_wh E_wh I_wh R_wh];
    else
        [~,N_ch2] = ode45(@SEIR,[0 Tc],IC_ch2,options,params_ch);    
    end
    S_ch2 = N_ch2(end,1);
    E_ch2 = N_ch2(end,2);
    I_ch2 = N_ch2(end,3);
    R_ch2 = N_ch2(end,4);
    
    % dynamics during commute part 2 - commute from work
    IC_cw2 = [S_ww E_ww I_ww R_ww];
    if Tc==0 % if no time spent commuting, there is no change in S E I and R
        N_cw2 = [S_ww E_ww I_ww R_ww];
    else
        [~,N_cw2] = ode45(@SEIR,[0 Tc],IC_cw2,options,params_cw);
    end
    S_cw2 = N_cw2(end,1);
    E_cw2 = N_cw2(end,2);
    I_cw2 = N_cw2(end,3);
    R_cw2 = N_cw2(end,4);

    % add commuters and non-commputers back together
    S_0 = S_ch2+S_cw2;
    E_0 = E_ch2+E_cw2;
    I_0 = I_ch2+I_cw2;
    R_0 = R_ch2+R_cw2;
    
    [S_series(t),E_series(t),I_series(t),R_series(t)] = deal(S_0,E_0,I_0,R_0);
    

    eps1=1e-5;
    if abs(abs(S_0+E_0+I_0+R_0)-abs(sum(IC)))>eps1; error('population changing size');end
    
end
